// Main application logic for MVP Influencer Dashboard

// Global variables
let currentCampaignId = '';
let influencersData = [];
let contentData = [];
let currentView = 'grid'; // 'grid' or 'table'
let currentContentView = 'grid'; // 'grid' or 'table'
let currentResultsFilter = 'all'; // 'all', 'tiktok', 'instagram', 'youtube'
let currentSortColumn = '';
let currentSortDirection = 'desc'; // 'asc' or 'desc'
let sortedContentData = [];
let currentContentSortBy = 'default'; // 'default', 'views', 'likes', etc.
let currentContentSortDirection = 'desc'; // 'asc' or 'desc'
let weeklyData = [];
let currentChartType = 'line';
let currentMetricFilter = 'all';
let charts = {};
let currentPlatform = 'all';      // top-level: all | tiktok | instagram | youtube | linkedin | facebook | twitter | bonus
let currentSubfilter = null;      // second-level: instagram_reel, instagram_stories, etc.
let useDateFilterForCalculations = false;
let benchmarkData = {
    instagram: {
        er: { min: 0, max: 0 },
        cpv: { min: 0, max: 0 }
    },
    tiktok: {
        er: { min: 0, max: 0 },
        cpv: { min: 0, max: 0 }
    }
};
let useDateFilterForResults = false
let currentAnalysisDateRange = null;
let paidMediaLookerUrl = '';

// =============================================================================
// DATE COMPARISON FUNCTIONS
// =============================================================================

let isComparisonMode = false;
let period1Data = null;
let period2Data = null;

/**
 * Load benchmark data from Google Sheets Benchmark tab
 */
async function loadBenchmarkData() {
    try {
        console.log('Loading benchmark data from sheet:', currentCampaignId);
        
        // Check if the Benchmark tab exists
        const benchmarkTabExists = await checkSheetTabExists(currentCampaignId, 'Benchmark');
        
        if (!benchmarkTabExists) {
            console.log('Benchmark tab not found');
            return;
        }
        
        console.log('Benchmark tab found - loading data');
        
        // Fetch data from the Benchmark sheet - columns A to E, rows 1-3
        const data = await fetchSheetData(currentCampaignId, 'Benchmark', 'A1:E3');
        
        if (data && data.length >= 3) {
            // Row 2 (index 1): ER values
            // A2="ER", B2=Instagram ER min, C2=Instagram ER max, D2=TikTok ER min, E2=TikTok ER max
            if (data[1] && data[1].length >= 5) {
                benchmarkData.instagram.er.min = parseFloat(data[1][1]) || 0; // B2
                benchmarkData.instagram.er.max = parseFloat(data[1][2]) || 0; // C2
                benchmarkData.tiktok.er.min = parseFloat(data[1][3]) || 0;    // D2
                benchmarkData.tiktok.er.max = parseFloat(data[1][4]) || 0;    // E2
            }
            
            // Row 3 (index 2): CPV values
            // A3="CPV", B3=Instagram CPV min, C3=Instagram CPV max, D3=TikTok CPV min, E3=TikTok CPV max
            if (data[2] && data[2].length >= 5) {
                benchmarkData.instagram.cpv.min = parseFloat(data[2][1]) || 0; // B3
                benchmarkData.instagram.cpv.max = parseFloat(data[2][2]) || 0; // C3
                benchmarkData.tiktok.cpv.min = parseFloat(data[2][3]) || 0;    // D3
                benchmarkData.tiktok.cpv.max = parseFloat(data[2][4]) || 0;    // E3
            }
            
            console.log('Benchmark data loaded:', benchmarkData);
        } else {
            console.warn('No benchmark data found or insufficient rows');
        }
        
    } catch (error) {
        console.error('Error loading benchmark data:', error);
    }
}

/**
 * Calculate and display ER benchmark comparison
 */
function displayERBenchmark(calculatedER, platform) {
    const benchmarkEl = document.getElementById('resultsTotalERBenchmark');
    
    if (!benchmarkEl) return;
    
    // Hide benchmark if platform is not supported or is a sub-filter
    const supportedPlatforms = ['instagram', 'tiktok'];
    const isSubFilter = platform.includes('_');
    
    if (!supportedPlatforms.includes(platform) || isSubFilter) {
        benchmarkEl.innerHTML = '';
        return;
    }
    
    // Get platform-specific benchmark data
    const platformBenchmark = benchmarkData[platform]?.er;
    
    if (!platformBenchmark || (platformBenchmark.min === 0 && platformBenchmark.max === 0)) {
        benchmarkEl.innerHTML = '';
        return;
    }
    
    const erValue = calculatedER;
    const minBench = platformBenchmark.min;
    const maxBench = platformBenchmark.max;
    
    let comparisonText = '';
    let emoji = '';
    
    if (erValue > maxBench) {
        // Above benchmark range
        const percentAbove = ((erValue - maxBench) / maxBench * 100).toFixed(0);
        comparisonText = `<span class="benchmark-value benchmark-above">${percentAbove}% above</span> industry benchmark`;
        emoji = '🚀';
    } else if (erValue < minBench) {
        // Below benchmark range
        const percentBelow = ((minBench - erValue) / minBench * 100).toFixed(0);
        comparisonText = `<span class="benchmark-value benchmark-below">${percentBelow}% below</span> industry benchmark`;
        emoji = '📉';
    } else {
        // Within benchmark range - show the range
        comparisonText = `<span class="benchmark-value benchmark-within">${minBench}% - ${maxBench}%</span> industry benchmark`;
        emoji = '✅';
    }
    
    benchmarkEl.innerHTML = `
        ${comparisonText}
        <span class="benchmark-emoji">${emoji}</span>
    `;
}

/**
 * Calculate and display CPV benchmark comparison for specific platform
 */
function displayCPVBenchmark(calculatedCPV, platform) {
    const benchmarkEl = document.getElementById('resultsTotalCPVBenchmark');
    
    if (!benchmarkEl) return;
    
    // Hide benchmark if platform is not supported or is a sub-filter
    const supportedPlatforms = ['instagram', 'tiktok'];
    const isSubFilter = platform.includes('_');
    
    if (!supportedPlatforms.includes(platform) || isSubFilter) {
        benchmarkEl.innerHTML = '';
        return;
    }
    
    // Get platform-specific benchmark data
    const platformBenchmark = benchmarkData[platform]?.cpv;
    
    if (!platformBenchmark || (platformBenchmark.min === 0 && platformBenchmark.max === 0)) {
        benchmarkEl.innerHTML = '';
        return;
    }
    
    const cpvValue = calculatedCPV;
    const minBench = platformBenchmark.min;
    const maxBench = platformBenchmark.max;
    
    let comparisonText = '';
    let emoji = '';
    
    if (cpvValue < minBench) {
        // Below benchmark range (better for CPV - lower is better)
        const percentBelow = ((minBench - cpvValue) / minBench * 100).toFixed(0);
        comparisonText = `<span class="benchmark-value benchmark-above">${percentBelow}% below</span> industry benchmark`;
        emoji = '🚀';
    } else if (cpvValue > maxBench) {
        // Above benchmark range (worse for CPV)
        const percentAbove = ((cpvValue - maxBench) / maxBench * 100).toFixed(0);
        comparisonText = `<span class="benchmark-value benchmark-below">${percentAbove}% above</span> industry benchmark`;
        emoji = '📉';
    } else {
        // Within benchmark range - show the range
        comparisonText = `<span class="benchmark-value benchmark-within">$${minBench.toFixed(2)} - $${maxBench.toFixed(2)}</span> industry benchmark`;
        emoji = '✅';
    }
    
    benchmarkEl.innerHTML = `
        ${comparisonText}
        <span class="benchmark-emoji">${emoji}</span>
    `;
}

/**
 * Load analysis date ranges from Info sheet - UPDATED VERSION
 * Scans all rows in columns O-P to find date ranges
 */
async function loadAnalysisDateRanges() {
    try {
        console.log('Loading analysis date ranges from sheet:', currentCampaignId);
        
        // Fetch MORE rows to scan for date ranges (O and P columns, rows 1-100)
        const data = await fetchSheetData(currentCampaignId, 'Info', 'O1:P100');
        
        console.log('Date ranges data received:', data);
        
        if (!data || data.length === 0) {
            console.warn('No data found in columns O-P');
            hideAnalysisDateFilter();
            return;
        }
        
        const ranges = [];
        const seenDatePairs = new Map(); // Track unique date pairs
        
        // Scan all rows looking for valid date pairs
        for (let rowIndex = 0; rowIndex < data.length; rowIndex++) {
            const row = data[rowIndex];
            if (!row || row.length < 2) continue;
            
            const startDateRaw = row[0] || ''; // Column O
            const endDateRaw = row[1] || '';   // Column P
            
            // Skip empty rows
            if (!startDateRaw || !endDateRaw) continue;
            
            // Try to parse dates
            const startDate = parseContentDate(startDateRaw);
            const endDate = parseContentDate(endDateRaw);
            
            // Only add if both dates are valid
            if (startDate && endDate) {
                const startFormatted = formatDateForInput(startDate);
                const endFormatted = formatDateForInput(endDate);
                
                // Create a unique key for this date pair
                const dateKey = `${startFormatted}|${endFormatted}`;
                
                // Check if we've already seen this date pair
                if (!seenDatePairs.has(dateKey)) {
                    // First time seeing this date pair - add it
                    const rangeData = {
                        id: `range-${ranges.length}`,
                        rowIndices: [rowIndex], // Store ALL row indices that match this date pair
                        startDate: startFormatted,
                        endDate: endFormatted,
                        startDateDisplay: formatDateDisplay(startDateRaw),
                        endDateDisplay: formatDateDisplay(endDateRaw),
                        label: `Period ${ranges.length + 1}`
                    };
                    
                    seenDatePairs.set(dateKey, ranges.length);
                    ranges.push(rangeData);
                    
                    console.log(`Found NEW date range at row ${rowIndex + 1}:`, {
                        start: startDateRaw,
                        end: endDateRaw,
                        parsed: {
                            start: startFormatted,
                            end: endFormatted
                        }
                    });
                } else {
                    // We've seen this date pair before - add this row index to existing range
                    const existingRangeIndex = seenDatePairs.get(dateKey);
                    ranges[existingRangeIndex].rowIndices.push(rowIndex);
                    
                    console.log(`Found DUPLICATE date range at row ${rowIndex + 1}, adding to existing range`);
                }
            }
        }
        
        console.log('Total UNIQUE date ranges found:', ranges.length);
        console.log('Date ranges with row mappings:', ranges);
        
        if (ranges.length > 0) {
            // Store ranges globally for filtering
            window.analysisDateRanges = ranges;
            
            // Populate the dropdown
            populateAnalysisDateDropdown(ranges);
            
            // Show all by default
            filterAnalysisByDateRangeIndex(null);
        } else {
            console.warn('No valid date ranges found in columns O-P');
            hideAnalysisDateFilter();
        }
        
    } catch (error) {
        console.error('Error loading analysis date ranges:', error);
        hideAnalysisDateFilter();
    }
}

/**
 * Populate the analysis date range dropdown
 */
function populateAnalysisDateDropdown(ranges) {
    const dropdown = document.getElementById('analysisDateRange');
    if (!dropdown) {
        console.warn('Analysis date range dropdown not found');
        return;
    }
    
    // Clear existing options - NO "All Timeframes" option
    dropdown.innerHTML = '';
    
    // Add option for each date range
    ranges.forEach((range, index) => {
        const option = document.createElement('option');
        option.value = index;
        option.textContent = `${range.label}: ${range.startDateDisplay} - ${range.endDateDisplay}`;
        dropdown.appendChild(option);
    });
    
    // Show the filter section
    const filterSection = dropdown.closest('.date-filter-section');
    if (filterSection) {
        filterSection.style.display = 'block';
    }
    
    // Set first period as default and display it
    if (ranges.length > 0) {
        dropdown.value = '0';
        filterAnalysisByDateRangeIndex(0);
    }
    
    console.log('Populated dropdown with', ranges.length, 'date ranges, default to first period');
}

/**
 * Hide the analysis date filter section
 */
function hideAnalysisDateFilter() {
    const dropdown = document.getElementById('analysisDateRange');
    const filterSection = dropdown?.closest('.date-filter-section');
    if (filterSection) {
        filterSection.style.display = 'none';
    }
    
    // Show all analysis cards when no date filter
    document.querySelectorAll('.analysis-card').forEach(card => {
        card.style.display = 'block';
    });
    
    // Show both sections
    const findingsSection = document.querySelector('.key-findings');
    const recommendationsSection = document.querySelector('.recommendations');
    if (findingsSection) findingsSection.style.display = 'block';
    if (recommendationsSection) recommendationsSection.style.display = 'block';
}

// Add this helper function to check if a section should be visible
function checkAnalysisSectionVisibility(sectionSelector) {
    const section = document.querySelector(sectionSelector);
    if (!section) return;
    
    const cards = section.querySelectorAll('.analysis-card');
    const visibleCards = Array.from(cards).filter(card => card.style.display !== 'none');
    
    section.style.display = visibleCards.length > 0 ? 'block' : 'none';
}

/**
 * Filter analysis content by matching row indices
 * When a date range is selected, only show analysis data from the same row
 */
function filterAnalysisByDateRange() {
    const dropdown = document.getElementById('analysisDateRange');
    if (!dropdown) return;
    
    const selectedValue = dropdown.value;
    
    if (selectedValue === '') {
        // This shouldn't happen now, but handle it just in case
        filterAnalysisByDateRangeIndex(0);
    } else {
        const rangeIndex = parseInt(selectedValue);
        filterAnalysisByDateRangeIndex(rangeIndex);
    }
}

/**
 * Filter analysis content by matching row indices
 * When a date range is selected, only show analysis data from the same row
 */
function filterAnalysisByDateRangeIndex(rangeIndex) {
    console.log('Displaying Analysis for period index:', rangeIndex);

    if (!window.analysisPeriodsData || window.analysisPeriodsData.length === 0) {
        console.warn('No analysis periods data available');
        return;
    }

    // Get the period data for the selected range
    const periodData = window.analysisPeriodsData[rangeIndex];
    
    if (!periodData) {
        console.warn('No data found for period index:', rangeIndex);
        return;
    }

    // Update Key Findings section
    updateAnalysisSectionWithPeriodData('findings', periodData.findings);
    
    // Update Recommendations section
    updateAnalysisSectionWithPeriodData('recommendations', periodData.recommendations);
    
    // Show both sections
    document.querySelector('.key-findings').style.display = 'block';
    document.querySelector('.recommendations').style.display = 'block';
}

/**
 * Clear analysis filters
 */
function clearAnalysisFilters() {
    const dropdown = document.getElementById('analysisDateRange');
    if (dropdown && dropdown.options.length > 0) {
        // Reset to first period instead of "All"
        dropdown.value = '0';
        filterAnalysisByDateRangeIndex(0);
    }
}

function updateAnalysisSectionWithPeriodData(sectionType, items) {
    const sectionClass = sectionType === 'findings' ? 'key-findings' : 'recommendations';
    const cardPrefix = sectionType === 'findings' ? 'finding' : 'recommendation';
    
    const sectionElement = document.querySelector(`.${sectionClass}`);
    if (!sectionElement) return;
    
    const cardsGrid = sectionElement.querySelector('.analysis-cards-grid');
    if (!cardsGrid) return;
    
    // Clear existing cards
    cardsGrid.innerHTML = '';
    
    // Check if any content exists
    const hasAnyContent = items.some(item =>
        (item.title && item.title.trim() !== '') || 
        (item.content && item.content.trim() !== '')
    );
    
    if (!hasAnyContent) {
        sectionElement.style.display = 'none';
        return;
    }
    
    sectionElement.style.display = 'block';
    
    // Create all 4 cards for this section
    items.forEach((item, index) => {
        // Only create card if either title or content has data
        if ((item.title && item.title.trim() !== '') || 
            (item.content && item.content.trim() !== '')) {
            
            const cardNumber = index + 1;
            const card = createAnalysisCard(
                cardPrefix, 
                cardNumber, 
                item.title, 
                item.content
            );
            
            cardsGrid.appendChild(card);
        }
    });
}

/**
 * Load analysis data from Info sheet columns E-L - UPDATED VERSION
 * Now reads ALL rows and matches them with date range rows
 */
async function loadAnalysisData() {
    try {
        console.log('Loading analysis data from sheet:', currentCampaignId);
        
        // Fetch data from Info sheet - columns E-L for analysis, rows 1-100
        const data = await fetchSheetData(currentCampaignId, 'Info', 'E1:L100');
        
        console.log('Analysis data received:', data);
        
        if (!data || data.length < 2) {
            console.warn('Insufficient analysis data');
            hideAnalysisTab();
            return;
        }
        
        // Check if there's any actual content (skip header row)
        const hasContent = data.slice(1).some(row => 
            row.some(cell => cell && cell.toString().trim() !== '')
        );
        
        if (!hasContent) {
            console.warn('No analysis content found');
            hideAnalysisTab();
            return;
        }
        
        showAnalysisTab();
        
        // Store analysis data globally for filtering
        // Structure: array of periods, each with titles and contents
        window.analysisPeriodsData = [];
        
        // Process data starting from row 2 (index 1)
        // Each period uses TWO rows: titles row and content row
        for (let i = 1; i < data.length; i += 2) {
            const titleRow = data[i] || [];
            const contentRow = data[i + 1] || [];
            
            // Check if this row pair has any content
            const hasRowContent = [...titleRow, ...contentRow].some(cell => 
                cell && cell.toString().trim() !== ''
            );
            
            if (!hasRowContent) continue;
            
            // Findings: columns E, F, G, H (indices 0-3)
            const findings = [];
            for (let col = 0; col < 4; col++) {
                const title = titleRow[col] || '';
                const content = contentRow[col] || '';
                findings.push({ title, content });
            }
            
            // Recommendations: columns I, J, K, L (indices 4-7)
            const recommendations = [];
            for (let col = 4; col < 8; col++) {
                const title = titleRow[col] || '';
                const content = contentRow[col] || '';
                recommendations.push({ title, content });
            }
            
            window.analysisPeriodsData.push({
                periodIndex: Math.floor((i - 1) / 2), // Map to date range index
                findings: findings,
                recommendations: recommendations
            });
        }
        
        console.log('Processed analysis periods:', window.analysisPeriodsData.length);
        
        // Load date ranges AFTER analysis data is loaded
        await loadAnalysisDateRanges();
        
    } catch (error) {
        console.error('Error loading analysis data:', error);
        hideAnalysisTab();
    }
}

/**
 * Update analysis section with timeframe data
 */
function updateAnalysisSectionWithTimeframes(sectionType, timeframesData) {
    const sectionClass = sectionType === 'findings' ? 'key-findings' : 'recommendations';
    const cardPrefix = sectionType === 'findings' ? 'finding' : 'recommendation';
    
    // Find the cards grid container
    const sectionElement = document.querySelector(`.${sectionClass}`);
    if (!sectionElement) return;
    
    const cardsGrid = sectionElement.querySelector('.analysis-cards-grid');
    if (!cardsGrid) return;
    
    // Clear existing cards
    cardsGrid.innerHTML = '';
    
    // Check if any content exists
    const hasAnyContent = timeframesData.some(timeframe =>
        timeframe.some(item => 
            (item.title && item.title.trim() !== '') || 
            (item.content && item.content.trim() !== '')
        )
    );
    
    if (!hasAnyContent) {
        sectionElement.style.display = 'none';
        return;
    }
    
    sectionElement.style.display = 'block';
    
    // Create cards for each timeframe and each finding/recommendation
    timeframesData.forEach((timeframe, timeframeIndex) => {
        timeframe.forEach((item, itemIndex) => {
            // Only create card if either title or content has data
            if ((item.title && item.title.trim() !== '') || 
                (item.content && item.content.trim() !== '')) {
                
                const cardNumber = itemIndex + 1;
                const card = createAnalysisCard(
                    cardPrefix, 
                    cardNumber, 
                    item.title, 
                    item.content
                );
                
                // Add data attributes for filtering
                card.dataset.timeframeIndex = timeframeIndex;
                card.dataset.itemIndex = itemIndex;
                
                cardsGrid.appendChild(card);
            }
        });
    });
}


/**
 * Count content pieces within a date range
 */
function countContentInDateRange(startDate, endDate) {
    if (!contentData || contentData.length === 0) return 0;
    
    return contentData.filter(content => {
        return isDateInRange(content.postedDate, startDate, endDate);
    }).length;
}

/**
 * Format date for display
 */
function formatDateDisplay(dateString) {
    if (!dateString) return '';
    
    const date = parseContentDate(dateString);
    if (!date) return dateString;
    
    return date.toLocaleDateString('en-US', {
        month: 'short',
        day: 'numeric',
        year: 'numeric'
    });
}

/**
 * Set default date range for Influencers tab based on content data
 */
function setDefaultInfluencersDateRange() {
    if (!contentData || contentData.length === 0) return;
    
    // Find oldest and newest dates
    let oldestDate = null;
    let newestDate = null;
    
    contentData.forEach(content => {
        if (content.postedDate) {
            const date = parseContentDate(content.postedDate);
            if (date) {
                if (!oldestDate || date < oldestDate) {
                    oldestDate = date;
                }
                if (!newestDate || date > newestDate) {
                    newestDate = date;
                }
            }
        }
    });
    
    // Set the date inputs if we found valid dates
    if (oldestDate && newestDate) {
        const startDateInput = document.getElementById('influencersStartDate');
        const endDateInput = document.getElementById('influencersEndDate');
        
        if (startDateInput && endDateInput) {
            // Add 1 day to the newest date to include it in the range
            const adjustedEndDate = new Date(newestDate);
            adjustedEndDate.setDate(adjustedEndDate.getDate() + 1);
            
            // Format dates as YYYY-MM-DD for input type="date"
            startDateInput.value = formatDateForInput(oldestDate);
            endDateInput.value = formatDateForInput(adjustedEndDate);
            
            console.log('Set default Influencers date range:', {
                originalStart: oldestDate,
                originalEnd: newestDate,
                adjustedEnd: adjustedEndDate,
                startInput: startDateInput.value,
                endInput: endDateInput.value
            });
        }
    }
}

/**
 * Clear Influencers tab date filters
 */
function clearInfluencersDateFilters() {
    document.getElementById('influencersStartDate').value = '';
    document.getElementById('influencersEndDate').value = '';
    
    // Reset to show all influencers
    filterInfluencers();
}

/**
 * Filter influencers by date range
 */
function filterInfluencersByDate() {
    // Simply refresh the display - it will automatically use the date filters
    filterInfluencers();
}

/**
 * Toggle comparison mode on/off
 */
function toggleComparison() {
    const compareToggle = document.getElementById('compareToggle');
    const comparisonRangeGroup = document.getElementById('comparisonRangeGroup');
    const swapDatesBtn = document.getElementById('swapDatesBtn');
    
    isComparisonMode = compareToggle.checked;
    
    console.log('toggleComparison called:', isComparisonMode);
    
    if (isComparisonMode) {
        useDateFilterForCalculations = true;
        comparisonRangeGroup.style.display = 'flex';
        if (swapDatesBtn) swapDatesBtn.classList.add('visible');
        // Auto-fill comparison dates based on Period 1
        suggestComparisonDates();
    } else {
        comparisonRangeGroup.style.display = 'none';
        if (swapDatesBtn) swapDatesBtn.classList.remove('visible');
        
        // Clear comparison dates
        document.getElementById('contentCompareStartDate').value = '';
        document.getElementById('contentCompareEndDate').value = '';
        
        // Clear comparison indicators
        clearComparisonIndicators();
        
        // Refresh to show just Period 1 data
        filterContent();
    }
}

// =============================================================================
// VIDEO POPUP FUNCTIONS - ANCHOR TEXT VERSION
// =============================================================================

/**
 * Open video popup with platform-specific embed
 */
function openVideoPopup(videoUrl) {
    const popup = document.getElementById('videoPopup');
    const videoContainer = document.getElementById('videoPopupContainer');
    
    if (!popup || !videoContainer) {
        console.error('Video popup elements not found');
        return;
    }
    
    // Clear previous content
    videoContainer.innerHTML = '';
    
    // Show loading state
    videoContainer.classList.add('loading');
    
    // Determine platform and create appropriate embed
    const embedHTML = createVideoEmbed(videoUrl);
    
    if (embedHTML) {
        setTimeout(() => {
            videoContainer.classList.remove('loading');
            videoContainer.innerHTML = embedHTML;
            popup.style.display = 'flex';
            document.body.style.overflow = 'hidden';
        }, 100);
    } else {
        videoContainer.classList.remove('loading');
        alert('Unable to load video. Please check the URL format.');
    }
}

/**
 * Close video popup
 */
function closeVideoPopup() {
    const popup = document.getElementById('videoPopup');
    const videoContainer = document.getElementById('videoPopupContainer');
    
    if (popup) {
        popup.style.display = 'none';
        document.body.style.overflow = '';
    }
    
    if (videoContainer) {
        videoContainer.innerHTML = '';
    }
}

/**
 * Create video embed based on platform
 */
function createVideoEmbed(url) {
    if (!url) return null;
    
    const cleanUrl = url.trim();
    
    // TikTok - Vertical video (9:16 aspect ratio)
    if (cleanUrl.includes('tiktok.com')) {
        const videoId = extractTikTokVideoId(cleanUrl);
        if (videoId) {
            return `
                <iframe 
                    src="https://www.tiktok.com/embed/v2/${videoId}" 
                    allowfullscreen 
                    scrolling="no" 
                    allow="encrypted-media;"
                    style="width: 325px; height: 730px; max-height: 90vh; border: none; border-radius: 8px;">
                </iframe>
            `;
        }
    }
    
    // YouTube - Horizontal video (16:9 aspect ratio)
    if (cleanUrl.includes('youtube.com') || cleanUrl.includes('youtu.be')) {
        const videoId = extractYouTubeVideoId(cleanUrl);
        if (videoId) {
            // Check if it's a YouTube Short (vertical)
            const isShort = cleanUrl.includes('/shorts/');
            
            if (isShort) {
                return `
                    <iframe 
                        src="https://www.youtube.com/embed/${videoId}?autoplay=1" 
                        allowfullscreen 
                        allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                        style="width: 360px; height: 640px; max-height: 90vh; border: none; border-radius: 8px;">
                    </iframe>
                `;
            } else {
                return `
                    <iframe 
                        src="https://www.youtube.com/embed/${videoId}?autoplay=1" 
                        allowfullscreen 
                        allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                        style="width: 90vw; max-width: 1000px; height: calc(90vw * 9 / 16); max-height: 562px; border: none; border-radius: 8px;">
                    </iframe>
                `;
            }
        }
    }
    
    // Instagram - Usually vertical (9:16 aspect ratio for Reels/Stories, can be square for posts)
    if (cleanUrl.includes('instagram.com')) {
        return `
            <iframe 
                src="${cleanUrl}embed/" 
                allowfullscreen 
                scrolling="no"
                style="width: 400px; height: 700px; max-height: 90vh; max-width: 90vw; border: none; border-radius: 8px; background: #000;">
            </iframe>
        `;
    }
    
    // Fallback
    return `
        <div style="width: 100%; height: 100%; display: flex; align-items: center; justify-content: center; flex-direction: column; color: white; text-align: center; padding: 2rem;">
            <p style="font-size: 1.2rem; margin-bottom: 1rem;">Unable to embed this video directly.</p>
            <a href="${cleanUrl}" target="_blank" style="color: #fc0060; text-decoration: none; font-size: 1.1rem; font-weight: 600; padding: 1rem 2rem; background: white; border-radius: 8px;">
                Open Video in New Tab
            </a>
        </div>
    `;
}

/**
 * Extract TikTok video ID from URL
 */
function extractTikTokVideoId(url) {
    const match = url.match(/\/video\/(\d+)/);
    return match ? match[1] : null;
}

/**
 * Extract YouTube video ID from URL
 */
function extractYouTubeVideoId(url) {
    let match = url.match(/[?&]v=([^&]+)/);
    if (match) return match[1];
    
    match = url.match(/youtu\.be\/([^?]+)/);
    if (match) return match[1];
    
    match = url.match(/\/embed\/([^?]+)/);
    if (match) return match[1];
    
    return null;
}

/**
 * Process analysis text to convert markdown-style links to clickable video links
 * Format: [Anchor Text](URL)
 */
function processAnalysisTextWithVideoLinks(text) {
    if (!text) return '';
    
    // Pattern to match markdown-style links: [text](url)
    const markdownLinkPattern = /\[([^\]]+)\]\((https?:\/\/[^\s)]+)\)/g;
    
    let processedText = text;
    
    // Replace markdown links with clickable video links
    processedText = processedText.replace(markdownLinkPattern, (match, anchorText, url) => {
        // Check if it's a video URL
        if (url.includes('tiktok.com') || url.includes('youtube.com') || 
            url.includes('youtu.be') || url.includes('instagram.com')) {
            // Use data attribute and addEventListener approach instead of inline onclick
            return `<a href="#" class="video-link" data-video-url="${url}">${anchorText}</a>`;
        }
        // Regular external link
        return `<a href="${url}" target="_blank" rel="noopener noreferrer" class="external-link">${anchorText}</a>`;
    });
    
    // Also handle plain URLs that aren't in markdown format
    const plainUrlPattern = /(^|[^"\(])(https?:\/\/[^\s<]+)(?![^\[]*\])/g;
    processedText = processedText.replace(plainUrlPattern, (match, prefix, url) => {
        if (url.includes('tiktok.com') || url.includes('youtube.com') || 
            url.includes('youtu.be') || url.includes('instagram.com')) {
            return `${prefix}<a href="#" class="video-link" data-video-url="${url}">
                <span class="video-link-icon">▶️</span> Watch Video
            </a>`;
        }
        return `${prefix}<a href="${url}" target="_blank" rel="noopener noreferrer" class="external-link">${url}</a>`;
    });
    
    return processedText;
}

/**
 * Clear all date filters
 */
function clearAllDateFilters() {
    document.getElementById('contentStartDate').value = '';
    document.getElementById('contentEndDate').value = '';
    document.getElementById('contentCompareStartDate').value = '';
    document.getElementById('contentCompareEndDate').value = '';
    document.getElementById('compareToggle').checked = false;
    
    isComparisonMode = false;
    useDateFilterForCalculations = false; // RESET THE FLAG
    document.getElementById('comparisonRangeGroup').style.display = 'none';
    
    clearComparisonIndicators();
    
    // Repopulate default dates (but they won't filter)
    setDefaultDateRange();
    
    // Update metrics to show ALL content
    updateContentMetricsSummary();
    
    // Update visual display
    filterContent();
}

/**
 * Set default date range based on content data
 */
function setDefaultDateRange() {
    if (!contentData || contentData.length === 0) return;
    
    // Find oldest and newest dates
    let oldestDate = null;
    let newestDate = null;
    
    contentData.forEach(content => {
        if (content.postedDate) {
            const date = parseContentDate(content.postedDate);
            if (date) {
                if (!oldestDate || date < oldestDate) {
                    oldestDate = date;
                }
                if (!newestDate || date > newestDate) {
                    newestDate = date;
                }
            }
        }
    });
    
    // Set the date inputs if we found valid dates
    if (oldestDate && newestDate) {
        const startDateInput = document.getElementById('contentStartDate');
        const endDateInput = document.getElementById('contentEndDate');
        
        if (startDateInput && endDateInput) {
            // Add 1 day to the newest date to include it in the range
            const adjustedEndDate = new Date(newestDate);
            adjustedEndDate.setDate(adjustedEndDate.getDate() + 1);
            
            // Format dates as YYYY-MM-DD for input type="date"
            startDateInput.value = formatDateForInput(oldestDate);
            endDateInput.value = formatDateForInput(adjustedEndDate);
            
            console.log('Set default date range:', {
                originalStart: oldestDate,
                originalEnd: newestDate,
                adjustedEnd: adjustedEndDate,
                startInput: startDateInput.value,
                endInput: endDateInput.value
            });
            
            // IMPORTANT: Enable date filtering and update metrics immediately
            useDateFilterForCalculations = true;
            
            // CRITICAL: Ensure comparison mode is OFF by default
            isComparisonMode = false;
            const compareToggle = document.getElementById('compareToggle');
            if (compareToggle) {
                compareToggle.checked = false;
            }
            
            // Hide comparison range group
            const comparisonRangeGroup = document.getElementById('comparisonRangeGroup');
            if (comparisonRangeGroup) {
                comparisonRangeGroup.style.display = 'none';
            }
            
            // Update metrics summary with the full date range
            updateContentMetricsSummary();
            
            setTimeout(() => {
                updateContentMetricsSummary();
                console.log('✅ Initial metrics calculated and displayed for full date range');
            }, 0);
            
            console.log('Initial metrics calculated for full date range');
        }
    }
}

/**
 * Format date object to YYYY-MM-DD string for date input
 */
function formatDateForInput(date) {
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
}

/**
 * Calculate metrics for a specific date range
 */

function calculateMetricsForDateRange(startDate, endDate, searchTerm, platformFilter, mediaTypeFilter) {
    let visibleContent = contentData.filter(content => {
        const title = content.title.toLowerCase();
        const platform = content.platform.toLowerCase();
        const influencer = content.influencer.toLowerCase();
        const mediaType = content.mediaType || '';
        
        // Search match
        const searchMatch = !searchTerm || 
            title.includes(searchTerm) || 
            influencer.includes(searchTerm);
        
        // Platform match
        let platformMatch = true;
        if (platformFilter === 'bonus') {
            platformMatch = content.bonus === true;
        } else if (platformFilter) {
            platformMatch = platform === platformFilter.toLowerCase();
        }
        
        // Media type match
        const mediaTypeMatch = !mediaTypeFilter || mediaType === mediaTypeFilter;
        
        // Date range match - UPDATED: Include content with missing dates when filtering
        let dateMatch = true;
        if (startDate && endDate) {
            // If content has a date, check if it's in range
            if (content.postedDate) {
                dateMatch = isDateInRange(content.postedDate, startDate, endDate);
            }
            // If content has no date, still include it (dateMatch stays true)
            // This ensures we don't lose data for content without dates
        }
        
        return searchMatch && platformMatch && mediaTypeMatch && dateMatch;
    });
    
    // Calculate UNIQUE influencers count based on Column A (influencer name)
    const uniqueInfluencers = new Set();
    visibleContent.forEach(content => {
        if (content.influencer && content.influencer.trim() !== '') {
            uniqueInfluencers.add(content.influencer.toLowerCase().trim());
        }
    });
    const influencersCount = uniqueInfluencers.size;
    
    // Calculate UNIQUE accounts count based on Column B (profile link)
    const uniqueAccounts = new Set();
    visibleContent.forEach(content => {
        if (content.profileLink && content.profileLink.trim() !== '') {
            uniqueAccounts.add(content.profileLink.toLowerCase().trim());
        }
    });
    const accountsCount = uniqueAccounts.size;
    
    // Calculate pieces of content (sum all, with special handling for Instagram Stories)
    let piecesOfContent = 0;
    visibleContent.forEach(content => {
        const isInstagramStories = content.platform.toLowerCase() === 'instagram' && 
                                  content.mediaType === 'story';
        
        if (isInstagramStories) {
            // For Instagram Stories, add the number of stories (Column T)
            piecesOfContent += content.analytics.stories || 1;
        } else {
            // For all other content, count as 1 piece
            piecesOfContent += 1;
        }
    });
    
    // Calculate ALL OTHER METRICS as simple sums
    const totals = visibleContent.reduce((acc, content) => {
        // Just sum everything
        acc.views += content.metrics.views || 0;
        acc.likes += content.metrics.likes || 0;
        acc.comments += content.metrics.comments || 0;
        acc.shares += content.metrics.shares || 0;
        acc.saves += content.metrics.saves || 0;
        acc.reach += content.metrics.reach || 0;
        acc.linkClicks += content.analytics.linkClicks || 0;
        acc.stickerTaps += content.analytics.stickerTaps || 0;
        
        return acc;
    }, {
        views: 0,
        likes: 0,
        comments: 0,
        shares: 0,
        saves: 0,
        reach: 0,
        linkClicks: 0,
        stickerTaps: 0
    });
    
    let totalEngagementForER = 0;
    let totalViewsForER = 0;
    
    visibleContent.forEach(content => {
    const isInstagramPostsOrStories = content.platform.toLowerCase() === 'instagram' && 
                                      (content.mediaType === 'post' || content.mediaType === 'story');
    
    // Exclude Instagram Posts AND Stories from ER calculation
    if (!isInstagramPostsOrStories) {
            const views = content.metrics.views || 0;
            
            if (views > 0) {
                const engagement = (content.metrics.likes || 0) + 
                                 (content.metrics.comments || 0) + 
                                 (content.metrics.shares || 0) + 
                                 (content.metrics.saves || 0);
                
                totalEngagementForER += engagement;
                totalViewsForER += views;
            }
        }
    });
    
    // Aggregate ER = (Total Engagement / Total Views) * 100
    const calculatedER = totalViewsForER > 0 ? (totalEngagementForER / totalViewsForER) * 100 : 0;
    
    console.log('Metrics calculated:', {
        influencersCount: influencersCount,
        accountsCount: accountsCount,
        piecesOfContent: piecesOfContent,
        views: totals.views,
        er: calculatedER,
        totalEngagementForER: totalEngagementForER,
        totalViewsForER: totalViewsForER
    });
    
    return {
        influencersCount,
        accountsCount,
        piecesOfContent,
        views: totals.views,
        likes: totals.likes,
        comments: totals.comments,
        shares: totals.shares,
        saves: totals.saves,
        reach: totals.reach,
        linkClicks: totals.linkClicks,
        stickerTaps: totals.stickerTaps,
        er: calculatedER  // ✅ Make sure this is returned
    };
}

// =============================================================================
// VIDEO POPUP FUNCTIONS - ANCHOR TEXT VERSION
// =============================================================================

/**
 * Open video popup with platform-specific embed
 */
function openVideoPopup(videoUrl) {
    const popup = document.getElementById('videoPopup');
    const videoContainer = document.getElementById('videoPopupContainer');
    
    if (!popup || !videoContainer) {
        console.error('Video popup elements not found');
        return;
    }
    
    // Clear previous content
    videoContainer.innerHTML = '';
    
    // Show loading state
    videoContainer.classList.add('loading');
    
    // Determine platform and create appropriate embed
    const embedHTML = createVideoEmbed(videoUrl);
    
    if (embedHTML) {
        setTimeout(() => {
            videoContainer.classList.remove('loading');
            videoContainer.innerHTML = embedHTML;
            popup.style.display = 'flex';
            document.body.style.overflow = 'hidden';
        }, 100);
    } else {
        videoContainer.classList.remove('loading');
        alert('Unable to load video. Please check the URL format.');
    }
}

/**
 * Close video popup
 */
function closeVideoPopup() {
    const popup = document.getElementById('videoPopup');
    const videoContainer = document.getElementById('videoPopupContainer');
    
    if (popup) {
        popup.style.display = 'none';
        document.body.style.overflow = '';
    }
    
    if (videoContainer) {
        videoContainer.innerHTML = '';
    }
}

/**
 * Create an individual analysis card with video link support
 */
function createAnalysisCard(cardPrefix, cardNumber, title, content) {
    const card = document.createElement('div');
    card.className = `analysis-card ${cardPrefix}-card-${cardNumber}`;
    
    const defaultTitles = {
        finding: [`Key Finding ${cardNumber}`, `Performance Insight ${cardNumber}`, `Data Point ${cardNumber}`, `Research Finding ${cardNumber}`],
        recommendation: [`Recommendation ${cardNumber}`, `Strategy Suggestion ${cardNumber}`, `Optimization Tip ${cardNumber}`, `Action Item ${cardNumber}`]
    };
    
    const defaultIcons = {
        finding: ['📊', '🎯', '📈', '👥'],
        recommendation: ['💡', '🚀', '🔄', '📱']
    };
    
    const cardType = cardPrefix === 'finding' ? 'finding' : 'recommendation';
    
    // Use default title if title is empty, undefined, or just whitespace
    const displayTitle = (title && title.trim() !== '') ? title : defaultTitles[cardType][(cardNumber - 1) % 4];
    const displayContent = (content && content.trim() !== '') ? content : `Loading ${cardType}...`;
    
    // FIXED: Get icon with proper modulo to cycle through available icons
    const iconIndex = (cardNumber - 1) % defaultIcons[cardType].length;
    const icon = defaultIcons[cardType][iconIndex];
    
    // Safety check: if icon is undefined, use a default emoji
    const safeIcon = icon || (cardType === 'finding' ? '📊' : '💡');
    
    // Process content to convert video URLs to clickable links
    const processedContent = processAnalysisTextWithVideoLinks(displayContent);
    
    card.innerHTML = `
        <div class="analysis-card-header">
            <div class="analysis-card-icon">${safeIcon}</div>
            <div class="analysis-card-title">${escapeHtml(displayTitle)}</div>
        </div>
        <div class="analysis-card-content">${processedContent}</div>
    `;
    
    // Add event listeners to video links after card is created
    setTimeout(() => {
        const videoLinks = card.querySelectorAll('.video-link');
        videoLinks.forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const videoUrl = link.getAttribute('data-video-url');
                if (videoUrl) {
                    openVideoPopup(videoUrl);
                }
            });
        });
    }, 0);
    
    return card;
}

/**
 * Suggest comparison dates based on Period 1 selection
 * Uses the same time range but shifted earlier
 */
function suggestComparisonDates() {
    const startDate = document.getElementById('contentStartDate')?.value;
    const endDate = document.getElementById('contentEndDate')?.value;
    
    if (!startDate || !endDate) {
        console.log('No Period 1 dates selected yet');
        return;
    }
    
    const start = new Date(startDate);
    const end = new Date(endDate);
    
    // Calculate the number of days in Period 1
    const timeDiff = end - start;
    const daysDiff = Math.ceil(timeDiff / (1000 * 60 * 60 * 24)) + 1; // +1 to include both start and end
    
    // Try to go back by the same duration
    // Period 2 end date = Period 1 start date - 1 day
    let compareEnd = new Date(start);
    compareEnd.setDate(compareEnd.getDate() - 1);
    
    // Period 2 start date = Period 2 end date - (duration - 1)
    let compareStart = new Date(compareEnd);
    compareStart.setDate(compareStart.getDate() - (daysDiff - 1));
    
    // Check if we're going to the previous year
    const startYear = start.getFullYear();
    
    if (compareStart.getFullYear() < startYear) {
        // Would cross into previous year - adjust to stay in same year
        console.log('Period 2 would go to previous year. Adjusting to stay in', startYear);
        
        // Option: Start from Jan 1 and keep same duration
        compareStart = new Date(startYear, 0, 1); // Jan 1st of same year
        compareEnd = new Date(compareStart);
        compareEnd.setDate(compareEnd.getDate() + daysDiff - 1);
        
        // If this overlaps with or exceeds Period 1, adjust to end right before Period 1
        if (compareEnd >= start) {
            compareEnd = new Date(start);
            compareEnd.setDate(compareEnd.getDate() - 1);
            
            compareStart = new Date(compareEnd);
            compareStart.setDate(compareStart.getDate() - (daysDiff - 1));
            
            // If STILL going to previous year (Period 1 is very early in the year)
            // then just use Jan 1 to day before Period 1 as the comparison range
            if (compareStart.getFullYear() < startYear) {
                compareStart = new Date(startYear, 0, 1);
                compareEnd = new Date(start);
                compareEnd.setDate(compareEnd.getDate() - 1);
                
                console.log('Period 1 too early in year. Using Jan 1 to day before Period 1');
            }
        }
    }
    
    // Set the comparison date inputs
    document.getElementById('contentCompareStartDate').value = formatDateForInput(compareStart);
    document.getElementById('contentCompareEndDate').value = formatDateForInput(compareEnd);
    
    console.log('Suggested comparison period:', {
        period1: { 
            start: startDate, 
            end: endDate, 
            days: daysDiff,
            year: startYear
        },
        period2: { 
            start: formatDateForInput(compareStart), 
            end: formatDateForInput(compareEnd), 
            days: Math.ceil((compareEnd - compareStart) / (1000 * 60 * 60 * 24)) + 1,
            year: compareStart.getFullYear()
        }
    });
    
    // Trigger filter to show comparison
    filterContent();
}

/**
 * Update comparison dates when Period 1 dates change
 */
function onPeriod1DateChange() {
    // Enable date filtering since user manually changed dates
    useDateFilterForCalculations = true;
    
    // If comparison mode is active, update the suggested dates
    if (isComparisonMode) {
        suggestComparisonDates();
    } else {
        // User manually changed dates - apply filter
        filterContent();
    }
    
    // Always update metrics summary when dates change
    updateContentMetricsSummary();
}

/**
 * Display comparison indicators
 */
/**
 * Display comparison indicators
 */
function displayComparison(period1Metrics, period2Metrics) {
    console.log('Displaying comparison', { period1Metrics, period2Metrics });
    
    const metricsSummary = document.getElementById('contentMetricsSummary');
    if (metricsSummary) {
        metricsSummary.classList.add('comparison-mode');
    }
    
    const metrics = [
        { id: 'contentInfluencersCount', key: 'influencersCount', format: 'number' },
        { id: 'contentAccountsCount', key: 'accountsCount', format: 'number' },
        { id: 'contentCount', key: 'piecesOfContent', format: 'number' },
        { id: 'contentTotalViews', key: 'views', format: 'number' },
        { id: 'contentTotalLikes', key: 'likes', format: 'number' },
        { id: 'contentTotalComments', key: 'comments', format: 'number' },
        { id: 'contentTotalShares', key: 'shares', format: 'number' },
        { id: 'contentTotalSaves', key: 'saves', format: 'number' },
        { id: 'contentTotalReach', key: 'reach', format: 'number' },
        { id: 'contentTotalLinkClicks', key: 'linkClicks', format: 'number' },
        { id: 'contentTotalER', key: 'er', format: 'percentage' }
    ];
    
    metrics.forEach(metric => {
        const value1 = period1Metrics[metric.key] || 0;
        const value2 = period2Metrics[metric.key] || 0;
        
        console.log(`Processing metric ${metric.id}: P1=${value1}, P2=${value2}`);
        
        // Update current period value
        const currentElement = document.getElementById(metric.id);
        if (currentElement) {
            const formattedValue1 = metric.format === 'percentage' ? 
                `${value1.toFixed(2)}%` : 
                formatNumber(value1);
            currentElement.textContent = formattedValue1;
        }
        
        // Update previous period value
        const previousElement = document.getElementById(`${metric.id}-previous`);
        if (previousElement) {
            if (value2 === 0 && value1 === 0) {
                previousElement.textContent = '-';
            } else {
                const formattedValue2 = metric.format === 'percentage' ? 
                    `${value2.toFixed(2)}%` : 
                    formatNumber(value2);
                previousElement.textContent = formattedValue2;
                console.log(`Set previous value for ${metric.id}: ${formattedValue2}`);
            }
        } else {
            console.warn(`Previous element not found: ${metric.id}-previous`);
        }
        
        // Calculate and display trend
        const trendElement = document.getElementById(`${metric.id}-trend`);
        if (trendElement) {
            if (value2 === 0 && value1 === 0) {
                trendElement.classList.remove('active', 'positive', 'negative', 'neutral');
                trendElement.innerHTML = '';
                return;
            }
            
            let percentageChange = 0;
            if (value2 !== 0) {
                percentageChange = ((value1 - value2) / value2) * 100;
            } else if (value1 > 0) {
                percentageChange = 100;
            }
            
            const arrow = percentageChange > 0 ? '▲' : percentageChange < 0 ? '▼' : '→';
            const changeClass = percentageChange > 0 ? 'positive' : percentageChange < 0 ? 'negative' : 'neutral';
            const absChange = Math.abs(percentageChange);
            
            trendElement.innerHTML = `
                <span class="trend-arrow">${arrow}</span>
                <span class="trend-value">${absChange.toFixed(1)}%</span>
            `;
            
            trendElement.classList.add('active', changeClass);
            trendElement.classList.remove(
                ...['positive', 'negative', 'neutral'].filter(c => c !== changeClass)
            );
            
            console.log(`Set trend for ${metric.id}: ${arrow} ${absChange.toFixed(1)}%`);
        } else {
            console.warn(`Trend element not found: ${metric.id}-trend`);
        }
    });
}

function clearComparisonIndicators() {
    const metricsSummary = document.getElementById('contentMetricsSummary');
    if (metricsSummary) {
        metricsSummary.classList.remove('comparison-mode');
    }
    
    // Clear all trend badges
    const trendElements = document.querySelectorAll('.metric-trend');
    trendElements.forEach(el => {
        el.innerHTML = '';
        el.classList.remove('active', 'positive', 'negative', 'neutral');
    });
    
    // Reset previous period values to "-"
    const previousElements = document.querySelectorAll('[id$="-previous"]');
    previousElements.forEach(el => {
        el.textContent = '-';
    });
}

/**
 * Update content metrics summary with comparison support
 */
function updateContentMetricsSummary() {
    // SAFETY CHECK: Don't run if no data
    if (!contentData || contentData.length === 0) {
        console.warn('No content data available for metrics summary');
        return;
    }
    
    console.log('📊 Updating content metrics summary');
    console.log('useDateFilterForCalculations:', useDateFilterForCalculations);
    console.log('isComparisonMode:', isComparisonMode);

    // Get filter values
    const searchTerm = document.getElementById('contentSearchInput')?.value.toLowerCase() || '';
    const platformFilter = document.getElementById('contentPlatformFilter')?.value || '';
    const mediaTypeFilter = document.getElementById('contentMediaTypeFilter')?.value || '';
    
    // Get date ranges
    const startDate = document.getElementById('contentStartDate')?.value || '';
    const endDate = document.getElementById('contentEndDate')?.value || '';
    const compareStartDate = document.getElementById('contentCompareStartDate')?.value || '';
    const compareEndDate = document.getElementById('contentCompareEndDate')?.value || '';
    
    console.log('Filters:', { searchTerm, platformFilter, mediaTypeFilter, startDate, endDate, useDateFilterForCalculations });
    
    // Calculate Period 1 metrics
    period1Data = calculateMetricsForDateRange(
        useDateFilterForCalculations ? startDate : '', 
        useDateFilterForCalculations ? endDate : '', 
        searchTerm, 
        platformFilter, 
        mediaTypeFilter
    );
    
    console.log('Period 1 data calculated:', period1Data);
    
    // Get the metrics summary container
    const metricsSummary = document.getElementById('contentMetricsSummary');
    if (!metricsSummary) {
        console.error('Metrics summary container not found');
        return;
    }
    
    // Determine platform-specific display logic
    const isInstagramStoriesOnly = platformFilter === 'Instagram' && mediaTypeFilter === 'story';
    const isInstagramPostsOnly = platformFilter === 'Instagram' && mediaTypeFilter === 'post';
    const isYouTubeOnly = platformFilter === 'YouTube';
    
    console.log('Platform filters:', { platformFilter, mediaTypeFilter, isInstagramStoriesOnly, isInstagramPostsOnly });
    
    // ==================================================================
    // UPDATE ALL METRIC VALUES
    // ==================================================================
    
    // Helper function to update element value
    const updateMetricValue = (id, value, isPercentage = false) => {
        const element = document.getElementById(id);
        if (element) {
            const formatted = isPercentage ? `${value.toFixed(2)}%` : formatNumber(value);
            element.textContent = formatted;
            console.log(`✓ Updated ${id} = ${formatted}`);
        } else {
            console.warn(`❌ Element not found: ${id}`);
        }
    };
    
    // Update all metrics
    updateMetricValue('contentInfluencersCount', period1Data.influencersCount);
    updateMetricValue('contentAccountsCount', period1Data.accountsCount);
    updateMetricValue('contentCount', period1Data.piecesOfContent);
    updateMetricValue('contentTotalViews', period1Data.views);
    updateMetricValue('contentTotalLikes', period1Data.likes);
    updateMetricValue('contentTotalComments', period1Data.comments);
    updateMetricValue('contentTotalShares', period1Data.shares);
    updateMetricValue('contentTotalSaves', period1Data.saves);
    updateMetricValue('contentTotalLinkClicks', period1Data.linkClicks);
    updateMetricValue('contentTotalER', period1Data.er, true);
    
    console.log('✅ All metric values updated successfully');
    
    // ==================================================================
    // HANDLE SAVES/STICKER TAPS
    // ==================================================================
    const savesElement = document.getElementById('contentTotalSaves');
    const savesLabel = document.getElementById('savesLabel');
    if (savesElement && savesLabel) {
        if (isInstagramStoriesOnly) {
            savesLabel.textContent = 'Sticker Taps';
            savesElement.textContent = formatNumber(period1Data.stickerTaps || 0);
        } else {
            savesLabel.textContent = 'Total Saves';
            savesElement.textContent = formatNumber(period1Data.saves);
        }
    }
    
    // ==================================================================
    // SHOW/HIDE METRICS BASED ON PLATFORM
    // ==================================================================
    const metricsToToggle = [
        { id: 'contentTotalViews', hide: isInstagramPostsOnly },
        { id: 'contentTotalComments', hide: period1Data.comments === 0 },
        { id: 'contentTotalShares', hide: period1Data.shares === 0 || isYouTubeOnly },
        { id: 'contentTotalSaves', hide: isInstagramStoriesOnly ? false : (period1Data.saves === 0 || isYouTubeOnly) },
        { id: 'contentTotalLinkClicks', hide: period1Data.linkClicks === 0 },
        { id: 'contentTotalER', hide: isInstagramStoriesOnly || isInstagramPostsOnly || period1Data.er === 0 }
    ];
    
    metricsToToggle.forEach(metric => {
        const element = document.getElementById(metric.id);
        // Use data-metric attribute to find the parent card
        const card = element?.closest('[data-metric]') || element?.closest('.summary-metric');
        
        if (card) {
            if (metric.hide) {
                card.style.display = 'none';
            } else {
                // Remove inline display style to let CSS handle it
                card.style.display = '';
            }
            console.log(`Metric ${metric.id}: ${metric.hide ? 'HIDDEN' : 'SHOWN'}`);
        } else {
            console.warn(`Could not find card for metric: ${metric.id}`);
        }
    });

    // ==================================================================
    // HANDLE COMPARISON MODE
    // ==================================================================
    if (isComparisonMode && compareStartDate && compareEndDate) {
        console.log('Comparison mode active, calculating Period 2');
        
        // Add comparison mode class
        metricsSummary.classList.add('comparison-mode');
        
        // Calculate Period 2 metrics
        period2Data = calculateMetricsForDateRange(
            compareStartDate, 
            compareEndDate, 
            searchTerm, 
            platformFilter, 
            mediaTypeFilter
        );
        
        console.log('Period 2 data:', period2Data);
        
        // Display comparison
        displayComparison(period1Data, period2Data);
    } else {
        console.log('Comparison mode not active or dates missing');
        // Clear comparison indicators if not in comparison mode
        metricsSummary.classList.remove('comparison-mode');
        clearComparisonIndicators();
    }
}

// Metric configurations for charts
const metricConfigs = {
    'Likes': { color: '#ff6b8a', icon: '❤️', category: 'engagement' },
    'Comments': { color: '#4ecdc4', icon: '💬', category: 'engagement' },
    'Views': { color: '#45b7d1', icon: '👁', category: 'reach' },
    'Shares': { color: '#96ceb4', icon: '📤', category: 'engagement' },
    'Saves': { color: '#feca57', icon: '🔖', category: 'actions' },
    'Link Clicks': { color: '#ff9ff3', icon: '🔗', category: 'actions' },
    'Reach': { color: '#54a0ff', icon: '📊', category: 'reach' },
    'Impressions': { color: '#5f27cd', icon: '👀', category: 'reach' }
};

// ---- Filters config ----
const PLATFORM_LABELS = {
  all: 'All',
  tiktok: 'TikTok',
  instagram: 'Instagram',
  youtube: 'YouTube',
  linkedin: 'LinkedIn',
  facebook: 'Facebook',
  twitter: 'Twitter',
  bonus: 'Bonus'
};

// Map mediaType -> subfilter id/label per platform (IDs MATCH updateResultsDisplay)
const MEDIA_SUBFILTER_MAP = {
  instagram: {
    post:  { id: 'instagram_posts',   label: 'Posts' },
    reel:  { id: 'instagram_reel',    label: 'Reels' },
    story: { id: 'instagram_stories', label: 'Stories' }
  },
  youtube: {
    short:       { id: 'youtube_shorts',       label: 'Shorts' },
    integration: { id: 'youtube_integrations', label: 'Integrations' }
  },
  linkedin: {
    post:  { id: 'linkedin_post',  label: 'Post' },
    video: { id: 'linkedin_video', label: 'Video' }
  },
  facebook: {
    post:  { id: 'facebook_post',  label: 'Post' },
    reel:  { id: 'facebook_reel',  label: 'Reel' }
  },
  twitter: {
    post:  { id: 'twitter_post',  label: 'Post' },
    video: { id: 'twitter_video', label: 'Video' }
  },
  tiktok: {} // no sub-types in the sheet for now
};


// Build the platform list from the actual data (+All, +Bonus if present)
function getAvailablePlatformsFromContent() {
  const set = new Set();
  let hasBonus = false;

  (contentData || []).forEach(c => {
    const p = (c.platform || '').toLowerCase();
    if (p) set.add(p);
    if (c.bonus === true) hasBonus = true;
  });

  // desired order when present
  const order = ['tiktok','instagram','youtube','linkedin','facebook','twitter'];
  const list = ['all', ...order.filter(p => set.has(p))];
  if (hasBonus) list.push('bonus');
  return list;
}

function renderPlatformFilters() {
  const wrap = document.getElementById('platformFilters');
  if (!wrap) return;
  wrap.innerHTML = '';

  const platforms = getAvailablePlatformsFromContent();

  platforms.forEach(p => {
    const btn = document.createElement('button');
    btn.className = 'platform-tab' + (currentPlatform === p && !currentSubfilter ? ' active' : '');
    btn.textContent = PLATFORM_LABELS[p] || p;
    btn.addEventListener('click', () => {
      currentPlatform = p;
      currentSubfilter = null;
      currentResultsFilter = p;   // top-level filters by platform (or All/Bonus)

      renderPlatformFilters();
      renderSubFilters(p);
      updateResultsDisplay();
    });
    wrap.appendChild(btn);
  });
}

function renderSubFilters(platform) {
  const container = document.getElementById('subFilters');
  if (!container) return;

  // Hide for All/Bonus or when no map exists
  if (platform === 'all' || platform === 'bonus' || !MEDIA_SUBFILTER_MAP[platform]) {
    container.style.display = 'none';
    container.innerHTML = '';
    return;
  }

  // Determine which media types actually exist for this platform
  const presentTypes = new Set();
  (contentData || []).forEach(c => {
    if ((c.platform || '').toLowerCase() === platform && c.mediaType) {
      presentTypes.add(c.mediaType); // values like 'post','reel','story','short','integration'
    }
  });

  // Build the buttons only for present media types
  const map = MEDIA_SUBFILTER_MAP[platform];
  const availableSubs = Object.keys(map)
    .filter(type => presentTypes.has(type))
    .map(type => map[type]);

  if (!availableSubs.length) {
    container.style.display = 'none';
    container.innerHTML = '';
    return;
  }

  container.style.display = 'flex';
  container.innerHTML = '';

  availableSubs.forEach(sf => {
    const btn = document.createElement('button');
    btn.className = 'subfilter-tab' + (currentSubfilter === sf.id ? ' active' : '');
    btn.textContent = sf.label;
    btn.addEventListener('click', () => {
      currentSubfilter = sf.id;
      currentResultsFilter = sf.id;   // sub-filter id is what updateResultsDisplay() expects
      renderPlatformFilters();
      renderSubFilters(platform);
      updateResultsDisplay();
    });
    container.appendChild(btn);
  });
}


// Sample weekly data structure - replace with actual data loading
const sampleWeeklyData = [
    { 
        metric: 'Likes', 
        week1: 57, week2: 74, week3: 96, week4: 125, week5: 150,
        week6: 175, week7: 200, week8: 225, week9: 250, week10: 275
    },
    { 
        metric: 'Comments', 
        week1: 96, week2: 125, week3: 162, week4: 211, week5: 250,
        week6: 290, week7: 330, week8: 370, week9: 410, week10: 450
    },
    { 
        metric: 'Views', 
        week1: 45336, week2: 58937, week3: 76618, week4: 99603, week5: 120000,
        week6: 145000, week7: 170000, week8: 195000, week9: 220000, week10: 245000
    },
    { 
        metric: 'Shares', 
        week1: 69, week2: 90, week3: 117, week4: 152, week5: 180,
        week6: 210, week7: 240, week8: 270, week9: 300, week10: 330
    },
    { 
        metric: 'Saves', 
        week1: 27, week2: 35, week3: 46, week4: 59, week5: 72,
        week6: 85, week7: 98, week8: 111, week9: 124, week10: 137
    },
    { 
        metric: 'Link Clicks', 
        week1: 987, week2: 1283, week3: 1668, week4: 2168, week5: 2600,
        week6: 3100, week7: 3600, week8: 4100, week9: 4600, week10: 5100
    },
    { 
        metric: 'Reach', 
        week1: 18774, week2: 24406, week3: 31728, week4: 41246, week5: 50000,
        week6: 60000, week7: 70000, week8: 80000, week9: 90000, week10: 100000
    },
    { 
        metric: 'Impressions', 
        week1: 69349, week2: 90154, week3: 117200, week4: 152360, week5: 185000,
        week6: 220000, week7: 255000, week8: 290000, week9: 325000, week10: 360000
    }
];

/**
 * Get platform icon
 */
function getPlatformIcon(platform) {
    const icons = {
        'instagram': '📸',
        'tiktok': '🎵',
        'youtube': '📺'
    };
    return icons[platform.toLowerCase()] || '📱';
}

/**
 * Get available platforms based on current data
 * Returns an array of platform names that have at least 1 post
 */
function getAvailablePlatforms() {
    const platformsWithContent = new Set();
    
    // Check content data for platforms with posts
    contentData.forEach(content => {
        if (content.platform) {
            platformsWithContent.add(content.platform);
        }
    });
    
    // Convert to array and sort alphabetically
    return Array.from(platformsWithContent).sort();
}

/**
 * Get available platforms from influencer data
 * Returns platforms that have at least 1 influencer
 */
function getAvailablePlatformsFromInfluencers() {
    const platformsWithInfluencers = new Set();
    
    influencersData.forEach(influencer => {
        if (influencer.socialMedia) {
            platformsWithInfluencers.add(influencer.socialMedia);
        }
    });
    
    return Array.from(platformsWithInfluencers).sort();
}

/**
 * Update influencers platform filter dropdown
 */
function updateInfluencersPlatformFilter() {
    const platformFilter = document.getElementById('platformFilter');
    if (!platformFilter) return;
    
    const availablePlatforms = getAvailablePlatformsFromInfluencers();
    
    // Clear existing options
    platformFilter.innerHTML = '<option value="">All Platforms</option>';
    
    // Add only platforms that have influencers
    availablePlatforms.forEach(platform => {
        const option = document.createElement('option');
        option.value = platform;
        option.textContent = platform;
        platformFilter.appendChild(option);
    });
}

function toNumberFromCurrency(val) {
  // Strip everything except digits, dot, and minus
  return parseFloat((val ?? '').toString().replace(/[^0-9.\-]/g, '')) || 0;
}

function detectCurrencySymbol(items) {
  for (const it of items) {
    const s = (it?.cost ?? '').toString();
    const m = s.match(/[$£€]/);
    if (m) return m[0];
    const sym = it?.costMetrics?.currencySymbol;
    if (sym) return sym;
  }
  return '$';
}

/**
 * Update content platform filter dropdown
 */
function updateContentPlatformFilter() {
    const platformFilter = document.getElementById('contentPlatformFilter');
    if (!platformFilter) return;
    
    const availablePlatforms = getAvailablePlatforms();
    
    // Store current selection
    const currentSelection = platformFilter.value;
    
    // Clear existing options
    platformFilter.innerHTML = '<option value="">All Platforms</option>';
    
    // Add only platforms that have content
    availablePlatforms.forEach(platform => {
        const option = document.createElement('option');
        option.value = platform;
        option.textContent = platform;
        platformFilter.appendChild(option);
    });
    
    // Always add bonus option if there's any bonus content
    const hasBonusContent = contentData.some(content => content.bonus === true);
    if (hasBonusContent) {
        const bonusOption = document.createElement('option');
        bonusOption.value = 'bonus';
        bonusOption.textContent = 'Bonus Content Only';
        platformFilter.appendChild(bonusOption);
    }
    
    // Restore selection if it still exists
    if (currentSelection && Array.from(platformFilter.options).some(opt => opt.value === currentSelection)) {
        platformFilter.value = currentSelection;
    }
}

// =============================================================================
// WEEKLY DATA FUNCTIONS
// =============================================================================

/**
 * Load weekly data from Google Sheets Weekly tab
 */
async function loadWeeklyData() {
    try {
        console.log('Checking for Weekly tab in sheet:', currentCampaignId);
        
        // First check if the Weekly tab exists
        const weeklyTabExists = await checkSheetTabExists(currentCampaignId, 'Weekly');
        
        if (!weeklyTabExists) {
            console.log('Weekly tab not found - hiding Weekly Performance Trends section');
            hideWeeklySection();
            return;
        }
        
        console.log('Weekly tab found - loading weekly data');
        showWeeklySection(); // Make sure section is visible
        
        // Fetch data from the Weekly sheet - EXTENDED RANGE to K column for 10 weeks
        const data = await fetchSheetData(currentCampaignId, 'Weekly', 'A1:K10');
        
        if (data && data.length > 1) {
            // Process the weekly data with 10 weeks
            weeklyData = processWeeklyData(data);
            window.weeklyData = weeklyData;
        } else {
            // Use sample data as fallback
            console.warn('No weekly data found, using sample data');
            weeklyData = sampleWeeklyData;
        }
        
        // Create charts and update results display
        createCharts();
        updateWeeklyResultsDisplay();
        
    } catch (error) {
        console.error('Error loading weekly data:', error);
        // Hide the weekly section if there's an error
        hideWeeklySection();
    }
}

// Add function to hide the weekly section
function hideWeeklySection() {
    const weeklySection = document.querySelector('.weekly-results-section');
    if (weeklySection) {
        weeklySection.style.display = 'none';
    }
    
    // Also hide any related data in the results stats that come from weekly data
    // These will show 0 or default values instead
    const weeklyRelatedStats = ['resultsTotalViews', 'resultsTotalLikes', 'resultsTotalComments', 'resultsTotalShares', 'resultsTotalSaves'];
    weeklyRelatedStats.forEach(statId => {
        const element = document.getElementById(statId);
        if (element) {
            element.textContent = '0';
        }
    });
}

// Add function to show the weekly section (in case it was hidden)
function showWeeklySection() {
    const weeklySection = document.querySelector('.weekly-results-section');
    if (weeklySection) {
        weeklySection.style.display = 'block';
    }
}


/**
 * Process Weekly sheet data into chart format
 */
function processWeeklyData(data) {
    const processedData = [];
    
    // Skip header row, process each metric row
    for (let i = 1; i < data.length; i++) {
        const row = data[i];
        if (row[0]) { // Check if metric name exists
            const metricData = {
                metric: row[0] // Column A: Metric name
            };
            
            // Dynamically add week data based on what's available
            // Only add weeks that have actual data (not undefined or null)
            for (let weekNum = 1; weekNum <= 10; weekNum++) {
                const value = row[weekNum]; // Columns B-K for weeks 1-10
                if (value !== undefined && value !== null && value !== '') {
                    const stringValue = value.toString().replace(/,/g, '');
                    metricData[`week${weekNum}`] = parseInt(value.toString().replace(/,/g, '')) || 0;
                }
            }
            
            // Only add the metric if it has at least one week of data
            const hasData = Object.keys(metricData).some(key => key.startsWith('week'));
            if (hasData) {
                processedData.push(metricData);
            }
        }
    }
    
    console.log('Processed weekly data:', processedData);
    return processedData;
}


/**
 * Create and render charts
 */
function createCharts() {
    const chartsGrid = document.getElementById('chartsGrid');
    if (!chartsGrid) return;
    
    chartsGrid.innerHTML = '';

    const filteredData = getFilteredMetrics();

    // First, determine the maximum week number that has data across all metrics
    let maxWeekWithData = 0;
    filteredData.forEach(metricData => {
        for (let i = 10; i >= 1; i--) {
            const weekKey = `week${i}`;
            // Check if this week has any non-null data
            if (metricData[weekKey] !== undefined && metricData[weekKey] !== null) {
                maxWeekWithData = Math.max(maxWeekWithData, i);
                break;
            }
        }
    });

    // If no data found, use week 1 as minimum
    if (maxWeekWithData === 0) {
        maxWeekWithData = 1;
    }

    console.log(`Maximum week with data: Week ${maxWeekWithData}`);

    filteredData.forEach(metricData => {
        const chartContainer = createChartContainer(metricData.metric);
        chartsGrid.appendChild(chartContainer);
        
        const canvas = chartContainer.querySelector('canvas');
        const ctx = canvas.getContext('2d');
        
        // Destroy existing chart if it exists
        if (charts[metricData.metric]) {
            charts[metricData.metric].destroy();
        }

        // Build data arrays - only up to the maximum week with data
        const weekLabels = [];
        const weekDataPoints = [];
        
        // Only show weeks up to the maximum week that has data
        for (let i = 1; i <= maxWeekWithData; i++) {
            const weekKey = `week${i}`;
            weekLabels.push(`Week ${i}`);
            // Use 0 for undefined/null values to maintain consistent axis
            weekDataPoints.push(metricData[weekKey] || 0);
        }

        charts[metricData.metric] = new Chart(ctx, {
            type: currentChartType,
            data: {
                labels: weekLabels,
                datasets: [{
                    label: metricData.metric,
                    data: weekDataPoints,
                    backgroundColor: currentChartType === 'bar' 
                        ? metricConfigs[metricData.metric].color + '80'
                        : metricConfigs[metricData.metric].color + '20',
                    borderColor: metricConfigs[metricData.metric].color,
                    borderWidth: 2,
                    fill: currentChartType === 'line',
                    tension: 0.4,
                    pointBackgroundColor: metricConfigs[metricData.metric].color,
                    pointBorderColor: '#fff',
                    pointBorderWidth: 2,
                    pointRadius: 4,
                    pointHoverRadius: 6
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                // Show full number in tooltip with proper formatting
                                const value = context.parsed.y;
                                const formattedValue = value.toLocaleString();
                                return `${metricData.metric}: ${formattedValue}`;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                // Show full numbers without abbreviation for values under 10,000
                                if (value === 0) {
                                    return '0';
                                } else if (value < 10000) {
                                    // For any value under 10,000, show the complete number
                                    return Math.round(value).toLocaleString();
                                } else if (value < 1000000) {
                                    // For 10K to 999K, show as K
                                    return Math.round(value / 1000) + 'K';
                                } else {
                                    // For millions
                                    return (value / 1000000).toFixed(1) + 'M';
                                }
                            },
                            // Ensure we have enough tick marks to show proper values
                            autoSkip: true,
                            maxTicksLimit: 10,
                            precision: 0
                        }
                    },
                    x: {
                        ticks: {
                            // Adjust font size based on number of weeks
                            font: {
                                size: weekLabels.length > 6 ? 10 : 12
                            },
                            // Rotate labels if we have many weeks
                            maxRotation: weekLabels.length > 6 ? 45 : 0,
                            minRotation: weekLabels.length > 6 ? 45 : 0
                        }
                    }
                }
            }
        });
    });
}

/**
 * Create chart container element
 */
function createChartContainer(metric) {
    const container = document.createElement('div');
    container.className = 'chart-container';
    
    const config = metricConfigs[metric];
    container.innerHTML = `
        <div class="chart-title">
            ${config.icon} ${metric}
        </div>
        <div class="chart-canvas">
            <canvas></canvas>
        </div>
    `;
    
    return container;
}

/**
 * Get filtered metrics based on current filter
 */
function getFilteredMetrics() {
    switch (currentMetricFilter) {
        case 'engagement':
            return weeklyData.filter(item => metricConfigs[item.metric] && metricConfigs[item.metric].category === 'engagement');
        case 'reach':
            return weeklyData.filter(item => metricConfigs[item.metric] && metricConfigs[item.metric].category === 'reach');
        case 'actions':
            return weeklyData.filter(item => metricConfigs[item.metric] && metricConfigs[item.metric].category === 'actions');
        default:
            return weeklyData;
    }
}

/**
 * Create charts with a specific week range
 * @param {number} startWeek - Starting week number (1-10)
 * @param {number} endWeek - Ending week number (1-10)
 */
function createChartsWithRange(startWeek, endWeek) {
    const chartsGrid = document.getElementById('chartsGrid');
    if (!chartsGrid) return;
    
    // Clear existing charts
    chartsGrid.innerHTML = '';
    
    // Get filtered metrics based on current filter
    const filteredData = getFilteredMetrics();
    
    // Create a chart for each metric
    filteredData.forEach(metricData => {
        // Create the chart container
        const chartContainer = createChartContainer(metricData.metric);
        chartsGrid.appendChild(chartContainer);
        
        // Get canvas context
        const canvas = chartContainer.querySelector('canvas');
        const ctx = canvas.getContext('2d');
        
        // Destroy existing chart if it exists
        if (charts[metricData.metric]) {
            charts[metricData.metric].destroy();
            delete charts[metricData.metric];
        }
        
        // Build data for selected range only
        const weekLabels = [];
        const weekDataPoints = [];
        
        for (let i = startWeek; i <= endWeek; i++) {
            const weekKey = `week${i}`;
            if (metricData[weekKey] !== undefined) {
                weekLabels.push(`Week ${i}`);
                weekDataPoints.push(metricData[weekKey]);
            }
        }
        
        // Skip creating chart if no data points
        if (weekDataPoints.length === 0) {
            chartContainer.innerHTML = `
                <div class="chart-title">
                    ${metricConfigs[metricData.metric].icon} ${metricData.metric}
                </div>
                <div class="empty-state" style="padding: 2rem; text-align: center; color: #64748b;">
                    No data available for selected week range
                </div>
            `;
            return;
        }
        
        // Create chart with filtered data
        charts[metricData.metric] = new Chart(ctx, {
            type: currentChartType,
            data: {
                labels: weekLabels,
                datasets: [{
                    label: metricData.metric,
                    data: weekDataPoints,
                    backgroundColor: currentChartType === 'bar' 
                        ? metricConfigs[metricData.metric].color + '80'
                        : metricConfigs[metricData.metric].color + '20',
                    borderColor: metricConfigs[metricData.metric].color,
                    borderWidth: 2,
                    fill: currentChartType === 'line',
                    tension: 0.4,
                    pointBackgroundColor: metricConfigs[metricData.metric].color,
                    pointBorderColor: '#fff',
                    pointBorderWidth: 2,
                    pointRadius: 4,
                    pointHoverRadius: 6,
                    pointHoverBackgroundColor: metricConfigs[metricData.metric].color,
                    pointHoverBorderColor: '#fff',
                    pointHoverBorderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                interaction: {
                    mode: 'index',
                    intersect: false
                },
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        backgroundColor: 'rgba(0, 0, 0, 0.8)',
                        titleColor: '#fff',
                        bodyColor: '#fff',
                        padding: 12,
                        cornerRadius: 8,
                        displayColors: false,
                        callbacks: {
                            label: function(context) {
                                // Show full number in tooltip with proper formatting
                                const value = context.parsed.y;
                                const formattedValue = value.toLocaleString();
                                return `${metricData.metric}: ${formattedValue}`;
                            },
                            afterLabel: function(context) {
                                // Calculate and show growth if not the first week
                                if (context.dataIndex > 0) {
                                    const currentValue = context.parsed.y;
                                    const previousValue = context.dataset.data[context.dataIndex - 1];
                                    if (previousValue > 0) {
                                        const growth = ((currentValue - previousValue) / previousValue * 100).toFixed(1);
                                        const arrow = growth >= 0 ? '↑' : '↓';
                                        const color = growth >= 0 ? 'green' : 'red';
                                        return `Growth: ${arrow} ${Math.abs(growth)}%`;
                                    }
                                }
                                return null;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)',
                            drawBorder: false
                        },
                        ticks: {
                            padding: 10,
                            callback: function(value) {
                                // Format y-axis labels - FIXED to show 2,944 correctly
                                if (value >= 1000000) {
                                    // For millions, show as M
                                    return (value / 1000000).toFixed(1) + 'M';
                                } else if (value >= 100000) {
                                    // For 100K+, show as K without decimals
                                    return Math.round(value / 1000) + 'K';
                                } else if (value >= 10000) {
                                    // For 10K-99K, show as K with one decimal
                                    return (value / 1000).toFixed(1) + 'K';
                                } else {
                                    // For anything under 10,000, show the full number with comma separator
                                    // This ensures 2,944 shows as "2,944" not "2"
                                    return value.toLocaleString();
                                }
                            },
                            font: {
                                size: 11
                            },
                            color: '#64748b',
                            // Ensure we have enough tick marks
                            autoSkip: true,
                            maxTicksLimit: 8
                        }
                    },
                    x: {
                        grid: {
                            display: false,
                            drawBorder: false
                        },
                        ticks: {
                            padding: 10,
                            font: {
                                size: weekLabels.length > 6 ? 10 : 12
                            },
                            maxRotation: weekLabels.length > 6 ? 45 : 0,
                            minRotation: weekLabels.length > 6 ? 45 : 0,
                            color: '#64748b'
                        }
                    }
                },
                animation: {
                    duration: 750,
                    easing: 'easeInOutQuart'
                }
            }
        });
    });
    
    // Update summary stats for the selected range
    updateRangeSummaryStats(startWeek, endWeek);
}

/**
 * Update summary statistics for the selected week range
 * @param {number} startWeek - Starting week number
 * @param {number} endWeek - Ending week number
 */
function updateRangeSummaryStats(startWeek, endWeek) {
    // Calculate totals for the selected range
    const rangeTotals = {};
    let hasData = false;
    
    weeklyData.forEach(metricData => {
        let total = 0;
        let count = 0;
        
        for (let i = startWeek; i <= endWeek; i++) {
            const weekKey = `week${i}`;
            if (metricData[weekKey] !== undefined && metricData[weekKey] > 0) {
                total += metricData[weekKey];
                count++;
                hasData = true;
            }
        }
        
        rangeTotals[metricData.metric] = {
            total: total,
            average: count > 0 ? Math.round(total / count) : 0,
            weeks: count
        };
    });
    
    // Update display if there's a summary section
    const summarySection = document.querySelector('.weekly-range-summary');
    if (summarySection && hasData) {
        summarySection.innerHTML = `
            <div class="range-summary-title">
                Week ${startWeek} - Week ${endWeek} Summary
            </div>
            <div class="range-summary-stats">
                ${Object.entries(rangeTotals).map(([metric, data]) => {
                    if (data.total > 0) {
                        return `
                            <div class="range-stat">
                                <div class="range-stat-label">${metric}</div>
                                <div class="range-stat-value">
                                    ${data.total.toLocaleString()}
                                </div>
                                <div class="range-stat-avg">
                                    Avg: ${data.average.toLocaleString()}/week
                                </div>
                            </div>
                        `;
                    }
                    return '';
                }).join('')}
            </div>
        `;
        summarySection.style.display = 'block';
    } else if (summarySection) {
        summarySection.style.display = 'none';
    }
}

/**
 * Update week range and refresh charts
 */
function updateWeekRange() {
    const startWeek = parseInt(document.getElementById('weekRangeStart').value);
    const endWeek = parseInt(document.getElementById('weekRangeEnd').value);
    
    // Validate inputs
    if (isNaN(startWeek) || isNaN(endWeek)) {
        alert('Please select valid week numbers');
        return;
    }
    
    if (startWeek < 1 || startWeek > 10 || endWeek < 1 || endWeek > 10) {
        alert('Week numbers must be between 1 and 10');
        return;
    }
    
    if (startWeek > endWeek) {
        alert('Start week must be before or equal to end week');
        return;
    }
    
    // Show loading state
    const chartsGrid = document.getElementById('chartsGrid');
    if (chartsGrid) {
        chartsGrid.innerHTML = '<div class="loading-message">Updating charts...</div>';
    }
    
    // Update charts with new range
    setTimeout(() => {
        createChartsWithRange(startWeek, endWeek);
    }, 100);
}

/**
 * Update chart type
 */
function updateChartType() {
    const chartTypeSelect = document.getElementById('chartType');
    if (chartTypeSelect) {
        currentChartType = chartTypeSelect.value;
        createCharts();
    }
}

/**
 * Update visible metrics based on filter
 */
function updateVisibleMetrics() {
    const metricSelector = document.getElementById('metricSelector');
    if (metricSelector) {
        currentMetricFilter = metricSelector.value;
        createCharts();
    }
}

/**
 * Refresh weekly data
 */
function refreshWeeklyData() {
    loadWeeklyData();
}

/**
 * Update results display with weekly totals
 */
function updateWeeklyResultsDisplay() {
    // This function now only updates the weekly charts
    // It doesn't override the results display stats anymore
    console.log('Weekly charts updated with weekly data');
}

// =============================================================================
// CONTENT FUNCTIONS
// =============================================================================

function sortContentGrid() {
    const dropdown = document.getElementById('contentSortDropdown');
    if (!dropdown) return;
    
    const sortValue = dropdown.value;
    const [sortBy, direction] = sortValue.split('-');
    
    currentContentSortBy = sortBy;
    currentContentSortDirection = direction || 'desc';
    
    // Get current filtered content
    const grid = document.getElementById('contentGrid');
    if (!grid) return;
    
    const cards = Array.from(grid.querySelectorAll('.content-card'));
    
    // Filter visible cards only
    const visibleCards = cards.filter(card => card.style.display !== 'none');
    
    if (sortBy === 'default') {
        // Restore original order by re-rendering
        updateContentGrid();
        // Re-apply current filters
        filterContent();
        return;
    }
    
    // Sort visible cards
    visibleCards.sort((cardA, cardB) => {
        const contentA = getContentDataFromCard(cardA);
        const contentB = getContentDataFromCard(cardB);
        
        if (!contentA || !contentB) return 0;
        
        let valueA, valueB;
        
        switch (sortBy) {
            case 'views':
                valueA = contentA.metrics.views || 0;
                valueB = contentB.metrics.views || 0;
                break;
            case 'likes':
                valueA = contentA.metrics.likes || 0;
                valueB = contentB.metrics.likes || 0;
                break;
            case 'engagement':
                valueA = getContentER(contentA);
                valueB = getContentER(contentB);
                break;
            case 'date':
                valueA = new Date(contentA.postedDate || '1970-01-01');
                valueB = new Date(contentB.postedDate || '1970-01-01');
                break;
            default:
                return 0;
        }
        
        // Sort based on direction
        if (direction === 'asc') {
            return valueA < valueB ? -1 : valueA > valueB ? 1 : 0;
        } else {
            return valueB < valueA ? -1 : valueB > valueA ? 1 : 0;
        }
    });
    
    // Clear grid and re-append sorted cards
    grid.innerHTML = '';
    
    // First add the sort dropdown
    const sortContainer = document.createElement('div');
    sortContainer.innerHTML = getContentSortDropdownHTML();
    sortContainer.style.gridColumn = '1 / -1';
    grid.appendChild(sortContainer);
    
    // Add sorted visible cards first
    visibleCards.forEach(card => {
        grid.appendChild(card);
    });
    
    // Add back filtered out cards at the end (hidden)
    cards.forEach(card => {
        if (card.style.display === 'none') {
            grid.appendChild(card);
        }
    });
    
    // Restore dropdown value
    const newDropdown = document.getElementById('contentSortDropdown');
    if (newDropdown) {
        newDropdown.value = sortValue;
    }
}

/**
 * Get content data from a card element
 */
function getContentDataFromCard(card) {
    const contentId = card.dataset.contentId;
    const title = card.dataset.title;
    
    // Use ID for lookup if available, fallback to title
    return contentId 
        ? contentData.find(content => content.id == contentId)
        : contentData.find(content => content.title.toLowerCase() === title.toLowerCase());
}

/**
 * Get the ER (Engagement Rate) for a content item using existing calculation
 */
function getContentER(content) {
    // Exclude Instagram Posts AND Stories
    const isInstagramPostsOrStories = content.platform.toLowerCase() === 'instagram' && 
                                      (content.mediaType === 'post' || content.mediaType === 'story');
    
    // Return 0 for Instagram Posts and Stories
    if (isInstagramPostsOrStories) {
        return 0;
    }
    
    // Calculate ER for all other content including Instagram Stories
    if (content.metrics.views > 0) {
        const totalEngagement = (content.metrics.likes || 0) + 
                               (content.metrics.comments || 0) + 
                               (content.metrics.shares || 0) + 
                               (content.metrics.saves || 0);
        return (totalEngagement / content.metrics.views) * 100;
    }
    
    return 0;
}

/**
 * Get the HTML for the sort dropdown
 */
function getContentSortDropdownHTML() {
    return `
        <div class="content-sort-container">
            <label for="contentSortDropdown">Sort by:</label>
            <select id="contentSortDropdown" onchange="sortContentGrid()">
                <option value="default">Default Order</option>
                <option value="views-desc">Most Views</option>
                <option value="views-asc">Least Views</option>
                <option value="likes-desc">Most Likes</option>
                <option value="likes-asc">Least Likes</option>
                <option value="engagement-desc">Highest ER</option>
                <option value="engagement-asc">Lowest ER</option>
                <option value="date-desc">Newest First</option>
                <option value="date-asc">Oldest First</option>
            </select>
        </div>
    `;
}

function updateAnalysisSection(sectionType, titles, contents) {
    const sectionClass = sectionType === 'findings' ? 'key-findings' : 'recommendations';
    const cardPrefix = sectionType === 'findings' ? 'finding' : 'recommendation';
    
    // Find the cards grid container
    const sectionElement = document.querySelector(`.${sectionClass}`);
    if (!sectionElement) return;
    
    const cardsGrid = sectionElement.querySelector('.analysis-cards-grid');
    if (!cardsGrid) return;
    
    // Clear existing cards
    cardsGrid.innerHTML = '';
    
    // Check if any content exists for this section
    const hasAnyContent = titles.some((title, index) => 
        (title && title.trim() !== '') || (contents[index] && contents[index].trim() !== '')
    );
    
    if (!hasAnyContent) {
        // Hide the entire section if no content
        sectionElement.style.display = 'none';
        return;
    }
    
    // Show the section
    sectionElement.style.display = 'block';
    
    // Create cards only for non-empty content
    titles.forEach((title, index) => {
        const content = contents[index] || '';
        
        // Only create card if either title or content has data
        if ((title && title.trim() !== '') || (content && content.trim() !== '')) {
            const card = createAnalysisCard(cardPrefix, index + 1, title, content);
            cardsGrid.appendChild(card);
        }
    });
}

// Update hideAnalysisTab to use inline styles
function hideAnalysisTab() {
    // Hide the analysis tab content with inline style (higher priority than CSS)
    const analysisTab = document.getElementById('analysis');
    if (analysisTab) {
        analysisTab.style.display = 'none';
        analysisTab.classList.remove('active');
    }
    
    // Hide the analysis nav item
    const analysisNavItem = document.querySelector('.nav-item[onclick="showTab(\'analysis\')"]');
    if (analysisNavItem) {
        analysisNavItem.style.display = 'none';
    }
    
    console.log('Analysis tab hidden due to no content');
}

// Add this function to check if a sheet tab exists
async function checkSheetTabExists(sheetId, tabName) {
    try {
        // Try to fetch just one cell from the sheet
        // If the sheet doesn't exist, this will throw an error
        const data = await fetchSheetData(sheetId, tabName, 'A1:A1');
        return true; // Sheet exists
    } catch (error) {
        console.log(`Sheet tab "${tabName}" not found:`, error.message);
        return false; // Sheet doesn't exist
    }
}

/**
 * Show the analysis tab (in case it was previously hidden)
 */
function showAnalysisTab() {
    // Show the analysis tab content
    const analysisTab = document.getElementById('analysis');
    if (analysisTab) {
        // Remove display none that was set inline
        analysisTab.style.display = '';
        // Let CSS handle the display through the active class
        // This will be handled by showTab() when clicked
    }
    
    // Show the analysis nav item
    const analysisNavItem = document.querySelector('.nav-item[onclick="showTab(\'analysis\')"]');
    if (analysisNavItem) {
        analysisNavItem.style.display = '';
    }
    
    console.log('Analysis tab shown with content');
}

/**
 * Format numbers with commas for paid media display
 */
function formatPaidMediaValue(value) {
    // If already formatted or is a string with % or $, return as-is
    if (typeof value === 'string') {
        // Check if it contains % or $ symbols
        if (value.includes('%') || value.includes('$') || value.includes('€') || value.includes('£')) {
            // Extract the number, format it, then add the symbol back
            const hasPercent = value.includes('%');
            const currencyMatch = value.match(/[$€£]/);
            const currency = currencyMatch ? currencyMatch[0] : '';
            
            // Extract numeric value
            const numericValue = parseFloat(value.replace(/[^0-9.\-]/g, ''));
            
            if (isNaN(numericValue)) return value;
            
            // Determine decimal places based on value size
            let minDecimals = 2;
            let maxDecimals = 2;
            
            // For very small numbers (less than 0.01), show up to 4 decimal places
            if (numericValue > 0 && numericValue < 0.01) {
                minDecimals = 3;
                maxDecimals = 4;
            }
            
            // Format with commas
            const formatted = numericValue.toLocaleString('en-US', {
                minimumFractionDigits: numericValue % 1 !== 0 ? minDecimals : 0,
                maximumFractionDigits: maxDecimals
            });
            
            // Return with appropriate symbol
            if (hasPercent) return formatted + '%';
            if (currency) return currency + formatted;
            return formatted;
        }
        
        // Try to parse as number
        const num = parseFloat(value.replace(/,/g, ''));
        if (isNaN(num)) return value;
        value = num;
    }
    
    // Format numeric values with commas
    if (typeof value === 'number') {
        // Determine decimal places based on value size
        let minDecimals = 2;
        let maxDecimals = 2;
        
        // For very small numbers (less than 0.01), show up to 4 decimal places
        if (value > 0 && value < 0.01) {
            minDecimals = 3;
            maxDecimals = 4;
        }
        
        return value.toLocaleString('en-US', {
            minimumFractionDigits: value % 1 !== 0 ? minDecimals : 0,
            maximumFractionDigits: maxDecimals
        });
    }
    
    return value;
}

// =============================================================================
// CAMPAIGN INFO FUNCTIONS
// =============================================================================

/**
 * Load client and campaign info from Info sheet
 */
async function loadCampaignInfo() {
    try {
        console.log('Loading campaign info from sheet:', currentCampaignId);
        
        // Fetch data from Info sheet - expanded range to include columns U and V
        const data = await fetchSheetData(currentCampaignId, 'Info', 'A1:V3');
        
        console.log('Campaign info data received:', data);
        
        if (data && data.length > 1) {
            const clientName = data[1][0] || 'Client'; // Column A, Row 2
            const campaignName = data[1][1] || 'Campaign'; // Column B, Row 2
            const keyObjectives = data[1][2] || 'Campaign objectives will be displayed here.'; // Column C, Row 2
            const creativeConcept = data[1][3] || 'Creative concept details will be displayed here.'; // Column D, Row 2
            paidMediaLookerUrl = (data[1][18] || '').toString().trim();
            
            updateLookerButtonVisibility();
            
            console.log('Extracted campaign data:', {
                clientName,
                campaignName,
                keyObjectives,
                creativeConcept
            });
            
            // Update existing elements
            const clientElement = document.getElementById('clientName');
            const campaignElement = document.getElementById('campaignName');
            const objectivesElement = document.getElementById('keyObjectives');
            const conceptElement = document.getElementById('creativeConcept');
            
            if (clientElement) clientElement.textContent = clientName;
            if (campaignElement) campaignElement.textContent = campaignName;
            if (objectivesElement) objectivesElement.textContent = keyObjectives;
            if (conceptElement) conceptElement.textContent = creativeConcept;
            
            // Extract Paid Media Findings and Recommendations from columns U and V
            // These are in row 3 (index 2)
            if (data.length > 2) {
                paidMediaData.findings = data[1][16] || ''; // Column Q, Row 3 (index 16)
                paidMediaData.recommendations = data[1][17] || ''; // Column R, Row 3 (index 17)
                
                console.log('Extracted Paid Media analysis:', {
                    findings: paidMediaData.findings,
                    recommendations: paidMediaData.recommendations
                });
                
                // Update Paid Media analysis cards if they exist
                updatePaidMediaAnalysisCards();
            }
        }
        
        // Load analysis data separately with conditional display logic
        await loadAnalysisData();
        
    } catch (error) {
        console.error('Error loading campaign info:', error);
        
        // Set defaults if info can't be loaded
        const clientElement = document.getElementById('clientName');
        const campaignElement = document.getElementById('campaignName');
        const objectivesElement = document.getElementById('keyObjectives');
        const conceptElement = document.getElementById('creativeConcept');
        
        if (clientElement) clientElement.textContent = 'Client';
        if (campaignElement) campaignElement.textContent = 'Campaign';
        if (objectivesElement) objectivesElement.textContent = 'Unable to load campaign objectives. Please check your Google Sheet configuration.';
        if (conceptElement) conceptElement.textContent = 'Unable to load creative concept. Please check your Google Sheet configuration.';
        
        // Hide analysis tab if campaign info fails to load
        hideAnalysisTab();
    }
}

/**
 * Show/hide Looker button based on URL availability
 */
function updateLookerButtonVisibility() {
    const buttonContainer = document.getElementById('lookerButtonContainer');
    if (!buttonContainer) return;
    
    if (paidMediaLookerUrl && isValidLookerUrl(paidMediaLookerUrl)) {
        buttonContainer.style.display = 'flex';
        console.log('Looker button shown, URL:', paidMediaLookerUrl);
    } else {
        buttonContainer.style.display = 'none';
        console.log('Looker button hidden - no valid URL');
    }
}

/**
 * Validate Looker Studio URL
 */
function isValidLookerUrl(url) {
    if (!url) return false;
    try {
        const urlObj = new URL(url);
        return urlObj.host === 'lookerstudio.google.com' || 
               urlObj.host === 'datastudio.google.com';
    } catch {
        return false;
    }
}

/**
 * Open Looker Studio popup
 */
function openLookerPopup() {
    if (!paidMediaLookerUrl || !isValidLookerUrl(paidMediaLookerUrl)) {
        alert('Looker Studio dashboard URL is not available.');
        return;
    }
    
    const popup = document.getElementById('lookerPopup');
    const container = document.getElementById('lookerIframeContainer');
    const loading = document.getElementById('lookerLoading');
    
    if (!popup || !container) return;
    
    // Show popup with loading state
    popup.classList.add('active');
    document.body.style.overflow = 'hidden';
    if (loading) loading.style.display = 'flex';
    container.innerHTML = '';
    
    // Load iframe after brief delay
    setTimeout(() => {
        container.innerHTML = `
            <iframe 
                src="${escapeHtml(paidMediaLookerUrl)}"
                frameborder="0"
                allowfullscreen
                sandbox="allow-scripts allow-same-origin allow-popups allow-forms"
                onload="document.getElementById('lookerLoading').style.display='none'">
            </iframe>
        `;
    }, 100);
}

/**
 * Close Looker Studio popup
 */
function closeLookerPopup() {
    const popup = document.getElementById('lookerPopup');
    const container = document.getElementById('lookerIframeContainer');
    
    if (popup) popup.classList.remove('active');
    if (container) container.innerHTML = '';
    document.body.style.overflow = '';
}

// Close Looker popup with Escape key
document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape') closeLookerPopup();
});


// =============================================================================
// COST CALCULATION FUNCTIONS
// =============================================================================

/**
 * Calculate CPE by content type using sum of costs / sum of engagement
 */
function calculateCPEByContentType(content, filter) {
    let totalCost = 0;
    let totalEngagement = 0;
    
    content.forEach(item => {
        const cost = toNumberFromCurrency(item.cost);
        const engagement = (item.metrics.likes || 0) + 
                          (item.metrics.comments || 0) + 
                          (item.metrics.shares || 0) + 
                          (item.metrics.saves || 0) + 
                          (item.analytics.stickerTaps || 0);
        
        // Include items with cost > 0 OR bonus items (even with 0 cost)
        if ((cost > 0 || item.bonus === true) && engagement > 0) {
            const platform = item.platform.toLowerCase();
            const mediaType = item.mediaType || '';
            
            let includeInCalculation = false;
            
            switch (filter) {
                case 'all':
                    includeInCalculation = true;
                    break;
                case 'tiktok':
                    includeInCalculation = platform === 'tiktok';
                    break;
                case 'instagram':
                    includeInCalculation = platform === 'instagram';
                    break;
                case 'youtube':
                    includeInCalculation = platform === 'youtube';
                    break;
                case 'youtube_shorts':
                    includeInCalculation = platform === 'youtube' && mediaType === 'short';
                    break;
                case 'youtube_integrations':
                    includeInCalculation = platform === 'youtube' && mediaType === 'integration';
                    break;
                case 'instagram_stories':
                    includeInCalculation = platform === 'instagram' && mediaType === 'story';
                    break;
                case 'instagram_reel':
                    includeInCalculation = platform === 'instagram' && mediaType === 'reel';
                    break;
                case 'instagram_posts':
                    includeInCalculation = platform === 'instagram' && mediaType === 'post';
                    break;
                case 'bonus':
                    includeInCalculation = item.bonus === true;
                    break;
                case 'linkedin_post':
                    includeInCalculation = platform === 'linkedin' && mediaType === 'post';
                    break;
                case 'linkedin_video':
                    includeInCalculation = platform === 'linkedin' && mediaType === 'video';
                    break;
                case 'facebook_post':
                    includeInCalculation = platform === 'facebook' && mediaType === 'post';
                    break;
                case 'facebook_reel':
                    includeInCalculation = platform === 'facebook' && mediaType === 'reel';
                    break;
                case 'twitter_post':
                    includeInCalculation = platform === 'twitter' && mediaType === 'post';
                    break;
                case 'twitter_video':
                    includeInCalculation = platform === 'twitter' && mediaType === 'video';
                    break;
                case 'linkedin':
                    includeInCalculation = platform === 'linkedin';
                    break;
                case 'facebook':
                    includeInCalculation = platform === 'facebook';
                    break;
                case 'twitter':
                    includeInCalculation = platform === 'twitter';
                    break;
            }
            
            if (includeInCalculation) {
                totalCost += cost;
                totalEngagement += engagement;
            }
        }
    });
    
    // CPE = total cost / total engagement
    return totalEngagement > 0 ? totalCost / totalEngagement : 0;
}

/**
 * Calculate CPV by content type using sum of costs / sum of views
 */
function calculateCPVByContentType(content, filter) {
    let totalCost = 0;
    let totalViews = 0;
    
    content.forEach(item => {
        const cost = toNumberFromCurrency(item.cost);
        const views = item.metrics.views || 0;
        
        // Include items with cost > 0 OR bonus items (even with 0 cost)
        if ((cost > 0 || item.bonus === true) && views > 0) {
            const platform = item.platform.toLowerCase();
            const mediaType = item.mediaType || '';
            
            let includeInCalculation = false;
            
            switch (filter) {
                case 'all':
                    includeInCalculation = true;
                    break;
                case 'tiktok':
                    includeInCalculation = platform === 'tiktok';
                    break;
                case 'instagram':
                    includeInCalculation = platform === 'instagram';
                    break;
                case 'youtube':
                    includeInCalculation = platform === 'youtube';
                    break;
                case 'youtube_shorts':
                    includeInCalculation = platform === 'youtube' && mediaType === 'short';
                    break;
                case 'youtube_integrations':
                    includeInCalculation = platform === 'youtube' && mediaType === 'integration';
                    break;
                case 'instagram_stories':
                    includeInCalculation = platform === 'instagram' && mediaType === 'story';
                    break;
                case 'instagram_reel':
                    includeInCalculation = platform === 'instagram' && mediaType === 'reel';
                    break;
                case 'instagram_posts':
                    includeInCalculation = platform === 'instagram' && mediaType === 'post';
                    break;
                case 'bonus':
                    includeInCalculation = item.bonus === true;
                    break;
                case 'linkedin_post':
                    includeInCalculation = platform === 'linkedin' && mediaType === 'post';
                    break;
                case 'linkedin_video':
                    includeInCalculation = platform === 'linkedin' && mediaType === 'video';
                    break;
                case 'facebook_post':
                    includeInCalculation = platform === 'facebook' && mediaType === 'post';
                    break;
                case 'facebook_reel':
                    includeInCalculation = platform === 'facebook' && mediaType === 'reel';
                    break;
                case 'twitter_post':
                    includeInCalculation = platform === 'twitter' && mediaType === 'post';
                    break;
                case 'twitter_video':
                    includeInCalculation = platform === 'twitter' && mediaType === 'video';
                    break;
                case 'linkedin':
                    includeInCalculation = platform === 'linkedin';
                    break;
                case 'facebook':
                    includeInCalculation = platform === 'facebook';
                    break;
                case 'twitter':
                    includeInCalculation = platform === 'twitter';
                    break;
            }
            
            if (includeInCalculation) {
                totalCost += cost;
                totalViews += views;
            }
        }
    });
    
    // CPV = total cost / total views
    return totalViews > 0 ? totalCost / totalViews : 0;
}

// =============================================================================
// MAIN FUNCTIONS
// =============================================================================

/**
 * Filter results by platform
 */
function filterResults(platform) {
    currentResultsFilter = platform;
    
    // Update active tab
    document.querySelectorAll('.platform-tab').forEach(tab => {
        tab.classList.remove('active');
    });
    event.target.classList.add('active');
    
    // Update results display
    updateResultsDisplay();
}

/**
 * Update secondary filter options based on primary platform selection
 */
function updateSecondaryFilter() {
    const primaryFilter = document.getElementById('contentPlatformFilter').value;
    const secondaryFilter = document.getElementById('contentMediaTypeFilter');
    
    // Clear existing options
    secondaryFilter.innerHTML = '';
    
    if (!primaryFilter || primaryFilter === '' || primaryFilter === 'bonus') {
        secondaryFilter.style.display = 'none';
        return;
    }
    
    // Show secondary filter
    secondaryFilter.style.display = 'block';
    
    // Add default "All Types" option
    const allOption = document.createElement('option');
    allOption.value = '';
    allOption.textContent = `All ${primaryFilter} Types`;
    secondaryFilter.appendChild(allOption);
    
    // Add platform-specific options
    const platformOptions = getMediaTypeOptions(primaryFilter);
    platformOptions.forEach(option => {
        const optionElement = document.createElement('option');
        optionElement.value = option.value;
        optionElement.textContent = option.label;
        secondaryFilter.appendChild(optionElement);
    });
}

/**
 * Override the existing getMediaTypeOptions function to only show available options
 */
function getMediaTypeOptions(platform) {
    const platformLower = platform.toLowerCase();
    const availableTypes = new Set();
    
    // Check what media types actually exist for this platform
    contentData.forEach(content => {
        if (content.platform.toLowerCase() === platformLower && content.mediaType) {
            availableTypes.add(content.mediaType);
        }
    });
    
    // Define all possible options for each platform
    const allOptions = {
        'instagram': [
            { value: 'post', label: 'Post' },
            { value: 'reel', label: 'Reel' },
            { value: 'story', label: 'Stories' }
        ],
        'tiktok': [],
        'youtube': [
            { value: 'integration', label: 'Integrations' },
            { value: 'short', label: 'Shorts' }
        ],
        'linkedin': [
            { value: 'post', label: 'Post' },
            { value: 'video', label: 'Video' }
        ],
        'facebook': [
            { value: 'post', label: 'Post' },
            { value: 'reel', label: 'Reel' }
        ],
        'twitter': [
            { value: 'post', label: 'Post' },
            { value: 'video', label: 'Video' }
        ]
    };
    
    const platformOptions = allOptions[platformLower] || [];
    
    // Filter to only include types that actually exist in the data
    return platformOptions.filter(option => availableTypes.has(option.value));
}

/**
 * Update results display based on current filter
 */
function updateResultsDisplay() {
  // Get date range values
  const startDate = document.getElementById('resultsStartDate')?.value;
  const endDate = document.getElementById('resultsEndDate')?.value;
  
  let filteredContent = contentData;
  const currencySymbol = detectCurrencySymbol(filteredContent);

  // Filter by date range ONLY if the flag is set (user manually changed dates)
  if (useDateFilterForResults && startDate && endDate) {
    filteredContent = filteredContent.filter(content => {
      return isDateInRange(content.postedDate, startDate, endDate);
    });
  }

  // Filter by platform/type if not 'all'
  if (currentResultsFilter !== 'all') {
    filteredContent = filteredContent.filter(content => {
      const platform = (content.platform || '').toLowerCase();
      const mediaType = content.mediaType || '';

      if (currentResultsFilter === 'bonus') return content.bonus === true;

      switch (currentResultsFilter) {
        case 'tiktok':               return platform === 'tiktok';
        case 'instagram':            return platform === 'instagram';
        case 'youtube':              return platform === 'youtube';
        case 'linkedin':             return platform === 'linkedin';
        case 'facebook':             return platform === 'facebook';
        case 'twitter':              return platform === 'twitter';

        // Instagram sub-filters
        case 'instagram_reel':       return platform === 'instagram' && mediaType === 'reel';
        case 'instagram_stories':    return platform === 'instagram' && mediaType === 'story';
        case 'instagram_posts':      return platform === 'instagram' && mediaType === 'post';

        // YouTube sub-filters
        case 'youtube_integrations': return platform === 'youtube' && mediaType === 'integration';
        case 'youtube_shorts':       return platform === 'youtube' && mediaType === 'short';

        // LinkedIn / Facebook / Twitter sub-filters
        case 'linkedin_post':        return platform === 'linkedin' && mediaType === 'post';
        case 'linkedin_video':       return platform === 'linkedin' && mediaType === 'video';
        case 'facebook_post':        return platform === 'facebook' && mediaType === 'post';
        case 'facebook_reel':        return platform === 'facebook' && mediaType === 'reel';
        case 'twitter_post':         return platform === 'twitter'  && mediaType === 'post';
        case 'twitter_video':        return platform === 'twitter'  && mediaType === 'video';
        default:                     return true;
      }
    });
  }

  // Aggregate totals
  const totals = filteredContent.reduce((acc, c) => {
    const m = c.metrics || {};
    acc.views       += m.views    || 0;
    acc.likes       += m.likes    || 0;
    acc.comments    += m.comments || 0;
    acc.shares      += m.shares   || 0;
    acc.saves       += m.saves    || 0;
    acc.linkClicks  += (c.analytics?.linkClicks   || 0);
    acc.stickerTaps += (c.analytics?.stickerTaps  || 0);

    const isIGStory = (c.platform || '').toLowerCase() === 'instagram' && c.mediaType === 'story';
    acc.content += isIGStory ? (c.analytics?.stories || 1) : 1;

    return acc;
  }, { views:0, likes:0, comments:0, shares:0, saves:0, content:0, linkClicks:0, stickerTaps:0 });

// Calculate ER: Aggregate method (Total Engagement / Total Views * 100)
// Exclude Instagram Posts AND Instagram Stories from ER calculation
    let totalEngagementForER = 0;
    let totalViewsForER = 0;
    
    filteredContent.forEach(content => {
        const isInstagramPostsOrStories = content.platform.toLowerCase() === 'instagram' && 
                                          (content.mediaType === 'post' || content.mediaType === 'story');
        
        if (!isInstagramPostsOrStories) {
            const views = content.metrics.views || 0;
            
            if (views > 0) {
                const engagement = (content.metrics.likes || 0) + 
                                 (content.metrics.comments || 0) + 
                                 (content.metrics.shares || 0) + 
                                 (content.metrics.saves || 0);
                
                totalEngagementForER += engagement;
                totalViewsForER += views;
            }
        }
    });
    
    const calculatedER = totalViewsForER > 0 ? (totalEngagementForER / totalViewsForER) * 100 : 0;

  // Helper to set text
  const setText = (id, val) => {
    const el = document.getElementById(id);
    if (el != null) el.textContent = formatNumber(val);
  };

  // Helper to show/hide card based on value
  const toggleCard = (id, value) => {
    const el = document.getElementById(id);
    const box = el?.closest('.result-stat-box');
    if (box) {
      box.style.display = value > 0 ? 'block' : 'none';
    }
    if (el && value > 0) {
      el.textContent = formatNumber(value);
    }
  };

  // Update all cards and hide if zero
  toggleCard('resultsTotalViews', totals.views);
  toggleCard('resultsTotalLikes', totals.likes);
  toggleCard('resultsTotalComments', totals.comments);
  toggleCard('resultsTotalShares', totals.shares);
  toggleCard('resultsTotalContent', totals.content);
  toggleCard('resultsTotalClicks', totals.linkClicks);

  // Saves vs Sticker Taps for IG Stories
  const isIGStories = currentResultsFilter === 'instagram_stories';
  const savesValueForCard = isIGStories ? totals.stickerTaps : totals.saves;

  const savesValueEl = document.getElementById('resultsTotalSaves');
  const savesBox     = savesValueEl?.closest('.result-stat-box');
  const savesLabelEl = savesBox?.querySelector('.result-stat-label');

  if (savesLabelEl) savesLabelEl.textContent = isIGStories ? 'Sticker Taps' : 'Saves';
  if (savesValueEl) savesValueEl.textContent = formatNumber(savesValueForCard);

  // Hide Saves/Shares for YouTube types or when zero
  const isYouTubeFilter = ['youtube','youtube_integrations','youtube_shorts'].includes(currentResultsFilter);
  if (savesBox)  savesBox.style.display  = (savesValueForCard === 0 || isYouTubeFilter) ? 'none' : 'block';

  // ER Card - hide for Instagram Posts and Instagram Stories
    const erBox = document.getElementById('resultsTotalERBox');
    const erValueEl = document.getElementById('resultsTotalER');
    
    const isInstagramPostsFilter = currentResultsFilter === 'instagram_posts';
    const isInstagramStoriesFilter = currentResultsFilter === 'instagram_stories';
    
    if (erBox && erValueEl) {
        // Hide ER for Instagram Posts or Stories sub-filters
        if (isInstagramPostsFilter || isInstagramStoriesFilter) {
            erBox.style.display = 'none';
        } 
        // For "All" view, check if campaign has ANY content with calculable ER
        else if (currentResultsFilter === 'all') {
            // Check if there's ANY content that's NOT Instagram Posts or Stories
            const hasContentWithER = filteredContent.some(content => {
                const isInstagram = content.platform.toLowerCase() === 'instagram';
                const isPostOrStory = content.mediaType === 'post' || content.mediaType === 'story';
                // Include if it's not Instagram, OR if it's Instagram but not a Post/Story
                return (!isInstagram || !isPostOrStory) && content.metrics.views > 0;
            });
            
            // Hide ER if no content with calculable ER exists OR if ER is 0
            if (!hasContentWithER || calculatedER === 0) {
                erBox.style.display = 'none';
            } else {
                erBox.style.display = 'block';
                erValueEl.textContent = `${calculatedER.toFixed(2)}%`;
                displayERBenchmark(calculatedER, currentResultsFilter);
            }
        }
        // For overall Instagram view, check if there's any content with calculable ER
        else if (currentResultsFilter === 'instagram') {
            // Check if there's any Instagram content that's NOT Posts or Stories (e.g., Reels)
            const hasInstagramContentWithER = filteredContent.some(content => {
                const isInstagram = content.platform.toLowerCase() === 'instagram';
                const isPostOrStory = content.mediaType === 'post' || content.mediaType === 'story';
                return isInstagram && !isPostOrStory && content.metrics.views > 0;
            });
            
            // Hide ER if no Instagram content with calculable ER exists OR if ER is 0
            if (!hasInstagramContentWithER || calculatedER === 0) {
                erBox.style.display = 'none';
            } else {
                erBox.style.display = 'block';
                erValueEl.textContent = `${calculatedER.toFixed(2)}%`;
                displayERBenchmark(calculatedER, currentResultsFilter);
            }
        }
        else {
            erBox.style.display = 'block';
            erValueEl.textContent = calculatedER > 0 ? `${calculatedER.toFixed(2)}%` : '0%';
            
            // Display benchmark comparison - pass the current filter
            displayERBenchmark(calculatedER, currentResultsFilter);
        }
    }

  // Cost cards
  const cpeBox = document.getElementById('resultsTotalCPE')?.closest('.result-stat-box');
  const cpvBox = document.getElementById('resultsTotalCPV')?.closest('.result-stat-box');

  const avgCPE = calculateCPEByContentType(filteredContent, currentResultsFilter);
  const avgCPV = calculateCPVByContentType(filteredContent, currentResultsFilter);

  // Helper function to format cost with appropriate decimal places
  const formatCostMetric = (value) => {
    if (value < 0.01 && value > 0) {
      if (value < 0.001) return value.toFixed(4);
      if (value < 0.01) return value.toFixed(3);
    }
    return value.toFixed(2);
  };

  // CPE - hide if zero
  if (cpeBox) {
    const cpeEl = document.getElementById('resultsTotalCPE');
    if (avgCPE > 0) { 
      if (cpeEl) cpeEl.textContent = `${currencySymbol}${formatCostMetric(avgCPE)}`; 
      cpeBox.style.display = 'block'; 
    } else {
      cpeBox.style.display = 'none';
    }
  }

  // CPV - hide if zero
  if (cpvBox) {
    const cpvEl = document.getElementById('resultsTotalCPV');
    if (avgCPV > 0) { 
      if (cpvEl) cpvEl.textContent = `${currencySymbol}${formatCostMetric(avgCPV)}`; 
      cpvBox.style.display = 'block';
      
      // Display CPV benchmark comparison - pass the current filter
      displayCPVBenchmark(avgCPV, currentResultsFilter);
    } else {
      cpvBox.style.display = 'none';
    }
  }
}

/**
* Initialize the dashboard application
*/
function initializeDashboard() {
   // IMMEDIATELY hide the analysis tab to prevent flash of content
   const analysisTab = document.getElementById('analysis');
   if (analysisTab) {
       analysisTab.style.display = 'none';
       analysisTab.classList.remove('active'); // Ensure it doesn't have active class
   }
   
   // Also hide the analysis nav item initially
   const analysisNavItem = document.querySelector('.nav-item[onclick="showTab(\'analysis\')"]');
   if (analysisNavItem) {
       analysisNavItem.style.display = 'none';
   }
   
   // Continue with the rest of initialization...
   // Validate configuration
   if (!validateConfig()) {
       showError('Configuration not properly set. Please check config.js file.');
       return;
   }
   
   // Get campaign ID from URL parameters
    const urlParams = new URLSearchParams(window.location.search);
    currentCampaignId = decodeURIComponent(urlParams.get('campaignId') || '');
    const validatedCampaignId = validateSheetId(currentCampaignId);
    if (!validatedCampaignId) {
        showError('Invalid or missing campaignId parameter in URL. Add ?campaignId=YOUR_GOOGLE_SHEET_ID');
        return;
    }
    currentCampaignId = validatedCampaignId;
   
   // Load the data
   loadCampaignInfo();
   loadInfluencerData();
   loadContentData();
   loadWeeklyData();
   loadPaidMediaData();
   loadBenchmarkData();
}

/**
* Load influencer data from Google Sheets Report tab
*/
async function loadInfluencerData() {
   try {
       showLoading();
       
       // Fetch data from the Report sheet
       const data = await fetchSheetData(
           currentCampaignId, 
           'Report',
           'A1:AD1000'
       );
       
       // Process the data for influencers
       const processedData = processReportData(data);
       influencersData = processedData.influencers;
       window.influencersData = influencersData;
       
       // Update the display
       updateInfluencersDisplay();
       showDashboard();

   } catch (error) {
       console.error('Error loading data:', error);
       showError(`Failed to load influencer data: ${error.message}`);
   }
}

/**
 * Set default date range for Results tab based on content data
 */
function setDefaultResultsDateRange() {
    if (!contentData || contentData.length === 0) return;
    
    // Find oldest and newest dates
    let oldestDate = null;
    let newestDate = null;
    
    contentData.forEach(content => {
        if (content.postedDate) {
            const date = parseContentDate(content.postedDate);
            if (date) {
                if (!oldestDate || date < oldestDate) {
                    oldestDate = date;
                }
                if (!newestDate || date > newestDate) {
                    newestDate = date;
                }
            }
        }
    });
    
    // Set the date inputs if we found valid dates
    if (oldestDate && newestDate) {
        const startDateInput = document.getElementById('resultsStartDate');
        const endDateInput = document.getElementById('resultsEndDate');
        
        if (startDateInput && endDateInput) {
            // Add 1 day to the newest date to include it in the range
            const adjustedEndDate = new Date(newestDate);
            adjustedEndDate.setDate(adjustedEndDate.getDate() + 1);
            
            // Format dates as YYYY-MM-DD for input type="date"
            startDateInput.value = formatDateForInput(oldestDate);
            endDateInput.value = formatDateForInput(adjustedEndDate);
            
            console.log('Set default Results date range (display only - NOT filtering):', {
                originalStart: oldestDate,
                originalEnd: newestDate,
                adjustedEnd: adjustedEndDate,
                startInput: startDateInput.value,
                endInput: endDateInput.value
            });
            
            // IMPORTANT: Do NOT set useDateFilterForResults = true here
        }
    }
}

/**
 * Clear Results tab date filters
 */
function clearResultsDateFilters() {
    document.getElementById('resultsStartDate').value = '';
    document.getElementById('resultsEndDate').value = '';
    
    useDateFilterForResults = false; // Reset the flag
    
    // Reset to show all data
    updateResultsDisplay();
}

/**
 * Filter results by date range
 */
function filterResultsByDate() {
    useDateFilterForResults = true; // Enable filtering when user manually changes dates
    
    // Simply refresh the display - it will automatically use the date filters
    updateResultsDisplay();
}

/**
 * Validate and sanitize Google Sheet ID
 */
function validateSheetId(sheetId) {
    if (!sheetId) return null;
    
    // Google Sheet IDs can contain letters, numbers, hyphens, and underscores
    // They are typically 44 characters long but can vary
    const cleanId = sheetId.trim();
    
    // Basic validation - ensure it's not empty and contains valid characters
    if (!/^[a-zA-Z0-9\-_]+$/.test(cleanId)) {
        console.error('Invalid Sheet ID format:', cleanId);
        return null;
    }
    
    return cleanId;
}

/**
* Fetch data from Google Sheets via backend API
*/
async function fetchSheetData(sheetId, sheetName, range) {
   // Properly encode all parameters
   const params = new URLSearchParams({
       sheetId: sheetId,
       sheetName: sheetName,
       range: range
   });
   const url = `${CONFIG.API_BASE_URL}/sheets?${params.toString()}`;
   
   const response = await fetch(url, {
       method: 'GET',
       headers: {
           'x-api-key': CONFIG.API_SECRET_KEY,
           'Content-Type': 'application/json'
       }
   });
   
   if (!response.ok) {
       let errorMessage;
       try {
           const error = await response.json();
           errorMessage = error.error || error.details || 'Unknown error';
       } catch {
           errorMessage = `HTTP ${response.status}: ${response.statusText}`;
       }
       throw new Error(errorMessage);
   }
   
   const result = await response.json();
   return result.data || [];
}

// Parses values like "$1,234.56", "£1.234,56", "€8,24" → number
function parseCurrencyValue(val) {
 if (val === undefined || val === null) return 0;
 let s = String(val).trim().replace(/\s+/g, '');
 if (s.includes(',') && !s.includes('.')) s = s.replace(/\./g, '').replace(',', '.');
 else s = s.replace(/,/g, '');
 s = s.replace(/[^0-9.\-]/g, '');
 const n = parseFloat(s);
 return Number.isFinite(n) ? n : 0;
}


/**
* Process Report data into influencers and content objects
*/
function processReportData(data) {
   if (data.length < 2) return { influencers: [], content: [] };
   
   const rows = data.slice(1); // Skip header row
   const influencersMap = new Map();
   const content = [];
   
   rows.forEach((row, index) => {
       const name = cleanString(row[0]) || '';           // A: Name
       const profileLink = cleanString(row[1]) || '';    // B: Profile Link
       
       // Parse followers - handle both "123456" and "123,456" formats
       const followersRaw = row[2] || 0;
       const followers = parseInt(followersRaw.toString().replace(/,/g, '')) || 0;
       
       const handle = formatHandle(cleanString(row[3]) || ''); // D: Handle
       const country = cleanString(row[4]) || '';        // E: Country
       const cost = cleanString(row[5]) || '';          // F: Cost
       const socialMedia = formatPlatformName(row[6]) || ''; // G: Social Media
       const profilePic = cleanString(row[7]) || '';     // H: Profile Picture
       const typeOfPost = cleanString(row[8]) || '';     // I: Type of post
       const linkToPost = cleanString(row[9]) || '';     // J: Link to Post
       const dateOfPost = cleanString(row[10]) || '';    // K: Date of post
       const thumbnail = cleanString(row[11]) || '';     // L: Thumbnail of post
       const caption = cleanString(row[12]) || '';       // M: Post Caption
       const bonus = cleanString(row[29]) || '';         // AD: Bonus (column 30, index 29)
       
       // NEW: Capture cost metrics from spreadsheet columns with currency detection
       const cpeRaw = cleanString(row[24]) || '';  // Y: CPE (column 25, index 24)
       const cpvRaw = cleanString(row[25]) || '';  // Z: CPV (column 26, index 25)
       const ecpmRaw = cleanString(row[28]) || ''; // AC: eCPM (column 29, index 28)
       
       const spreadsheetCPE = parseCurrencyValue(cpeRaw);
       const spreadsheetCPV = parseCurrencyValue(cpvRaw);
       const spreadsheetECPM = parseCurrencyValue(ecpmRaw);
       
       // Skip rows without name, social media, or profile link
       if (!name || !socialMedia || !profileLink) return;
       
       // Create/update influencer entry using Profile Link as unique key
       if (!influencersMap.has(profileLink)) {
           influencersMap.set(profileLink, {
               id: influencersMap.size + 1,
               name: name,
               profileLink: profileLink,
               followers: followers,  // Now properly parsed with or without commas
               handle: handle,
               country: country,
               cost: cost,
               socialMedia: socialMedia,
               profilePic: profilePic,
               followersFormatted: formatNumber(followers)
           });
       } else {
           const existingInfluencer = influencersMap.get(profileLink);
           
           if (!existingInfluencer.profilePic && profilePic) {
               existingInfluencer.profilePic = profilePic;
           }
           
           if (country && country !== existingInfluencer.country) {
               if (!existingInfluencer.country) {
                   existingInfluencer.country = country;
               } else if (!existingInfluencer.country.includes(country)) {
                   existingInfluencer.country = existingInfluencer.country + ', ' + country;
               }
           }
       }
       
       // Create content entry if there's post data
       if (linkToPost || thumbnail || caption || typeOfPost) {
           // Store raw metrics for ER calculation later
           const likes = parseMetricNumber(row[13]);      // N: Likes
           const comments = parseMetricNumber(row[14]);   // O: Comments
           const shares = parseMetricNumber(row[15]);     // P: Shares
           const saves = parseMetricNumber(row[16]);      // Q: Saves
           const views = parseMetricNumber(row[17]);      // R: Views
           const stickerTaps = parseMetricNumber(row[20]); // U: Sticker Taps
           
           content.push({
               id: content.length + 1,
               influencer: name,
               profileLink: profileLink,  // ⭐ IMPORTANT: Store profile link with content
               followers: followers,  // Now properly parsed with or without commas
               title: caption || `${socialMedia} ${typeOfPost || 'post'} by ${name}`,
               thumbnail: thumbnail,
               platform: socialMedia,
               mediaType: getMediaTypeValue(socialMedia, typeOfPost),
               postedDate: dateOfPost,
               videoLink: linkToPost,
               cost: cost,
               bonus: /^yes$/i.test(bonus.trim()),
               metrics: {
                   likes: likes,
                   comments: comments,
                   shares: shares,
                   saves: saves,
                   views: views,
                   reach: parseMetricNumber(row[22]),      // W: Reach
                   impressions: parseMetricNumber(row[23]), // X: Impressions
                   impressions2: parseMetricNumber(row[27]) // AB: Impressions2
               },
               analytics: {
                   er: 0, // Will be calculated in summary functions
                   stories: parseMetricNumber(row[19]),    // T: N. of Stories
                   stickerTaps: stickerTaps,              // U: Sticker Taps
                   linkClicks: parseMetricNumber(row[21])  // V: Link Clicks
               },
               costMetrics: {
                   cpe: spreadsheetCPE,    // Y: CPE from spreadsheet
                   cpv: spreadsheetCPV,    // Z: CPV from spreadsheet
                   ecpm: spreadsheetECPM   // AC: eCPM from spreadsheet
               }
           });
       }
   });
   
   return {
       influencers: Array.from(influencersMap.values()),
       content: content
   };
}

/**
 * Swap the date ranges between Period 1 and Period 2
 */
function swapDateRanges() {
    const startDate1 = document.getElementById('contentStartDate');
    const endDate1 = document.getElementById('contentEndDate');
    const startDate2 = document.getElementById('contentCompareStartDate');
    const endDate2 = document.getElementById('contentCompareEndDate');
    
    if (!startDate1 || !endDate1 || !startDate2 || !endDate2) return;
    
    // Store Period 1 values
    const tempStart = startDate1.value;
    const tempEnd = endDate1.value;
    
    // Swap: Period 1 <- Period 2
    startDate1.value = startDate2.value;
    endDate1.value = endDate2.value;
    
    // Swap: Period 2 <- Period 1 (stored values)
    startDate2.value = tempStart;
    endDate2.value = tempEnd;
    
    console.log('Date ranges swapped:', {
        newPeriod1: { start: startDate1.value, end: endDate1.value },
        newPeriod2: { start: startDate2.value, end: endDate2.value }
    });
    
    // Refresh the display with swapped dates
    filterContent();
}

/**
* Map social media and type of post to media type values
*/
function getMediaTypeValue(socialMedia, typeOfPost) {
   const platform = socialMedia.toLowerCase();
   const type = typeOfPost.toLowerCase();
   
   if (platform === 'instagram') {
       if (type.includes('post')) return 'post';
       if (type.includes('reel')) return 'reel';
       if (type.includes('story') || type.includes('stories')) return 'story';
   } else if (platform === 'youtube') {
       if (type.includes('integration')) return 'integration';
       if (type.includes('short')) return 'short';
   } else if (platform === 'linkedin') {
       if (type.includes('video')) return 'video';
       return 'post';
   } else if (platform === 'facebook') {
       if (type.includes('reel')) return 'reel';
       return 'post';
   } else if (platform === 'twitter') {
       if (type.includes('video')) return 'video';
       return 'post';
   }
   
   return type || ''; // Return original type or empty for TikTok
}

// =============================================================================
// DISPLAY FUNCTIONS
// =============================================================================

/**
* Update both influencer displays
*/
function updateInfluencersDisplay() {
   updateInfluencersGrid();
   updateInfluencersTable();
   updateMetricsSummary();
   updateOverviewDisplay();
   updateInfluencersPlatformFilter(); // Added this line
}

/**
* Update the influencers grid display
*/
function updateInfluencersGrid() {
   const grid = document.getElementById('influencersGrid');
   grid.innerHTML = '';
   
   if (influencersData.length === 0) {
       grid.innerHTML = `
           <div class="empty-state">
               <h3>No influencer data found</h3>
               <p>Make sure your Google Sheet has data in the "Report" tab with the correct column structure</p>
           </div>
       `;
       return;
   }
   
   influencersData.forEach(influencer => {
       const card = createInfluencerCard(influencer);
       grid.appendChild(card);
   });
}

function safeImageUrl(url) {
    if (!url || url.trim() === '') return '';
    
    url = url.trim();
    
    // Remove protocol without regex
    let urlWithoutProtocol = url;
    if (url.startsWith('https://')) {
        urlWithoutProtocol = url.substring(8);
    } else if (url.startsWith('http://')) {
        urlWithoutProtocol = url.substring(7);
    }
    
    return `https://images.weserv.nl/?url=${encodeURIComponent(urlWithoutProtocol)}&w=300&h=400&fit=cover`;
}

/**
* Create image element with automatic fallback to proxy
*/
function createImageWithFallback(src, alt, className, fallbackHTML) {
    const img = document.createElement('img');
    img.src = src;
    img.alt = alt;
    img.className = className;
    
    let proxyAttempted = false;
    
    img.onerror = function() {
        if (!proxyAttempted) {
            proxyAttempted = true;
            // Try proxy
            this.src = getProxyImageUrl(src);
        } else {
            // Proxy also failed, replace with fallback
            const fallbackDiv = document.createElement('div');
            fallbackDiv.innerHTML = fallbackHTML;
            const fallbackElement = fallbackDiv.firstChild;
            this.parentNode.replaceChild(fallbackElement, this);
        }
    };
    
    return img.outerHTML;
}

/**
* Create inline SVG avatar (no external requests)
*/
function createAvatarSVG(initials, backgroundColor) {
   return `
       <svg width="60" height="60" xmlns="http://www.w3.org/2000/svg">
           <rect width="60" height="60" fill="${backgroundColor}" rx="30"/>
           <text x="50%" y="50%" dominant-baseline="middle" text-anchor="middle" fill="white" font-size="24" font-weight="bold">${initials}</text>
       </svg>
   `;
}

function createInfluencerCard(influencer) {
    const card = document.createElement('div');
    card.className = 'influencer-card';
    card.dataset.name = influencer.name.toLowerCase();
    card.dataset.platform = influencer.socialMedia.toLowerCase();
    card.dataset.profileLink = influencer.profileLink;
    
    // Handle profile image
    let avatarHTML;
    if (influencer.profilePic && influencer.profilePic.trim() !== '') {
        const proxiedImageUrl = safeImageUrl(influencer.profilePic);
        const initials = getInitials(influencer.name);
        const avatarColor = getAvatarColor(influencer.socialMedia);
        
        avatarHTML = `
            <img src="${proxiedImageUrl}" 
                 alt="${influencer.name}" 
                 class="avatar"
                 onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
            <div class="avatar" style="display:none; background: ${avatarColor}; color: white; font-size: 1.8rem; font-weight: bold; align-items: center; justify-content: center;">
                ${initials}
            </div>
        `;
    } else {
        const initials = getInitials(influencer.name);
        const avatarColor = getAvatarColor(influencer.socialMedia);
        avatarHTML = `
            <div class="avatar" style="background: ${avatarColor}; color: white; font-size: 1.8rem; font-weight: bold; display: flex; align-items: center; justify-content: center;">
                ${initials}
            </div>
        `;
    }
    
    // Rest of the card creation code...
    let handleHTML;
    if (influencer.profileLink && influencer.profileLink.trim() !== '') {
        handleHTML = `
            <a href="${influencer.profileLink}" target="_blank" class="clickable-handle" onclick="event.stopPropagation();">
                ${escapeHtml(influencer.handle)}
                <span class="profile-link-icon">🔗</span>
            </a>
        `;
    } else {
        handleHTML = `<div class="handle">${escapeHtml(influencer.handle)}</div>`;
    }
    
    card.innerHTML = `
        <div class="influencer-header">
            ${avatarHTML}
            <div class="influencer-info">
                <h3>${escapeHtml(influencer.name)}</h3>
                ${handleHTML}
            </div>
        </div>
        
        <div class="influencer-content">
            <div class="metrics-row">
                <div class="metric">
                    <div class="metric-label">Followers</div>
                    <div class="metric-value">${influencer.followersFormatted}</div>
                </div>
                <div class="metric">
                    <div class="metric-label">Country</div>
                    <div class="metric-value" style="font-size: 1.2rem;">${influencer.country || 'N/A'}</div>
                </div>
            </div>
            <div class="platform-badge platform-${influencer.socialMedia.toLowerCase()}">
                ${influencer.socialMedia}
            </div>
        </div>
    `;
    
    return card;
}

/**
* Get a nice avatar color based on platform
*/
function getAvatarColor(platform) {
   const colors = {
       'instagram': 'linear-gradient(45deg, #f09433 0%, #e6683c 25%, #dc2743 50%, #cc2366 75%, #bc1888 100%)',
       'tiktok': '#000000',
       'youtube': '#ff0000',
       'facebook': '#4267B2',
       'twitter': '#1DA1F2',
       'linkedin': '#0077B5'
   };
   return colors[platform.toLowerCase()] || '#fc0060';
}

/**
* Update the influencers table display
*/
function updateInfluencersTable() {
   const tableContent = document.getElementById('tableContent');
   if (!tableContent) return;
   
   tableContent.innerHTML = '';
   
   if (influencersData.length === 0) {
       tableContent.innerHTML = `
           <div class="table-row">
               <div style="grid-column: 1/-1; text-align: center; padding: 2rem; color: #64748b;">
                   <h3>No influencer data found</h3>
                   <p>Make sure your Google Sheet has data in the "Report" tab</p>
               </div>
           </div>
       `;
       return;
   }
   
   influencersData.forEach(influencer => {
       const row = createInfluencerTableRow(influencer);
       tableContent.appendChild(row);
   });
}

/**
* Update metrics summary based on visible influencers
*/
function updateMetricsSummary() {
   let visibleInfluencers = [];
   
   if (currentView === 'grid') {
       const cards = document.querySelectorAll('.influencer-card');
       cards.forEach(card => {
           if (card.style.display !== 'none') {
               const profileLink = card.dataset.profileLink;
               const influencer = influencersData.find(inf => inf.profileLink === profileLink);
               if (influencer) visibleInfluencers.push(influencer);
           }
       });
   } else {
       const rows = document.querySelectorAll('.table-row');
       rows.forEach(tableRow => {
           if (tableRow.dataset.profileLink && tableRow.style.display !== 'none') {
               const profileLink = tableRow.dataset.profileLink;
               const influencer = influencersData.find(inf => inf.profileLink === profileLink);
               if (influencer) visibleInfluencers.push(influencer);
           }
       });
   }
   
   // Calculate metrics
   const uniqueInfluencers = new Set(visibleInfluencers.map(inf => inf.name.toLowerCase())).size;
   const totalAccounts = visibleInfluencers.length;
   
   // Sum followers from ALL visible accounts (not unique influencers)
   const totalFollowers = visibleInfluencers.reduce((sum, inf) => sum + inf.followers, 0);
   
   document.getElementById('influencersCount').textContent = uniqueInfluencers;
   document.getElementById('accountsCount').textContent = totalAccounts;
   document.getElementById('totalFollowers').textContent = formatNumber(totalFollowers);
}

/**
* Update overview display with current data
*/
function updateOverviewDisplay() {
   // Calculate overview metrics
   const uniqueInfluencers = new Set(influencersData.map(inf => inf.name.toLowerCase())).size;
   const totalAccounts = influencersData.length;
   
   // Calculate total followers by summing ALL accounts (not unique influencers)
   const totalFollowers = influencersData.reduce((sum, inf) => sum + inf.followers, 0);
   
   // Calculate ORGANIC link clicks from content data (Column V - linkClicks)
   const organicLinkClicks = contentData.reduce((sum, content) => sum + (content.analytics.linkClicks || 0), 0);
   
   // Get paid clicks from paid media data (sum of all "Clicks" columns)
   const paidClicks = paidMediaData.totalPaidClicks || 0;
   
   // TOTAL CLICKS = Organic + Paid
   const totalClicks = organicLinkClicks + paidClicks;
   
   console.log('Clicks breakdown:', {
       organic: organicLinkClicks,
       paid: paidClicks,
       total: totalClicks
   });
   
   // Calculate content pieces with stories logic
   const totalContentPieces = calculateTotalContentPieces();
   
   // Get unique countries
   const countries = getUniqueCountries();
   
   // Update main stats
   const overviewInfluencersEl = document.getElementById('overviewInfluencers');
   const overviewAccountsEl = document.getElementById('overviewAccounts');
   const overviewCountriesEl = document.getElementById('overviewCountries');
   const overviewFollowersEl = document.getElementById('overviewFollowers');
   const overviewContentEl = document.getElementById('overviewContent');
   const overviewLinkClicksEl = document.getElementById('overviewLinkClicks');
   
   if (overviewInfluencersEl) overviewInfluencersEl.textContent = uniqueInfluencers;
   if (overviewAccountsEl) overviewAccountsEl.textContent = totalAccounts;
   if (overviewCountriesEl) overviewCountriesEl.textContent = countries;
   if (overviewFollowersEl) overviewFollowersEl.textContent = formatNumber(totalFollowers);
   if (overviewContentEl) overviewContentEl.textContent = totalContentPieces;
   
   if (overviewLinkClicksEl) {
       const linkClicksCard = overviewLinkClicksEl.closest('.overview-stat-card');
       if (linkClicksCard) {
           if (totalClicks === 0) {
               // Remove the card entirely from the DOM
               linkClicksCard.remove();
           } else {
               // Update the value with combined clicks
               overviewLinkClicksEl.textContent = formatNumber(totalClicks);
           }
       }
   }
   
   // Update platform breakdowns
   updateFollowersBreakdown();
   updateContentBreakdown();
   updateContentTypesBreakdown();
}

/**
* Calculate total content pieces: all rows + sum of column T - count of "story" in column I
*/
function calculateTotalContentPieces() {
   let totalPieces = 0;
   
   // Process each content row
   contentData.forEach(content => {
       const isInstagramStories = content.platform.toLowerCase() === 'instagram' && 
                                 content.mediaType === 'story';
       
       if (isInstagramStories) {
           // For Instagram Stories, add the number of stories (Column T)
           totalPieces += content.analytics.stories || 1;
       } else {
           // For regular content, count as 1 piece
           totalPieces += 1;
       }
   });
   
   return totalPieces;
}

/**
* Get unique countries from influencer data (Column E)
*/
function getUniqueCountries() {
   const allCountries = new Set();
   
   influencersData.forEach(influencer => {
       if (influencer.country && influencer.country.trim()) {
           // Split by comma and clean up each country
           const countries = influencer.country.split(',')
               .map(country => country.trim())
               .filter(country => country.length > 0);
           
           countries.forEach(country => allCountries.add(country));
       }
   });
   
   return Array.from(allCountries).join(', ') || 'N/A';
}

/**
* Update followers breakdown by platform
*/
function updateFollowersBreakdown() {
   const container = document.getElementById('followersBreakdown');
   if (!container) return;
   
   container.innerHTML = '';
   
   // Group followers by platform - sum ALL accounts
   const followersByPlatform = {};
   influencersData.forEach(influencer => {
       const platform = influencer.socialMedia.toLowerCase();
       if (!followersByPlatform[platform]) {
           followersByPlatform[platform] = 0;
       }
       // Add followers from this account
       followersByPlatform[platform] += influencer.followers;
   });
   
   // Create breakdown items
   Object.entries(followersByPlatform).forEach(([platform, followers]) => {
       const item = document.createElement('div');
       item.className = 'overview-breakdown-item';
       item.innerHTML = `
           <div class="overview-breakdown-item-label">
               <div class="platform-icon ${platform}">
                   ${platform.charAt(0).toUpperCase()}
               </div>
               ${formatPlatformName(platform)}
           </div>
           <div class="overview-breakdown-item-value">${formatNumber(followers)}</div>
       `;
       container.appendChild(item);
   });
}

/**
* Update content breakdown by platform with stories logic
*/
function updateContentBreakdown() {
   const container = document.getElementById('contentBreakdown');
   if (!container) return;
   
   container.innerHTML = '';
   
   // Group content by platform with special handling for Instagram
   const contentByPlatform = {};
   
   contentData.forEach(content => {
       const platform = content.platform.toLowerCase();
       if (!contentByPlatform[platform]) {
           contentByPlatform[platform] = 0;
       }
       
       const isInstagramStories = platform === 'instagram' && content.mediaType === 'story';
       
       if (isInstagramStories) {
           // For Instagram Stories, add the number of stories
           contentByPlatform[platform] += content.analytics.stories || 1;
       } else {
           // For regular content, count as 1 piece
           contentByPlatform[platform] += 1;
       }
   });
   
   // Create breakdown items
   Object.entries(contentByPlatform).forEach(([platform, count]) => {
       const item = document.createElement('div');
       item.className = 'overview-breakdown-item';
       item.innerHTML = `
           <div class="overview-breakdown-item-label">
               <div class="platform-icon ${platform}">
                   ${platform.charAt(0).toUpperCase()}
               </div>
               ${formatPlatformName(platform)}
           </div>
           <div class="overview-breakdown-item-value">${count}</div>
       `;
       container.appendChild(item);
   });
}

/**
* Update content types breakdown (replaces platforms breakdown)
*/
function updateContentTypesBreakdown() {
   const container = document.getElementById('contentTypesBreakdown');
   if (!container) return;
   
   container.innerHTML = '';
   
   // Group content by platform and media type
   const contentTypes = {};
   
   contentData.forEach(content => {
       const platform = content.platform;
       const mediaType = content.mediaType;
       
       let typeKey;
       let displayName;
       
       if (platform.toLowerCase() === 'instagram') {
           if (mediaType === 'story') {
               typeKey = 'instagram_stories';
               displayName = 'Instagram Stories';
           } else if (mediaType === 'reel') {
               typeKey = 'instagram_reels';
               displayName = 'Instagram Reels';
           } else if (mediaType === 'post') {
               typeKey = 'instagram_posts';
               displayName = 'Instagram Posts';
           } else {
               typeKey = 'instagram_other';
               displayName = 'Instagram Content';
           }
       } else if (platform.toLowerCase() === 'youtube') {
           if (mediaType === 'integration') {
               typeKey = 'youtube_integrations';
               displayName = 'YouTube Integrations';
           } else if (mediaType === 'short') {
               typeKey = 'youtube_shorts';
               displayName = 'YouTube Shorts';
           } else {
               typeKey = 'youtube_other';
               displayName = 'YouTube Content';
           }
       } else if (platform.toLowerCase() === 'tiktok') {
           typeKey = 'tiktok_content';
           displayName = 'TikTok Content';
       } else {
           typeKey = `${platform.toLowerCase()}_content`;
           displayName = `${platform} Content`;
       }
       
       if (!contentTypes[typeKey]) {
           contentTypes[typeKey] = {
               count: 0,
               displayName: displayName,
               platform: platform.toLowerCase()
           };
       }
       
       // Count logic: Instagram Stories use story count, others count as 1
       if (typeKey === 'instagram_stories') {
           contentTypes[typeKey].count += content.analytics.stories || 1;
       } else {
           contentTypes[typeKey].count += 1;
       }
   });
   
   // Create breakdown items
   Object.entries(contentTypes).forEach(([key, data]) => {
       const item = document.createElement('div');
       item.className = 'overview-breakdown-item';
       item.innerHTML = `
           <div class="overview-breakdown-item-label">
               <div class="platform-icon ${data.platform}">
                   ${data.platform.charAt(0).toUpperCase()}
               </div>
               ${data.displayName}
           </div>
           <div class="overview-breakdown-item-value">${data.count}</div>
       `;
       container.appendChild(item);
   });
}

/**
* Create an individual table row for an influencer
*/
function createInfluencerTableRow(influencer) {
    const row = document.createElement('div');
    row.className = 'table-row';
    row.dataset.name = influencer.name.toLowerCase();
    row.dataset.platform = influencer.socialMedia.toLowerCase();
    row.dataset.profileLink = influencer.profileLink;
    
    // Handle profile image with fallback
    let avatarHTML;
    if (influencer.profilePic && influencer.profilePic.trim() !== '') {
        const proxiedImageUrl = safeImageUrl(influencer.profilePic);
        avatarHTML = `
            <img src="${proxiedImageUrl}" 
                 alt="${influencer.name}" 
                 class="table-avatar"
                 onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
            <div class="table-avatar" style="display:none;">
                ${getInitials(influencer.name)}
            </div>
        `;
    } else {
        avatarHTML = `
            <div class="table-avatar">
                ${getInitials(influencer.name)}
            </div>
        `;
    }
    
    row.innerHTML = `
        ${avatarHTML}
        <div class="table-influencer-info">
            <div class="table-name">${escapeHtml(influencer.name)}</div>
            <div class="table-handle">${escapeHtml(influencer.handle)}</div>
        </div>
        <div class="table-metric">${influencer.followersFormatted}</div>
        <div>
            <span class="table-platform-badge platform-${influencer.socialMedia.toLowerCase()}">
                ${influencer.socialMedia}
            </span>
        </div>
        <div class="table-handle">${escapeHtml(influencer.handle)}</div>
    `;
    
    return row;
}

/**
* Create content card for grid view with media type and stories support
*/
function createContentCard(content) {
    const card = document.createElement('div');
    card.className = 'content-card';
    card.dataset.contentId = content.id;
    card.dataset.title = content.title.toLowerCase();
    card.dataset.platform = content.platform.toLowerCase();
    card.dataset.influencer = content.influencer.toLowerCase();
    card.dataset.mediaType = content.mediaType || '';
    
    // Check if this is Instagram Stories or Posts
    const isInstagramStories = content.platform.toLowerCase() === 'instagram' && 
                              content.mediaType === 'story';
    const isInstagramPosts = content.platform.toLowerCase() === 'instagram' && 
                            content.mediaType === 'post';
    
    // Calculate individual ER for this content piece
    let individualER = 0;
    if (content.metrics.views > 0 && !isInstagramPosts) {
        const totalEngagement = (content.metrics.likes || 0) + 
                               (content.metrics.comments || 0) + 
                               (content.metrics.shares || 0) + 
                               (content.metrics.saves || 0);
        individualER = (totalEngagement / content.metrics.views) * 100;
    }

    
    // Make card clickable if video link exists
    if (content.videoLink && content.videoLink.trim() !== '') {
        card.style.cursor = 'pointer';
        card.onclick = () => window.open(content.videoLink, '_blank');
    }
    
    // Thumbnail with fallback and play overlay - different for stories
    let thumbnailHTML;
    if (isInstagramStories) {
        // Instagram Stories - no thumbnail, show stories icon and count
        thumbnailHTML = `
            <div class="stories-thumbnail-container">
                <div class="stories-icon-container">
                    <span class="stories-icon">📖</span>
                    <div class="stories-count">${content.analytics.stories || 1}</div>
                    <div class="stories-label">Stories</div>
                </div>
            </div>
        `;
    } else {
        // Regular content with thumbnail
        if (content.thumbnail && content.thumbnail.trim() !== '') {
            const proxiedThumbnail = safeImageUrl(content.thumbnail);
            thumbnailHTML = `
                <img src="${proxiedThumbnail}" 
                     alt="${content.title}" 
                     class="content-thumbnail"
                     onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                <div class="content-thumbnail-fallback" style="display:none;">
                    <span class="play-icon">▶</span>
                </div>
                ${content.videoLink ? '<div class="video-overlay"><div class="play-button">▶</div></div>' : ''}
            `;
        } else {
            thumbnailHTML = `
                <div class="content-thumbnail-fallback">
                    <span class="play-icon">▶</span>
                </div>
                ${content.videoLink ? '<div class="video-overlay"><div class="play-button">▶</div></div>' : ''}
            `;
        }
    }
    
    // Create metrics HTML - only show non-zero metrics
    const metrics = [
        { key: 'views', icon: '👁', label: 'Views' },
        { key: 'likes', icon: '❤️', label: 'Likes' },
        { key: 'comments', icon: '💬', label: isInstagramStories ? 'Replies' : 'Comments' },
        { key: 'shares', icon: '📤', label: 'Shares' },
        { key: 'saves', icon: '🔖', label: 'Saves' }
    ];
    
    const visibleMetrics = metrics
        .filter(metric => content.metrics[metric.key] > 0)
        .map(metric => `
            <div class="content-metric">
                <span class="metric-icon">${metric.icon}</span>
                <span class="metric-value">${formatNumber(content.metrics[metric.key])}</span>
                <span class="metric-label">${metric.label}</span>
            </div>
        `).join('');
    
    // Add stories-specific metrics if it's Instagram Stories
    let storiesSpecificMetrics = '';
    if (isInstagramStories && content.analytics.stickerTaps > 0) {
        storiesSpecificMetrics = `
            <div class="content-metric">
                <span class="metric-icon">👆</span>
                <span class="metric-value">${formatNumber(content.analytics.stickerTaps)}</span>
                <span class="metric-label">Sticker Taps</span>
            </div>
        `;
    }
    
    // Add link clicks metric for Instagram (all types)
    let linkClicksMetricHTML = '';
    if (content.analytics.linkClicks > 0) {
        linkClicksMetricHTML = `
            <div class="content-metric">
                <span class="metric-icon">🔗</span>
                <span class="metric-value">${formatNumber(content.analytics.linkClicks)}</span>
                <span class="metric-label">Link Clicks</span>
            </div>
        `;
    }
    
    // Add individual ER metric (exclude Instagram Posts and Instagram Stories)
    let erMetricHTML = '';
    if (!isInstagramPosts && !isInstagramStories && individualER > 0) {
        erMetricHTML = `
            <div class="content-metric">
                <span class="metric-icon">📈</span>
                <span class="metric-value">${individualER.toFixed(2)}%</span>
                <span class="metric-label">ER</span>
            </div>
        `;
    }
    
    // Create platform badge with proper content type display
    let platformBadgeText = content.platform;
    if (content.mediaType) {
        if (content.mediaType === 'post') {
            platformBadgeText += ' Post';
        } else if (content.mediaType === 'reel') {
            platformBadgeText += ' Reel';
        } else if (content.mediaType === 'story') {
            platformBadgeText += ' Stories';
        } else if (content.mediaType === 'integration') {
            platformBadgeText += ' Integration';
        } else if (content.mediaType === 'short') {
            platformBadgeText += ' Shorts';
        } else if (content.mediaType === 'video') {
            platformBadgeText += ' Video';
        }
    }
    
    // Add bonus label if applicable
    const bonusLabel = content.bonus ? '<span class="bonus-label">Bonus content</span>' : '';
    
    card.innerHTML = `
        <div class="content-thumbnail-container">
            ${thumbnailHTML}
        </div>
        <div class="content-info">
            <h3 class="content-title">${escapeHtml(content.title)}</h3>
            <div class="content-creator">
                <span>${escapeHtml(content.influencer)}</span>
                <span class="creator-followers">${content.followers > 0 ? `• ${formatNumber(content.followers)} followers` : ''}</span>
            </div>
            <div class="content-date">${content.postedDate}</div>
        </div>
        <div class="content-footer">
            <span class="platform-badge platform-${content.platform.toLowerCase()} content-platform-badge">
                ${platformBadgeText}
            </span>
            ${bonusLabel}
        </div>
        <div class="content-metrics-expanded">
            ${visibleMetrics}
            ${storiesSpecificMetrics}
            ${linkClicksMetricHTML}
            ${erMetricHTML}
        </div>
    `;
    
    return card;
}

/**
* Update both content displays
*/
function updateContentDisplay() {
   updateContentGrid();
   updateContentTable();
   updateContentMetricsSummary();
   updateResultsDisplay();
   updateWeeklyResultsDisplay();
   updateOverviewDisplay();
   updateContentPlatformFilter();    // Added this line
   renderPlatformFilters();           // Added this line
   renderSubFilters(currentPlatform); // Added this line
}

/**
* Update content grid view with period sections
*/
function updateContentGrid() {
   const grid = document.getElementById('contentGrid');
   if (!grid) return;
   
   grid.innerHTML = '';
   
   // Add sort dropdown at the top
   const sortContainer = document.createElement('div');
   sortContainer.innerHTML = getContentSortDropdownHTML();
   sortContainer.style.gridColumn = '1 / -1';
   grid.appendChild(sortContainer);
   
   if (contentData.length === 0) {
       const emptyState = document.createElement('div');
       emptyState.className = 'empty-state';
       emptyState.style.gridColumn = '1 / -1';
       emptyState.innerHTML = `
           <h3>No content data found</h3>
           <p>Make sure your Google Sheet has data in the "Report" tab</p>
       `;
       grid.appendChild(emptyState);
       return;
   }
   
   // Check if comparison mode is active
   const startDate = document.getElementById('contentStartDate')?.value;
   const endDate = document.getElementById('contentEndDate')?.value;
   const compareStartDate = document.getElementById('contentCompareStartDate')?.value;
   const compareEndDate = document.getElementById('contentCompareEndDate')?.value;
   
   console.log('Comparison mode check:', {
       isComparisonMode,
       startDate,
       endDate,
       compareStartDate,
       compareEndDate
   });
   
   if (isComparisonMode && startDate && endDate && compareStartDate && compareEndDate) {
       // COMPARISON MODE: Show two separate sections
       console.log('Rendering comparison view');
       
       // Period 1 Section
       const period1Content = getFilteredContentForPeriod(startDate, endDate);
       console.log('Period 1 content:', period1Content.length, 'items');
       createPeriodSection(grid, 'Period 1', startDate, endDate, period1Content, 'period-1');
       
       // Period 2 Section
       const period2Content = getFilteredContentForPeriod(compareStartDate, compareEndDate);
       console.log('Period 2 content:', period2Content.length, 'items');
       createPeriodSection(grid, 'Period 2', compareStartDate, compareEndDate, period2Content, 'period-2');
       
   } else {
       // NORMAL MODE: Show all filtered content (with optional date filter)
       console.log('Rendering normal view');
       
       const searchTerm = document.getElementById('contentSearchInput')?.value.toLowerCase() || '';
       const platformFilter = document.getElementById('contentPlatformFilter')?.value || '';
       const mediaTypeFilter = document.getElementById('contentMediaTypeFilter')?.value || '';
       
       let filteredContent = contentData.filter(content => {
           const title = content.title.toLowerCase();
           const platform = content.platform.toLowerCase();
           const influencer = content.influencer.toLowerCase();
           const mediaType = content.mediaType || '';
           const isBonus = content.bonus === true;
           
           const searchMatch = !searchTerm || title.includes(searchTerm) || influencer.includes(searchTerm);
           
           let platformMatch = true;
           if (platformFilter === 'bonus') {
               platformMatch = isBonus === true;
           } else if (platformFilter) {
               platformMatch = platform === platformFilter.toLowerCase();
           }
           
           const mediaTypeMatch = !mediaTypeFilter || mediaType === mediaTypeFilter;
           
           // Apply date filter only if dates are set AND user has manually changed them
           let dateMatch = true;
           if (useDateFilterForCalculations && startDate && endDate) {
               dateMatch = isDateInRange(content.postedDate, startDate, endDate);
           }
           
           return searchMatch && platformMatch && mediaTypeMatch && dateMatch;
       });
       
       // Show period header if date filtering is active
       if (useDateFilterForCalculations && startDate && endDate) {
           createPeriodSection(grid, 'Filtered Period', startDate, endDate, filteredContent, 'single-period');
       } else {
           // No period header, just show cards
           filteredContent.forEach(content => {
               const card = createContentCard(content);
               grid.appendChild(card);
           });
       }
   }
   
   // Restore previous sort if any
   const dropdown = document.getElementById('contentSortDropdown');
   if (dropdown && currentContentSortBy !== 'default') {
       dropdown.value = `${currentContentSortBy}-${currentContentSortDirection}`;
       setTimeout(() => sortContentGrid(), 100);
   }
}

/**
 * Get filtered content for a specific date period
 */
function getFilteredContentForPeriod(startDate, endDate) {
   const searchTerm = document.getElementById('contentSearchInput')?.value.toLowerCase() || '';
   const platformFilter = document.getElementById('contentPlatformFilter')?.value || '';
   const mediaTypeFilter = document.getElementById('contentMediaTypeFilter')?.value || '';
   
   return contentData.filter(content => {
       const title = content.title.toLowerCase();
       const platform = content.platform.toLowerCase();
       const influencer = content.influencer.toLowerCase();
       const mediaType = content.mediaType || '';
       const isBonus = content.bonus === true;
       
       const searchMatch = !searchTerm || title.includes(searchTerm) || influencer.includes(searchTerm);
       
       let platformMatch = true;
       if (platformFilter === 'bonus') {
           platformMatch = isBonus === true;
       } else if (platformFilter) {
           platformMatch = platform === platformFilter.toLowerCase();
       }
       
       const mediaTypeMatch = !mediaTypeFilter || mediaType === mediaTypeFilter;
       const dateMatch = isDateInRange(content.postedDate, startDate, endDate);
       
       return searchMatch && platformMatch && mediaTypeMatch && dateMatch;
   });
}

/**
 * Create a period section with header and content cards
 */
function createPeriodSection(grid, periodLabel, startDate, endDate, content, periodClass) {
   // Create period section header (spans full width)
   const sectionHeader = document.createElement('div');
   sectionHeader.className = 'period-section-header-wrapper';
   sectionHeader.style.gridColumn = '1 / -1'; // THIS IS KEY - spans all columns
   
   // Format dates for display
   const startDateObj = new Date(startDate);
   const endDateObj = new Date(endDate);
   const dateFormat = { month: 'short', day: 'numeric', year: 'numeric' };
   const formattedStart = startDateObj.toLocaleDateString('en-US', dateFormat);
   const formattedEnd = endDateObj.toLocaleDateString('en-US', dateFormat);
   
   // Calculate stats for this period
   const stats = {
       pieces: content.length,
       views: content.reduce((sum, c) => sum + (c.metrics.views || 0), 0),
       likes: content.reduce((sum, c) => sum + (c.metrics.likes || 0), 0)
   };
   
   // Create header HTML
   sectionHeader.innerHTML = `
       <div class="period-section-header ${periodClass}">
           <div class="period-section-icon">📅</div>
           <div class="period-section-title">
               <div class="period-section-label">${periodLabel}</div>
               <div class="period-section-dates">${formattedStart} - ${formattedEnd}</div>
           </div>
           <div class="period-section-stats">
               <div class="period-stat-item">
                   <div class="period-stat-label">Content</div>
                   <div class="period-stat-value">${stats.pieces}</div>
               </div>
               <div class="period-stat-item">
                   <div class="period-stat-label">Views</div>
                   <div class="period-stat-value">${formatNumber(stats.views)}</div>
               </div>
               <div class="period-stat-item">
                   <div class="period-stat-label">Likes</div>
                   <div class="period-stat-value">${formatNumber(stats.likes)}</div>
               </div>
           </div>
       </div>
   `;
   
   // Append header to grid
   grid.appendChild(sectionHeader);
   
   // Add content cards for this period
   if (content.length === 0) {
       const emptyMessage = document.createElement('div');
       emptyMessage.className = 'empty-period-message';
       emptyMessage.style.gridColumn = '1 / -1'; // Also spans full width
       emptyMessage.innerHTML = `
           <div class="empty-period-message-icon">📭</div>
           <div>No content found for this period</div>
       `;
       grid.appendChild(emptyMessage);
   } else {
       content.forEach(contentItem => {
           const card = createContentCard(contentItem);
           grid.appendChild(card);
       });
   }
}

// =============================================================================
// TABLE FUNCTIONS
// =============================================================================

/**
* Sort content table by column
*/
function sortContentTable(column) {
   // Toggle sort direction if clicking the same column
   if (currentSortColumn === column) {
       currentSortDirection = currentSortDirection === 'asc' ? 'desc' : 'asc';
   } else {
       currentSortColumn = column;
       currentSortDirection = 'desc';
   }
   
   // Clear all sort indicators
   document.querySelectorAll('.sort-indicator').forEach(indicator => {
       indicator.textContent = '';
   });
   
   // Set current sort indicator
   const indicator = document.getElementById(`sort-${column}`);
   if (indicator) {
       indicator.textContent = currentSortDirection === 'asc' ? '↑' : '↓';
   }
   
   // Get visible content based on current filters
   let visibleContent = getVisibleContentForTable();
   
   // Sort the visible content
   visibleContent.sort((a, b) => {
       let valueA, valueB;
       
       switch (column) {
           case 'title':
               valueA = a.title.toLowerCase();
               valueB = b.title.toLowerCase();
               break;
           case 'influencer':
               valueA = a.influencer.toLowerCase();
               valueB = b.influencer.toLowerCase();
               break;
           case 'platform':
               valueA = a.platform.toLowerCase();
               valueB = b.platform.toLowerCase();
               break;
           case 'postedDate':
               valueA = new Date(a.postedDate);
               valueB = new Date(b.postedDate);
               break;
           case 'views':
               valueA = a.metrics.views || 0;
               valueB = b.metrics.views || 0;
               break;
           case 'likes':
               valueA = a.metrics.likes || 0;
               valueB = b.metrics.likes || 0;
               break;
           case 'comments':
               valueA = a.metrics.comments || 0;
               valueB = b.metrics.comments || 0;
               break;
           case 'shares':
               valueA = a.metrics.shares || 0;
               valueB = b.metrics.shares || 0;
               break;
           case 'saves':
               valueA = a.metrics.saves || 0;
               valueB = b.metrics.saves || 0;
               break;
           case 'reach':
               valueA = a.metrics.reach || 0;
               valueB = b.metrics.reach || 0;
               break;
           case 'stickerTaps':
               valueA = a.analytics.stickerTaps || 0;
               valueB = b.analytics.stickerTaps || 0;
               break;
           case 'linkClicks':
               valueA = a.analytics.linkClicks || 0;
               valueB = b.analytics.linkClicks || 0;
               break;
           default:
               return 0;
       }
       
       // Handle different data types
       if (typeof valueA === 'string') {
           if (currentSortDirection === 'asc') {
               return valueA.localeCompare(valueB);
           } else {
               return valueB.localeCompare(valueA);
           }
       } else {
           if (currentSortDirection === 'asc') {
               return valueA - valueB;
           } else {
               return valueB - valueA;
           }
       }
   });
   
   // Store sorted data
   sortedContentData = visibleContent;
   
   // Re-render the table with sorted data
   renderSortedContentTable(visibleContent);
}

/**
* Get visible content based on current filters
*/
function getVisibleContentForTable() {
   const searchTerm = document.getElementById('contentSearchInput').value.toLowerCase();
   const platformFilter = document.getElementById('contentPlatformFilter').value;
   const mediaTypeFilter = document.getElementById('contentMediaTypeFilter').value;
   
   return contentData.filter(content => {
       const title = content.title.toLowerCase();
       const platform = content.platform.toLowerCase();
       const influencer = content.influencer.toLowerCase();
       const mediaType = content.mediaType || '';
       
       const searchMatch = !searchTerm || title.includes(searchTerm) || influencer.includes(searchTerm);
       const platformMatch = !platformFilter || platform === platformFilter.toLowerCase();
       const mediaTypeMatch = !mediaTypeFilter || mediaType === mediaTypeFilter;
       
       return searchMatch && platformMatch && mediaTypeMatch;
   });
}

/**
* Render sorted content table
*/
function renderSortedContentTable(sortedData) {
   const tableContent = document.getElementById('contentTableContent');
   if (!tableContent) return;
   
   tableContent.innerHTML = '';
   
   if (sortedData.length === 0) {
       tableContent.innerHTML = `
           <div class="content-table-row">
               <div style="grid-column: 1/-1; text-align: center; padding: 2rem; color: #64748b;">
                   <h3>No content data found</h3>
               </div>
           </div>
       `;
       return;
   }
   
   sortedData.forEach(content => {
       const row = createContentTableRowDetailed(content);
       tableContent.appendChild(row);
   });
}

/**
* Create detailed content table row with separate columns for each metric
*/
function createContentTableRowDetailed(content) {
   const row = document.createElement('div');
   row.className = 'content-table-row';
   row.dataset.contentId = content.id; // Add unique ID
   row.dataset.title = content.title.toLowerCase();
   row.dataset.platform = content.platform.toLowerCase();
   row.dataset.influencer = content.influencer.toLowerCase();
   row.dataset.mediaType = content.mediaType || '';
   
   // Create clickable title if video link exists
   let titleHTML;
   if (content.videoLink && content.videoLink.trim() !== '') {
       titleHTML = `
           <a href="${content.videoLink}" target="_blank" class="table-content-title-link">
               ${escapeHtml(content.title)}
               <span class="external-link-icon">🔗</span>
           </a>
       `;
   } else {
       titleHTML = `<div class="table-content-title">${escapeHtml(content.title)}</div>`;
   }
   
   // Create platform badge with media type
   let platformBadgeText = content.platform;
   if (content.mediaType) {
       if (content.mediaType === 'post') {
           platformBadgeText += ' Post';
       } else if (content.mediaType === 'reel') {
           platformBadgeText += ' Reel';
       } else if (content.mediaType === 'story') {
           platformBadgeText += ' Stories';
       } else if (content.mediaType === 'integration') {
           platformBadgeText += ' Integration';
       } else if (content.mediaType === 'short') {
           platformBadgeText += ' Shorts';
       } else if (content.mediaType === 'video') {
           platformBadgeText += ' Video';
       }
   }
   
   // Add bonus label if applicable
   const bonusLabel = content.bonus ? '<span class="bonus-label">Bonus</span>' : '';
   
   // Helper function to format metric values
   const formatMetric = (value) => {
       return value > 0 ? formatNumber(value) : '-';
   };
   
   row.innerHTML = `
       ${titleHTML}
       <div class="table-content-creator">${escapeHtml(content.influencer)}</div>
       <div>
           <span class="table-platform-badge platform-${content.platform.toLowerCase()}">
               ${platformBadgeText}
           </span>
           ${bonusLabel}
       </div>
       <div class="table-content-date">${content.postedDate}</div>
       <div class="table-metric">${formatMetric(content.metrics.views)}</div>
       <div class="table-metric">${formatMetric(content.metrics.likes)}</div>
       <div class="table-metric">${formatMetric(content.metrics.comments)}</div>
       <div class="table-metric">${formatMetric(content.metrics.shares)}</div>
       <div class="table-metric">${formatMetric(content.metrics.saves)}</div>
       <div class="table-metric">${formatMetric(content.analytics.stickerTaps)}</div>
       <div class="table-metric">${formatMetric(content.analytics.linkClicks)}</div>
   `;
   
   return row;
}

/**
* Update content table view
*/
function updateContentTable() {
   // If we have sorted data, use that; otherwise use original data
   const dataToRender = sortedContentData.length > 0 ? sortedContentData : contentData;
   
   const tableContent = document.getElementById('contentTableContent');
   if (!tableContent) return;
   
   tableContent.innerHTML = '';
   
   if (dataToRender.length === 0) {
       tableContent.innerHTML = `
           <div class="content-table-row">
               <div style="grid-column: 1/-1; text-align: center; padding: 2rem; color: #64748b;">
                   <h3>No content data found</h3>
               </div>
           </div>
       `;
       return;
   }
   
   dataToRender.forEach(content => {
       const row = createContentTableRowDetailed(content);
       tableContent.appendChild(row);
   });
}

// =============================================================================
// VIEW SWITCHING FUNCTIONS
// =============================================================================

/**
* Switch to grid view
*/
function switchToGridView() {
   currentView = 'grid';
   
   // Update button states
   document.getElementById('gridViewBtn').classList.add('active');
   document.getElementById('tableViewBtn').classList.remove('active');
   
   // Show/hide views
   document.getElementById('influencersGrid').style.display = 'grid';
   document.getElementById('influencersTable').style.display = 'none';
}

/**
* Switch to table view
*/
function switchToTableView() {
   currentView = 'table';
   
   // Update button states
   document.getElementById('gridViewBtn').classList.remove('active');
   document.getElementById('tableViewBtn').classList.add('active');
   
   // Show/hide views
   document.getElementById('influencersGrid').style.display = 'none';
   document.getElementById('influencersTable').style.display = 'block';
   
   // Update table content
   updateInfluencersTable();
}

/**
* Switch to content grid view
*/
function switchToContentGridView() {
   currentContentView = 'grid';
   
   document.getElementById('contentGridViewBtn').classList.add('active');
   document.getElementById('contentTableViewBtn').classList.remove('active');
   
   document.getElementById('contentGrid').style.display = 'grid';
   document.getElementById('contentTable').style.display = 'none';
}

/**
* Switch to content table view
*/
function switchToContentTableView() {
   currentContentView = 'table';
   
   document.getElementById('contentGridViewBtn').classList.remove('active');
   document.getElementById('contentTableViewBtn').classList.add('active');
   
   document.getElementById('contentGrid').style.display = 'none';
   document.getElementById('contentTable').style.display = 'block';
}

// =============================================================================
// FILTER FUNCTIONS
// =============================================================================

/**
* Filter influencers based on search and platform selection
*/
function filterInfluencers() {
   const searchTerm = document.getElementById('searchInput').value.toLowerCase();
   const platformFilter = document.getElementById('platformFilter').value;
   
   // Get date range values
   const startDate = document.getElementById('influencersStartDate')?.value;
   const endDate = document.getElementById('influencersEndDate')?.value;
   
   // Get influencers who have content in the selected date range
   let validInfluencerProfileLinks = new Set();
   
   if (startDate && endDate) {
       // Filter content by date range
       contentData.forEach(content => {
           if (isDateInRange(content.postedDate, startDate, endDate)) {
               // Add this influencer's profile link to the valid set
               if (content.profileLink) {
                   validInfluencerProfileLinks.add(content.profileLink.toLowerCase().trim());
               }
           }
       });
   }
   
   if (currentView === 'grid') {
       // Filter grid view
       const cards = document.querySelectorAll('.influencer-card');
       
       cards.forEach(card => {
           const name = card.dataset.name;
           const platform = card.dataset.platform;
           const profileLink = card.dataset.profileLink;
           
           const searchMatch = !searchTerm || name.includes(searchTerm);
           const platformMatch = !platformFilter || platform.toLowerCase() === platformFilter.toLowerCase();
           
           // Date range match - only filter if dates are set
           let dateMatch = true;
           if (startDate && endDate) {
               dateMatch = validInfluencerProfileLinks.has(profileLink.toLowerCase().trim());
           }
           
           if (searchMatch && platformMatch && dateMatch) {
               card.style.display = 'block';
           } else {
               card.style.display = 'none';
           }
       });
       
   } else {
       // Filter table view
       const rows = document.querySelectorAll('.table-row');
       
       rows.forEach(tableRow => {
           // Skip header row
           if (!tableRow.dataset.profileLink) return;
           
           const name = tableRow.dataset.name;
           const platform = tableRow.dataset.platform;
           const profileLink = tableRow.dataset.profileLink;
           
           const searchMatch = !searchTerm || name.includes(searchTerm);
           const platformMatch = !platformFilter || platform.toLowerCase() === platformFilter.toLowerCase();
           
           // Date range match - only filter if dates are set
           let dateMatch = true;
           if (startDate && endDate) {
               dateMatch = validInfluencerProfileLinks.has(profileLink.toLowerCase().trim());
           }
           
           if (searchMatch && platformMatch && dateMatch) {
               tableRow.style.display = '';
           } else {
               tableRow.style.display = 'none';
           }
       });
   }
   
   // Update metrics summary
   updateMetricsSummary();
}

/**
* Filter content based on search and platform selection
*/
function filterContent() {
   const searchTerm = document.getElementById('contentSearchInput').value.toLowerCase();
   const platformFilter = document.getElementById('contentPlatformFilter').value;
   const mediaTypeFilter = document.getElementById('contentMediaTypeFilter').value;
   
   // Get date range values
   const startDate = document.getElementById('contentStartDate')?.value;
   const endDate = document.getElementById('contentEndDate')?.value;
   
   // Clear sorted data when filtering
   sortedContentData = [];
   currentSortColumn = '';
   currentSortDirection = 'desc';
   
   // Clear sort indicators
   document.querySelectorAll('.sort-indicator').forEach(indicator => {
       indicator.textContent = '';
   });
   
   // Update the metrics summary - it will handle comparison logic
   updateContentMetricsSummary();
   
   // IMPORTANT: Rebuild the entire grid to show period sections
   if (currentContentView === 'grid') {
       updateContentGrid(); // THIS IS KEY - rebuilds the grid with period sections
   } else {
       // For table view, just filter normally
       const rows = document.querySelectorAll('.content-table-row');
       rows.forEach(row => {
           const contentId = row.dataset.contentId;
           const title = row.dataset.title;
           const platform = row.dataset.platform;
           const influencer = row.dataset.influencer;
           const mediaType = row.dataset.mediaType || '';
           
           // Use ID for lookup if available, fallback to title
           const contentItem = contentId 
               ? contentData.find(c => c.id == contentId)
               : contentData.find(c => c.title.toLowerCase() === title);
           const isBonus = contentItem ? contentItem.bonus === true : false;
           
           // Check date range ONLY if useDateFilterForCalculations is true
           const dateMatch = !useDateFilterForCalculations || !startDate || !endDate ? true : 
               isDateInRange(contentItem?.postedDate, startDate, endDate);
                      
           const searchMatch = !searchTerm || title.includes(searchTerm) || influencer.includes(searchTerm);
           
           let platformMatch = true;
           if (platformFilter === 'bonus') {
               platformMatch = isBonus === true;
           } else if (platformFilter) {
               platformMatch = platform.toLowerCase() === platformFilter.toLowerCase();
           }
           
           const mediaTypeMatch = !mediaTypeFilter || mediaType === mediaTypeFilter;
           
           row.style.display = (searchMatch && platformMatch && mediaTypeMatch && dateMatch) ? '' : 'none';
       });
   }
}

// Check if comparison mode is active
console.log('Comparison mode:', isComparisonMode);
console.log('Period 1 data:', period1Data);
console.log('Period 2 data:', period2Data);


// =============================================================================
// PAID MEDIA TAB FUNCTIONS
// =============================================================================

// Global variables for Paid Media
let paidMediaData = {
    tiktok: {},
    instagram: {},
    youtube: {},
    findings: '',
    recommendations: '',
    totalPaidClicks: 0
};

let currentPaidMediaFilter = null;

/**
 * Load Paid Media data from Google Sheets Paid tab
 */
async function loadPaidMediaData() {
    try {
        console.log('Loading Paid Media data from sheet:', currentCampaignId);
        
        // Check if the Paid tab exists
        const paidTabExists = await checkSheetTabExists(currentCampaignId, 'Paid');
        
        if (!paidTabExists) {
            console.log('Paid tab not found - hiding Paid Media section');
            hidePaidMediaTab();
            return;
        }
        
        console.log('Paid tab found - loading paid media data');
        showPaidMediaTab();
        
        // Fetch data from the Paid sheet - columns A to BH (60 columns)
        const data = await fetchSheetData(currentCampaignId, 'Paid', 'A1:BH3');
        
        if (data && data.length >= 3) {
            // Process the data
            processPaidMediaData(data);
            
            // Display the data
            updatePaidMediaDisplay();
            
            // UPDATE OVERVIEW TO INCLUDE PAID CLICKS
            updateOverviewDisplay();
            
        } else {
            console.warn('No paid media data found or insufficient rows in Paid sheet');
            hidePaidMediaTab();
        }
        
    } catch (error) {
        console.error('Error loading paid media data:', error);
        hidePaidMediaTab();
    }
}

/**
 * Process Paid Media data from the sheet
 */
function processPaidMediaData(data) {
    const headers = data[1] || [];  // Row 2 (index 1)
    const values = data[2] || [];   // Row 3 (index 2)
    
    console.log('Processing paid media data');
    
    // Reset total paid clicks
    paidMediaData.totalPaidClicks = 0;
    
    // TikTok data: columns A-T (indices 0-19)
    for (let i = 0; i <= 19; i++) {
        if (headers[i] && values[i] !== undefined && values[i] !== '') {
            const key = headers[i].toString().trim();
            paidMediaData.tiktok[key] = values[i];
            
            // Sum any "Clicks" metric
            if (key.toLowerCase().includes('click')) {
                const clickValue = parseFloat(values[i].toString().replace(/[^0-9.\-]/g, '')) || 0;
                paidMediaData.totalPaidClicks += clickValue;
            }
        }
    }
    
    // Instagram data: columns U-AN (indices 20-39)
    for (let i = 20; i <= 39; i++) {
        if (headers[i] && values[i] !== undefined && values[i] !== '') {
            const key = headers[i].toString().trim();
            paidMediaData.instagram[key] = values[i];
            
            // Sum any "Clicks" metric
            if (key.toLowerCase().includes('click')) {
                const clickValue = parseFloat(values[i].toString().replace(/[^0-9.\-]/g, '')) || 0;
                paidMediaData.totalPaidClicks += clickValue;
            }
        }
    }
    
    // YouTube data: columns AO-BH (indices 40-59)
    for (let i = 40; i <= 59; i++) {
        if (headers[i] && values[i] !== undefined && values[i] !== '') {
            const key = headers[i].toString().trim();
            paidMediaData.youtube[key] = values[i];
            
            // Sum any "Clicks" metric
            if (key.toLowerCase().includes('click')) {
                const clickValue = parseFloat(values[i].toString().replace(/[^0-9.\-]/g, '')) || 0;
                paidMediaData.totalPaidClicks += clickValue;
            }
        }
    }
    
    // Set initial filter to first available platform with data
    if (!currentPaidMediaFilter) {
        if (Object.keys(paidMediaData.tiktok).length > 0) {
            currentPaidMediaFilter = 'tiktok';
        } else if (Object.keys(paidMediaData.instagram).length > 0) {
            currentPaidMediaFilter = 'instagram';
        } else if (Object.keys(paidMediaData.youtube).length > 0) {
            currentPaidMediaFilter = 'youtube';
        }
    }
    
    console.log('Paid media data processed:', paidMediaData);
    console.log('Total paid clicks:', paidMediaData.totalPaidClicks);
    console.log('Initial filter set to:', currentPaidMediaFilter);
}

/**
 * Update Paid Media display based on current filter
 */
function updatePaidMediaDisplay() {
    // First, update the filter buttons to show only platforms with data
    updatePaidMediaFilterButtons();
    
    const container = document.getElementById('paidMediaMetrics');
    if (!container) return;
    
    container.innerHTML = '';
    
    // Use specific platform data only
    const dataToDisplay = paidMediaData[currentPaidMediaFilter] || {};
    
    // Check if there's any data to display
    if (Object.keys(dataToDisplay).length === 0) {
        container.innerHTML = `
            <div class="empty-state" style="grid-column: 1/-1;">
                <h3>No paid media data available</h3>
                <p>No data found for the selected platform</p>
            </div>
        `;
        // Still update analysis cards even if no metrics
        updatePaidMediaAnalysisCards();
        return;
    }
    
    // Create cards for each metric, but only add non-null cards
    let hasVisibleCards = false;
    Object.entries(dataToDisplay).forEach(([metric, value]) => {
        const card = createPaidMediaCard(metric, value);
        if (card !== null) {
            container.appendChild(card);
            hasVisibleCards = true;
        }
    });
    
    // If no cards were added (all were zero), show empty state
    if (!hasVisibleCards) {
        container.innerHTML = `
            <div class="empty-state" style="grid-column: 1/-1;">
                <h3>No paid media data available</h3>
                <p>All metrics are zero for the selected platform</p>
            </div>
        `;
    }
    
    // Update the analysis cards (findings and recommendations)
    updatePaidMediaAnalysisCards();
}

/**
 * Update Paid Media analysis cards (Findings and Recommendations)
 */
function updatePaidMediaAnalysisCards() {
    const findingsElement = document.getElementById('paidMediaFindings');
    const recommendationsElement = document.getElementById('paidMediaRecommendations');
    
    if (findingsElement) {
        const findings = paidMediaData.findings || 'No findings available.';
        findingsElement.textContent = findings;
    }
    
    if (recommendationsElement) {
        const recommendations = paidMediaData.recommendations || 'No recommendations available.';
        recommendationsElement.textContent = recommendations;
    }
    
    console.log('Updated Paid Media analysis cards');
}

/**
 * Update paid media filter buttons to show only platforms with data
 */
function updatePaidMediaFilterButtons() {
    const filtersContainer = document.getElementById('paidMediaFilters');
    if (!filtersContainer) return;
    
    filtersContainer.innerHTML = '';
    
    // Check which platforms have data
    const platformsWithData = [];
    
    if (Object.keys(paidMediaData.tiktok).length > 0) {
        platformsWithData.push({ id: 'tiktok', label: 'TikTok' });
    }
    if (Object.keys(paidMediaData.instagram).length > 0) {
        platformsWithData.push({ id: 'instagram', label: 'Instagram' });
    }
    if (Object.keys(paidMediaData.youtube).length > 0) {
        platformsWithData.push({ id: 'youtube', label: 'YouTube' });
    }
    
    // Create buttons for platforms with data
    platformsWithData.forEach((platform, index) => {
        const button = document.createElement('button');
        button.className = 'platform-tab';
        button.textContent = platform.label;
        button.onclick = () => filterPaidMedia(platform.id);
        
        // Set first button as active if current filter matches or is not set
        if (platform.id === currentPaidMediaFilter) {
            button.classList.add('active');
        }
        
        filtersContainer.appendChild(button);
    });
}

/**
/**
 * Create a paid media metric card (returns null if value is 0)
 */
function createPaidMediaCard(metric, value) {
    // Parse the value to check if it's zero
    const numericValue = parseFloat(value.toString().replace(/[^0-9.\-]/g, '')) || 0;
    
    // Return null if value is 0
    if (numericValue === 0) {
        return null;
    }
    
    const card = document.createElement('div');
    card.className = 'paid-media-stat-box';
    
    // Choose an appropriate icon based on metric name
    const icon = getPaidMediaIcon(metric);
    
    // Format the label
    const label = formatPaidMediaLabel(metric);
    
    // Format the value with commas
    const formattedValue = formatPaidMediaValue(value);
    
    card.innerHTML = `
        <div class="paid-media-stat-icon">${icon}</div>
        <div class="paid-media-stat-label">${label}</div>
        <div class="paid-media-stat-value">${formattedValue}</div>
    `;
    
    return card;
}

/**
 * Get icon for paid media metric
 */
function getPaidMediaIcon(metric) {
    const metricLower = metric.toLowerCase();
    
    if (metricLower.includes('spend')) return '💰';
    if (metricLower.includes('cpc')) return '💲';
    if (metricLower.includes('impression')) return '👁';
    if (metricLower.includes('click')) return '🖱️';
    if (metricLower.includes('ctr')) return '📊';
    if (metricLower.includes('engagement')) return '💬';
    if (metricLower.includes('er')) return '📈';
    if (metricLower.includes('conversion')) return '🎯';
    if (metricLower.includes('cpa')) return '💸';
    if (metricLower.includes('cvr')) return '🔄';
    if (metricLower.includes('view')) return '👀';
    if (metricLower.includes('reach')) return '📡';
    if (metricLower.includes('frequency')) return '🔁';
    if (metricLower.includes('roi')) return '📊';
    if (metricLower.includes('roas')) return '💹';
    
    return '📊'; // Default icon
}

/**
 * Format paid media label for display
 */
function formatPaidMediaLabel(metric) {
    // Return the metric as-is since it's from the header
    return metric;
}

/**
 * Clear date range filter
 */
function clearDateFilter() {
    document.getElementById('contentStartDate').value = '';
    document.getElementById('contentEndDate').value = '';
    filterContent();
}

/**
 * Parse date string from various formats to Date object
 */
function parseContentDate(dateString) {
    if (!dateString) return null;
    
    // Clean the date string
    const cleaned = dateString.toString().trim();
    if (!cleaned) return null;
    
    // Try parsing common formats
    // Format 1: M/D/YYYY (e.g., 7/13/2025, 04/30/2025)
    const slashMatch = cleaned.match(/^(\d{1,2})\/(\d{1,2})\/(\d{4})$/);
    if (slashMatch) {
        const month = parseInt(slashMatch[1]) - 1; // Month is 0-indexed
        const day = parseInt(slashMatch[2]);
        const year = parseInt(slashMatch[3]);
        return new Date(year, month, day);
    }
    
    // Format 2: YYYY-MM-DD (e.g., 2025-07-13)
    const dashMatch = cleaned.match(/^(\d{4})-(\d{2})-(\d{2})$/);
    if (dashMatch) {
        const year = parseInt(dashMatch[1]);
        const month = parseInt(dashMatch[2]) - 1; // Month is 0-indexed
        const day = parseInt(dashMatch[3]);
        return new Date(year, month, day);
    }
    
    // Try standard Date parsing as fallback
    const date = new Date(cleaned);
    
    // Check if valid date
    if (isNaN(date.getTime())) {
        console.warn('Could not parse date:', dateString);
        return null;
    }
    
    return date;
}

/**
 * Check if content date is within selected range
 */
function isDateInRange(contentDate, startDate, endDate) {
    if (!contentDate) return false;
    
    const date = parseContentDate(contentDate);
    if (!date) return false;
    
    // Normalize to start of day for comparison
    const contentDateNormalized = new Date(date.getFullYear(), date.getMonth(), date.getDate());
    
    if (startDate) {
        const start = new Date(startDate);
        const startNormalized = new Date(start.getFullYear(), start.getMonth(), start.getDate());
        
        // Content date must be >= start date
        if (contentDateNormalized < startNormalized) {
            return false;
        }
    }
    
    if (endDate) {
        const end = new Date(endDate);
        const endNormalized = new Date(end.getFullYear(), end.getMonth(), end.getDate());
        
        // Content date must be <= end date
        if (contentDateNormalized > endNormalized) {
            return false;
        }
    }
    
    return true;
}

/**
 * Format paid media numbers
 */
function formatPaidMediaNumber(num) {
    if (typeof num !== 'number' || isNaN(num)) return '0';
    
    // For very large numbers
    if (num >= 1000000) {
        return (num / 1000000).toFixed(1) + 'M';
    }
    if (num >= 1000) {
        return (num / 1000).toFixed(1) + 'K';
    }
    
    // For decimal numbers
    if (num < 1 && num > 0) {
        return num.toFixed(2);
    }
    
    // For regular numbers
    return Math.round(num).toLocaleString();
}

/**
 * Filter paid media by platform
 */
function filterPaidMedia(platform) {
    currentPaidMediaFilter = platform;
    
    // Update active tab
    const buttons = document.querySelectorAll('#paidMediaFilters .platform-tab');
    buttons.forEach(btn => {
        btn.classList.remove('active');
        if (btn.textContent.toLowerCase().includes(platform)) {
            btn.classList.add('active');
        }
    });
    
    // Update display
    updatePaidMediaDisplay();
}

/**
 * Hide the Paid Media tab
 */
function hidePaidMediaTab() {
    const paidMediaTab = document.getElementById('paidMedia');
    if (paidMediaTab) {
        paidMediaTab.style.display = 'none';
        paidMediaTab.classList.remove('active');
    }
    
    const paidMediaNavItem = document.querySelector('.nav-item[onclick="showTab(\'paidMedia\')"]');
    if (paidMediaNavItem) {
        paidMediaNavItem.style.display = 'none';
    }
    
    console.log('Paid Media tab hidden due to no content');
}

/**
 * Show the Paid Media tab
 */
function showPaidMediaTab() {
    const paidMediaTab = document.getElementById('paidMedia');
    if (paidMediaTab) {
        paidMediaTab.style.display = '';
    }
    
    const paidMediaNavItem = document.querySelector('.nav-item[onclick="showTab(\'paidMedia\')"]');
    if (paidMediaNavItem) {
        paidMediaNavItem.style.display = '';
    }
    
    console.log('Paid Media tab shown with content');
}

/**
* Show specific tab
*/
function showTab(tabName) {
    // Hide all tab content
    document.querySelectorAll('.tab-content').forEach(tab => {
        tab.classList.remove('active');
        tab.style.display = 'none';
    });
    
    // Remove active class from all nav items
    document.querySelectorAll('.nav-item').forEach(item => {
        item.classList.remove('active');
    });
    
    // Show selected tab
    const selectedTab = document.getElementById(tabName);
    if (selectedTab) {
        selectedTab.classList.add('active');
        selectedTab.style.display = 'block';
    }
    
    // Add active class to clicked nav item
    if (event && event.target) {
        event.target.classList.add('active');
    }
    
    // Update displays when tabs are shown
    if (tabName === 'overview') {
        updateOverviewDisplay();
    } else if (tabName === 'influencers') {
        // Ensure dates are set when showing Influencers tab
        if (contentData && contentData.length > 0) {
            const startInput = document.getElementById('influencersStartDate');
            const endInput = document.getElementById('influencersEndDate');
            
            // Only set dates if empty
            if (startInput && endInput && !startInput.value && !endInput.value) {
                setDefaultInfluencersDateRange();
            }
        }
    } else if (tabName === 'results') {
        // Ensure dates are set when showing Results tab
        if (contentData && contentData.length > 0) {
            const startInput = document.getElementById('resultsStartDate');
            const endInput = document.getElementById('resultsEndDate');
            
            // Only set dates if empty
            if (startInput && endInput && !startInput.value && !endInput.value) {
                useDateFilterForResults = false; // Make sure flag is false
                setDefaultResultsDateRange();
            }
        }
        
        renderPlatformFilters();
        renderSubFilters(currentPlatform);
        updateResultsDisplay();
    } else if (tabName === 'content') {
        // Ensure metrics are updated when showing content tab
        if (contentData && contentData.length > 0) {
            const startInput = document.getElementById('contentStartDate');
            const endInput = document.getElementById('contentEndDate');
            
            // Only set calendar dates if empty (doesn't trigger filtering)
            if (!startInput.value || !endInput.value) {
                setDefaultDateRange();
            } 
                useDateFilterForCalculations = true;
            
            // Update metrics - will show all content if no active date filter
            updateContentMetricsSummary();
        }
    } else if (tabName === 'analysis') {
        // NEW: Ensure date ranges are loaded and initialized when showing Analysis tab
        if (contentData && contentData.length > 0) {
            // Check if date ranges are already loaded
            const rangeButtons = document.querySelectorAll('.date-range-button');
            if (rangeButtons.length === 0) {
                // Date ranges not loaded yet, load them now
                loadAnalysisDateRanges();
            } else {
                // Date ranges already loaded, just ensure proper filter is applied
                if (currentAnalysisDateRange === null) {
                    // No filter selected, select "All Periods" by default
                    filterAnalysisByDateRange(null);
                }
            }
        }
    }
}

/**
* Refresh the dashboard data
*/
function refreshData() {
   loadInfluencerData();
   loadContentData();
   loadWeeklyData();
}

// =============================================================================
// UTILITY FUNCTIONS
// =============================================================================

/**
* Format numbers for display (e.g., 1000 -> 1K, 1000000 -> 1M)
*/
function formatNumber(num) {
   if (typeof num !== 'number' || isNaN(num)) return '0';
   if (num >= 1000000) return (num / 1000000).toFixed(1) + 'M';
   if (num >= 1000) return (num / 1000).toFixed(1) + 'K';
   return num.toLocaleString();
}

/**
* Parse metric numbers that might have commas
*/
function parseMetricNumber(value) {
   if (!value) return 0;
   const cleaned = value.toString().replace(/,/g, '');
   return parseInt(cleaned) || 0;
}

/**
* Clean and trim string input
*/
function cleanString(str) {
   if (!str) return '';
   return str.toString().trim();
}

/**
* Format handle to include @ if missing
*/
function formatHandle(handle) {
   if (!handle) return '';
   return handle.startsWith('@') ? handle : `@${handle}`;
}

/**
* Format platform names to proper case
*/
function formatPlatformName(platform) {
   const platformMap = {
       'tiktok': 'TikTok',
       'instagram': 'Instagram', 
       'youtube': 'YouTube',
       'linkedin': 'LinkedIn',
       'facebook': 'Facebook',
       'twitter': 'Twitter'
   };
   
   const normalized = platform ? platform.toLowerCase() : '';
   return platformMap[normalized] || platform;
}

/**
* Get initials from a name
*/
function getInitials(name) {
   if (!name) return '?';
   return name.split(' ')
              .map(word => word.charAt(0))
              .join('')
              .toUpperCase()
              .substring(0, 2);
}

/**
* Escape HTML to prevent XSS
*/
function escapeHtml(text) {
   const div = document.createElement('div');
   div.textContent = text;
   return div.innerHTML;
}

// =============================================================================
// UI STATE MANAGEMENT
// =============================================================================

/**
* Show loading state
*/
function showLoading() {
   document.getElementById('loading').style.display = 'flex';
   document.getElementById('dashboard').style.display = 'none';
   document.getElementById('error').style.display = 'none';
}

/**
* Show dashboard state
*/
function showDashboard() {
   document.getElementById('loading').style.display = 'none';
   document.getElementById('dashboard').style.display = 'block';
   document.getElementById('error').style.display = 'none';
}

/**
* Show error state
*/
function showError(message) {
   document.getElementById('loading').style.display = 'none';
   document.getElementById('dashboard').style.display = 'none';
   document.getElementById('error').style.display = 'flex';
   document.getElementById('errorMessage').textContent = message;
}

/**
* Validate configuration
*/
function validateConfig() {
   return typeof CONFIG !== 'undefined' && CONFIG.API_BASE_URL && CONFIG.API_SECRET_KEY;
}

/**
* Load content data from Google Sheets Report tab
*/
async function loadContentData() {
    try {
        const data = await fetchSheetData(
            currentCampaignId, 
            'Report',
            'A1:AD1000'
        );
        
        const processedData = processReportData(data);
        contentData = processedData.content;
        window.contentData = contentData;
        
        console.log('Content data loaded:', contentData.length, 'items');
        
        updateContentDisplay();
        
        // Set default date range for Content tab AND calculate metrics
        setTimeout(() => {
            setDefaultDateRange();
            // Metrics are now updated inside setDefaultDateRange()
            console.log('Content tab initialized with metrics');
        }, 100);
        
        // Set default date range for Results tab
        setTimeout(() => {
            setDefaultResultsDateRange();
        }, 100);
        
        // Set default date range for Influencers tab
        setTimeout(() => {
            setDefaultInfluencersDateRange();
        }, 100);

    } catch (error) {
        console.error('Error loading content data:', error);
        contentData = [];
        updateContentDisplay();
    }
}

/**
 * Wait for DOM to be ready then initialize
 */
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeDashboard);
} else {
    // DOM is already ready
    initializeDashboard();
}
