// Main application logic for MVP Influencer Dashboard

// Global variables
let currentCampaignId = '';
let influencersData = [];
let contentData = [];
let currentView = 'grid'; // 'grid' or 'table'
let currentContentView = 'grid'; // 'grid' or 'table'
let currentResultsFilter = 'all'; // 'all', 'tiktok', 'instagram', 'youtube'
let currentSortColumn = '';
let currentSortDirection = 'desc'; // 'asc' or 'desc'
let sortedContentData = [];
let currentContentSortBy = 'default'; // 'default', 'views', 'likes', etc.
let currentContentSortDirection = 'desc'; // 'asc' or 'desc'
let weeklyData = [];
let currentChartType = 'line';
let currentMetricFilter = 'all';
let charts = {};
let currentPlatform = 'all';      // top-level: all | tiktok | instagram | youtube | linkedin | facebook | twitter | bonus
let currentSubfilter = null;      // second-level: instagram_reel, instagram_stories, etc.


// Metric configurations for charts
const metricConfigs = {
    'Likes': { color: '#ff6b8a', icon: '❤️', category: 'engagement' },
    'Comments': { color: '#4ecdc4', icon: '💬', category: 'engagement' },
    'Views': { color: '#45b7d1', icon: '👁', category: 'reach' },
    'Shares': { color: '#96ceb4', icon: '📤', category: 'engagement' },
    'Saves': { color: '#feca57', icon: '🔖', category: 'actions' },
    'Link Clicks': { color: '#ff9ff3', icon: '🔗', category: 'actions' },
    'Reach': { color: '#54a0ff', icon: '📊', category: 'reach' },
    'Impressions': { color: '#5f27cd', icon: '👀', category: 'reach' }
};

// ---- Filters config ----
const PLATFORM_LABELS = {
  all: 'All',
  tiktok: 'TikTok',
  instagram: 'Instagram',
  youtube: 'YouTube',
  linkedin: 'LinkedIn',
  facebook: 'Facebook',
  twitter: 'Twitter',
  bonus: 'Bonus'
};

// Map mediaType -> subfilter id/label per platform (IDs MATCH updateResultsDisplay)
const MEDIA_SUBFILTER_MAP = {
  instagram: {
    post:  { id: 'instagram_posts',   label: 'Posts' },
    reel:  { id: 'instagram_reel',    label: 'Reels' },
    story: { id: 'instagram_stories', label: 'Stories' }
  },
  youtube: {
    short:       { id: 'youtube_shorts',       label: 'Shorts' },
    integration: { id: 'youtube_integrations', label: 'Integrations' }
  },
  linkedin: {
    post:  { id: 'linkedin_post',  label: 'Post' },
    video: { id: 'linkedin_video', label: 'Video' }
  },
  facebook: {
    post:  { id: 'facebook_post',  label: 'Post' },
    reel:  { id: 'facebook_reel',  label: 'Reel' }
  },
  twitter: {
    post:  { id: 'twitter_post',  label: 'Post' },
    video: { id: 'twitter_video', label: 'Video' }
  },
  tiktok: {} // no sub-types in the sheet for now
};


// Build the platform list from the actual data (+All, +Bonus if present)
function getAvailablePlatformsFromContent() {
  const set = new Set();
  let hasBonus = false;

  (contentData || []).forEach(c => {
    const p = (c.platform || '').toLowerCase();
    if (p) set.add(p);
    if (c.bonus === true) hasBonus = true;
  });

  // desired order when present
  const order = ['tiktok','instagram','youtube','linkedin','facebook','twitter'];
  const list = ['all', ...order.filter(p => set.has(p))];
  if (hasBonus) list.push('bonus');
  return list;
}

function renderPlatformFilters() {
  const wrap = document.getElementById('platformFilters');
  if (!wrap) return;
  wrap.innerHTML = '';

  const platforms = getAvailablePlatformsFromContent();

  platforms.forEach(p => {
    const btn = document.createElement('button');
    btn.className = 'platform-tab' + (currentPlatform === p && !currentSubfilter ? ' active' : '');
    btn.textContent = PLATFORM_LABELS[p] || p;
    btn.addEventListener('click', () => {
      currentPlatform = p;
      currentSubfilter = null;
      currentResultsFilter = p;   // top-level filters by platform (or All/Bonus)

      renderPlatformFilters();
      renderSubFilters(p);
      updateResultsDisplay();
    });
    wrap.appendChild(btn);
  });
}

function renderSubFilters(platform) {
  const container = document.getElementById('subFilters');
  if (!container) return;

  // Hide for All/Bonus or when no map exists
  if (platform === 'all' || platform === 'bonus' || !MEDIA_SUBFILTER_MAP[platform]) {
    container.style.display = 'none';
    container.innerHTML = '';
    return;
  }

  // Determine which media types actually exist for this platform
  const presentTypes = new Set();
  (contentData || []).forEach(c => {
    if ((c.platform || '').toLowerCase() === platform && c.mediaType) {
      presentTypes.add(c.mediaType); // values like 'post','reel','story','short','integration'
    }
  });

  // Build the buttons only for present media types
  const map = MEDIA_SUBFILTER_MAP[platform];
  const availableSubs = Object.keys(map)
    .filter(type => presentTypes.has(type))
    .map(type => map[type]);

  if (!availableSubs.length) {
    container.style.display = 'none';
    container.innerHTML = '';
    return;
  }

  container.style.display = 'flex';
  container.innerHTML = '';

  availableSubs.forEach(sf => {
    const btn = document.createElement('button');
    btn.className = 'subfilter-tab' + (currentSubfilter === sf.id ? ' active' : '');
    btn.textContent = sf.label;
    btn.addEventListener('click', () => {
      currentSubfilter = sf.id;
      currentResultsFilter = sf.id;   // sub-filter id is what updateResultsDisplay() expects
      renderPlatformFilters();
      renderSubFilters(platform);
      updateResultsDisplay();
    });
    container.appendChild(btn);
  });
}


// Sample weekly data structure - replace with actual data loading
const sampleWeeklyData = [
    { 
        metric: 'Likes', 
        week1: 57, week2: 74, week3: 96, week4: 125, week5: 150,
        week6: 175, week7: 200, week8: 225, week9: 250, week10: 275
    },
    { 
        metric: 'Comments', 
        week1: 96, week2: 125, week3: 162, week4: 211, week5: 250,
        week6: 290, week7: 330, week8: 370, week9: 410, week10: 450
    },
    { 
        metric: 'Views', 
        week1: 45336, week2: 58937, week3: 76618, week4: 99603, week5: 120000,
        week6: 145000, week7: 170000, week8: 195000, week9: 220000, week10: 245000
    },
    { 
        metric: 'Shares', 
        week1: 69, week2: 90, week3: 117, week4: 152, week5: 180,
        week6: 210, week7: 240, week8: 270, week9: 300, week10: 330
    },
    { 
        metric: 'Saves', 
        week1: 27, week2: 35, week3: 46, week4: 59, week5: 72,
        week6: 85, week7: 98, week8: 111, week9: 124, week10: 137
    },
    { 
        metric: 'Link Clicks', 
        week1: 987, week2: 1283, week3: 1668, week4: 2168, week5: 2600,
        week6: 3100, week7: 3600, week8: 4100, week9: 4600, week10: 5100
    },
    { 
        metric: 'Reach', 
        week1: 18774, week2: 24406, week3: 31728, week4: 41246, week5: 50000,
        week6: 60000, week7: 70000, week8: 80000, week9: 90000, week10: 100000
    },
    { 
        metric: 'Impressions', 
        week1: 69349, week2: 90154, week3: 117200, week4: 152360, week5: 185000,
        week6: 220000, week7: 255000, week8: 290000, week9: 325000, week10: 360000
    }
];

/**
 * Get platform icon
 */
function getPlatformIcon(platform) {
    const icons = {
        'instagram': '📸',
        'tiktok': '🎵',
        'youtube': '📺'
    };
    return icons[platform.toLowerCase()] || '📱';
}

/**
 * Get available platforms based on current data
 * Returns an array of platform names that have at least 1 post
 */
function getAvailablePlatforms() {
    const platformsWithContent = new Set();
    
    // Check content data for platforms with posts
    contentData.forEach(content => {
        if (content.platform) {
            platformsWithContent.add(content.platform);
        }
    });
    
    // Convert to array and sort alphabetically
    return Array.from(platformsWithContent).sort();
}

/**
 * Get available platforms from influencer data
 * Returns platforms that have at least 1 influencer
 */
function getAvailablePlatformsFromInfluencers() {
    const platformsWithInfluencers = new Set();
    
    influencersData.forEach(influencer => {
        if (influencer.socialMedia) {
            platformsWithInfluencers.add(influencer.socialMedia);
        }
    });
    
    return Array.from(platformsWithInfluencers).sort();
}

/**
 * Update influencers platform filter dropdown
 */
function updateInfluencersPlatformFilter() {
    const platformFilter = document.getElementById('platformFilter');
    if (!platformFilter) return;
    
    const availablePlatforms = getAvailablePlatformsFromInfluencers();
    
    // Clear existing options
    platformFilter.innerHTML = '<option value="">All Platforms</option>';
    
    // Add only platforms that have influencers
    availablePlatforms.forEach(platform => {
        const option = document.createElement('option');
        option.value = platform;
        option.textContent = platform;
        platformFilter.appendChild(option);
    });
}

function toNumberFromCurrency(val) {
  // Strip everything except digits, dot, and minus
  return parseFloat((val ?? '').toString().replace(/[^0-9.\-]/g, '')) || 0;
}

function detectCurrencySymbol(items) {
  for (const it of items) {
    const s = (it?.cost ?? '').toString();
    const m = s.match(/[$£€]/);
    if (m) return m[0];
    const sym = it?.costMetrics?.currencySymbol;
    if (sym) return sym;
  }
  return '$';
}

/**
 * Update content platform filter dropdown
 */
function updateContentPlatformFilter() {
    const platformFilter = document.getElementById('contentPlatformFilter');
    if (!platformFilter) return;
    
    const availablePlatforms = getAvailablePlatforms();
    
    // Store current selection
    const currentSelection = platformFilter.value;
    
    // Clear existing options
    platformFilter.innerHTML = '<option value="">All Platforms</option>';
    
    // Add only platforms that have content
    availablePlatforms.forEach(platform => {
        const option = document.createElement('option');
        option.value = platform;
        option.textContent = platform;
        platformFilter.appendChild(option);
    });
    
    // Always add bonus option if there's any bonus content
    const hasBonusContent = contentData.some(content => content.bonus === true);
    if (hasBonusContent) {
        const bonusOption = document.createElement('option');
        bonusOption.value = 'bonus';
        bonusOption.textContent = 'Bonus Content Only';
        platformFilter.appendChild(bonusOption);
    }
    
    // Restore selection if it still exists
    if (currentSelection && Array.from(platformFilter.options).some(opt => opt.value === currentSelection)) {
        platformFilter.value = currentSelection;
    }
}

// =============================================================================
// WEEKLY DATA FUNCTIONS
// =============================================================================

/**
 * Load weekly data from Google Sheets Weekly tab
 */
async function loadWeeklyData() {
    try {
        console.log('Checking for Weekly tab in sheet:', currentCampaignId);
        
        // First check if the Weekly tab exists
        const weeklyTabExists = await checkSheetTabExists(currentCampaignId, 'Weekly');
        
        if (!weeklyTabExists) {
            console.log('Weekly tab not found - hiding Weekly Performance Trends section');
            hideWeeklySection();
            return;
        }
        
        console.log('Weekly tab found - loading weekly data');
        showWeeklySection(); // Make sure section is visible
        
        // Fetch data from the Weekly sheet - EXTENDED RANGE to K column for 10 weeks
        const data = await fetchSheetData(currentCampaignId, 'Weekly', 'A1:K10');
        
        if (data && data.length > 1) {
            // Process the weekly data with 10 weeks
            weeklyData = processWeeklyData(data);
            window.weeklyData = weeklyData;
        } else {
            // Use sample data as fallback
            console.warn('No weekly data found, using sample data');
            weeklyData = sampleWeeklyData;
        }
        
        // Create charts and update results display
        createCharts();
        updateWeeklyResultsDisplay();
        
    } catch (error) {
        console.error('Error loading weekly data:', error);
        // Hide the weekly section if there's an error
        hideWeeklySection();
    }
}

// Add function to hide the weekly section
function hideWeeklySection() {
    const weeklySection = document.querySelector('.weekly-results-section');
    if (weeklySection) {
        weeklySection.style.display = 'none';
    }
    
    // Also hide any related data in the results stats that come from weekly data
    // These will show 0 or default values instead
    const weeklyRelatedStats = ['resultsTotalViews', 'resultsTotalLikes', 'resultsTotalComments', 'resultsTotalShares', 'resultsTotalSaves'];
    weeklyRelatedStats.forEach(statId => {
        const element = document.getElementById(statId);
        if (element) {
            element.textContent = '0';
        }
    });
}

// Add function to show the weekly section (in case it was hidden)
function showWeeklySection() {
    const weeklySection = document.querySelector('.weekly-results-section');
    if (weeklySection) {
        weeklySection.style.display = 'block';
    }
}


/**
 * Process Weekly sheet data into chart format
 */
function processWeeklyData(data) {
    const processedData = [];
    
    // Skip header row, process each metric row
    for (let i = 1; i < data.length; i++) {
        const row = data[i];
        if (row[0]) { // Check if metric name exists
            const metricData = {
                metric: row[0] // Column A: Metric name
            };
            
            // Dynamically add week data based on what's available
            // Only add weeks that have actual data (not undefined or null)
            for (let weekNum = 1; weekNum <= 10; weekNum++) {
                const value = row[weekNum]; // Columns B-K for weeks 1-10
                if (value !== undefined && value !== null && value !== '') {
                    const stringValue = value.toString().replace(/,/g, '');
                    metricData[`week${weekNum}`] = parseInt(value.toString().replace(/,/g, '')) || 0;
                }
            }
            
            // Only add the metric if it has at least one week of data
            const hasData = Object.keys(metricData).some(key => key.startsWith('week'));
            if (hasData) {
                processedData.push(metricData);
            }
        }
    }
    
    console.log('Processed weekly data:', processedData);
    return processedData;
}


/**
 * Create and render charts
 */
function createCharts() {
    const chartsGrid = document.getElementById('chartsGrid');
    if (!chartsGrid) return;
    
    chartsGrid.innerHTML = '';

    const filteredData = getFilteredMetrics();

    // First, determine the maximum week number that has data across all metrics
    let maxWeekWithData = 0;
    filteredData.forEach(metricData => {
        for (let i = 10; i >= 1; i--) {
            const weekKey = `week${i}`;
            // Check if this week has any non-null data
            if (metricData[weekKey] !== undefined && metricData[weekKey] !== null) {
                maxWeekWithData = Math.max(maxWeekWithData, i);
                break;
            }
        }
    });

    // If no data found, use week 1 as minimum
    if (maxWeekWithData === 0) {
        maxWeekWithData = 1;
    }

    console.log(`Maximum week with data: Week ${maxWeekWithData}`);

    filteredData.forEach(metricData => {
        const chartContainer = createChartContainer(metricData.metric);
        chartsGrid.appendChild(chartContainer);
        
        const canvas = chartContainer.querySelector('canvas');
        const ctx = canvas.getContext('2d');
        
        // Destroy existing chart if it exists
        if (charts[metricData.metric]) {
            charts[metricData.metric].destroy();
        }

        // Build data arrays - only up to the maximum week with data
        const weekLabels = [];
        const weekDataPoints = [];
        
        // Only show weeks up to the maximum week that has data
        for (let i = 1; i <= maxWeekWithData; i++) {
            const weekKey = `week${i}`;
            weekLabels.push(`Week ${i}`);
            // Use 0 for undefined/null values to maintain consistent axis
            weekDataPoints.push(metricData[weekKey] || 0);
        }

        charts[metricData.metric] = new Chart(ctx, {
            type: currentChartType,
            data: {
                labels: weekLabels,
                datasets: [{
                    label: metricData.metric,
                    data: weekDataPoints,
                    backgroundColor: currentChartType === 'bar' 
                        ? metricConfigs[metricData.metric].color + '80'
                        : metricConfigs[metricData.metric].color + '20',
                    borderColor: metricConfigs[metricData.metric].color,
                    borderWidth: 2,
                    fill: currentChartType === 'line',
                    tension: 0.4,
                    pointBackgroundColor: metricConfigs[metricData.metric].color,
                    pointBorderColor: '#fff',
                    pointBorderWidth: 2,
                    pointRadius: 4,
                    pointHoverRadius: 6
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                // Show full number in tooltip with proper formatting
                                const value = context.parsed.y;
                                const formattedValue = value.toLocaleString();
                                return `${metricData.metric}: ${formattedValue}`;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                // Show full numbers without abbreviation for values under 10,000
                                if (value === 0) {
                                    return '0';
                                } else if (value < 10000) {
                                    // For any value under 10,000, show the complete number
                                    return Math.round(value).toLocaleString();
                                } else if (value < 1000000) {
                                    // For 10K to 999K, show as K
                                    return Math.round(value / 1000) + 'K';
                                } else {
                                    // For millions
                                    return (value / 1000000).toFixed(1) + 'M';
                                }
                            },
                            // Ensure we have enough tick marks to show proper values
                            autoSkip: true,
                            maxTicksLimit: 10,
                            precision: 0
                        }
                    },
                    x: {
                        ticks: {
                            // Adjust font size based on number of weeks
                            font: {
                                size: weekLabels.length > 6 ? 10 : 12
                            },
                            // Rotate labels if we have many weeks
                            maxRotation: weekLabels.length > 6 ? 45 : 0,
                            minRotation: weekLabels.length > 6 ? 45 : 0
                        }
                    }
                }
            }
        });
    });
}

/**
 * Create chart container element
 */
function createChartContainer(metric) {
    const container = document.createElement('div');
    container.className = 'chart-container';
    
    const config = metricConfigs[metric];
    container.innerHTML = `
        <div class="chart-title">
            ${config.icon} ${metric}
        </div>
        <div class="chart-canvas">
            <canvas></canvas>
        </div>
    `;
    
    return container;
}

/**
 * Get filtered metrics based on current filter
 */
function getFilteredMetrics() {
    switch (currentMetricFilter) {
        case 'engagement':
            return weeklyData.filter(item => metricConfigs[item.metric] && metricConfigs[item.metric].category === 'engagement');
        case 'reach':
            return weeklyData.filter(item => metricConfigs[item.metric] && metricConfigs[item.metric].category === 'reach');
        case 'actions':
            return weeklyData.filter(item => metricConfigs[item.metric] && metricConfigs[item.metric].category === 'actions');
        default:
            return weeklyData;
    }
}

/**
 * Create charts with a specific week range
 * @param {number} startWeek - Starting week number (1-10)
 * @param {number} endWeek - Ending week number (1-10)
 */
function createChartsWithRange(startWeek, endWeek) {
    const chartsGrid = document.getElementById('chartsGrid');
    if (!chartsGrid) return;
    
    // Clear existing charts
    chartsGrid.innerHTML = '';
    
    // Get filtered metrics based on current filter
    const filteredData = getFilteredMetrics();
    
    // Create a chart for each metric
    filteredData.forEach(metricData => {
        // Create the chart container
        const chartContainer = createChartContainer(metricData.metric);
        chartsGrid.appendChild(chartContainer);
        
        // Get canvas context
        const canvas = chartContainer.querySelector('canvas');
        const ctx = canvas.getContext('2d');
        
        // Destroy existing chart if it exists
        if (charts[metricData.metric]) {
            charts[metricData.metric].destroy();
            delete charts[metricData.metric];
        }
        
        // Build data for selected range only
        const weekLabels = [];
        const weekDataPoints = [];
        
        for (let i = startWeek; i <= endWeek; i++) {
            const weekKey = `week${i}`;
            if (metricData[weekKey] !== undefined) {
                weekLabels.push(`Week ${i}`);
                weekDataPoints.push(metricData[weekKey]);
            }
        }
        
        // Skip creating chart if no data points
        if (weekDataPoints.length === 0) {
            chartContainer.innerHTML = `
                <div class="chart-title">
                    ${metricConfigs[metricData.metric].icon} ${metricData.metric}
                </div>
                <div class="empty-state" style="padding: 2rem; text-align: center; color: #64748b;">
                    No data available for selected week range
                </div>
            `;
            return;
        }
        
        // Create chart with filtered data
        charts[metricData.metric] = new Chart(ctx, {
            type: currentChartType,
            data: {
                labels: weekLabels,
                datasets: [{
                    label: metricData.metric,
                    data: weekDataPoints,
                    backgroundColor: currentChartType === 'bar' 
                        ? metricConfigs[metricData.metric].color + '80'
                        : metricConfigs[metricData.metric].color + '20',
                    borderColor: metricConfigs[metricData.metric].color,
                    borderWidth: 2,
                    fill: currentChartType === 'line',
                    tension: 0.4,
                    pointBackgroundColor: metricConfigs[metricData.metric].color,
                    pointBorderColor: '#fff',
                    pointBorderWidth: 2,
                    pointRadius: 4,
                    pointHoverRadius: 6,
                    pointHoverBackgroundColor: metricConfigs[metricData.metric].color,
                    pointHoverBorderColor: '#fff',
                    pointHoverBorderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                interaction: {
                    mode: 'index',
                    intersect: false
                },
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        backgroundColor: 'rgba(0, 0, 0, 0.8)',
                        titleColor: '#fff',
                        bodyColor: '#fff',
                        padding: 12,
                        cornerRadius: 8,
                        displayColors: false,
                        callbacks: {
                            label: function(context) {
                                // Show full number in tooltip with proper formatting
                                const value = context.parsed.y;
                                const formattedValue = value.toLocaleString();
                                return `${metricData.metric}: ${formattedValue}`;
                            },
                            afterLabel: function(context) {
                                // Calculate and show growth if not the first week
                                if (context.dataIndex > 0) {
                                    const currentValue = context.parsed.y;
                                    const previousValue = context.dataset.data[context.dataIndex - 1];
                                    if (previousValue > 0) {
                                        const growth = ((currentValue - previousValue) / previousValue * 100).toFixed(1);
                                        const arrow = growth >= 0 ? '↑' : '↓';
                                        const color = growth >= 0 ? 'green' : 'red';
                                        return `Growth: ${arrow} ${Math.abs(growth)}%`;
                                    }
                                }
                                return null;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)',
                            drawBorder: false
                        },
                        ticks: {
                            padding: 10,
                            callback: function(value) {
                                // Format y-axis labels - FIXED to show 2,944 correctly
                                if (value >= 1000000) {
                                    // For millions, show as M
                                    return (value / 1000000).toFixed(1) + 'M';
                                } else if (value >= 100000) {
                                    // For 100K+, show as K without decimals
                                    return Math.round(value / 1000) + 'K';
                                } else if (value >= 10000) {
                                    // For 10K-99K, show as K with one decimal
                                    return (value / 1000).toFixed(1) + 'K';
                                } else {
                                    // For anything under 10,000, show the full number with comma separator
                                    // This ensures 2,944 shows as "2,944" not "2"
                                    return value.toLocaleString();
                                }
                            },
                            font: {
                                size: 11
                            },
                            color: '#64748b',
                            // Ensure we have enough tick marks
                            autoSkip: true,
                            maxTicksLimit: 8
                        }
                    },
                    x: {
                        grid: {
                            display: false,
                            drawBorder: false
                        },
                        ticks: {
                            padding: 10,
                            font: {
                                size: weekLabels.length > 6 ? 10 : 12
                            },
                            maxRotation: weekLabels.length > 6 ? 45 : 0,
                            minRotation: weekLabels.length > 6 ? 45 : 0,
                            color: '#64748b'
                        }
                    }
                },
                animation: {
                    duration: 750,
                    easing: 'easeInOutQuart'
                }
            }
        });
    });
    
    // Update summary stats for the selected range
    updateRangeSummaryStats(startWeek, endWeek);
}

/**
 * Update summary statistics for the selected week range
 * @param {number} startWeek - Starting week number
 * @param {number} endWeek - Ending week number
 */
function updateRangeSummaryStats(startWeek, endWeek) {
    // Calculate totals for the selected range
    const rangeTotals = {};
    let hasData = false;
    
    weeklyData.forEach(metricData => {
        let total = 0;
        let count = 0;
        
        for (let i = startWeek; i <= endWeek; i++) {
            const weekKey = `week${i}`;
            if (metricData[weekKey] !== undefined && metricData[weekKey] > 0) {
                total += metricData[weekKey];
                count++;
                hasData = true;
            }
        }
        
        rangeTotals[metricData.metric] = {
            total: total,
            average: count > 0 ? Math.round(total / count) : 0,
            weeks: count
        };
    });
    
    // Update display if there's a summary section
    const summarySection = document.querySelector('.weekly-range-summary');
    if (summarySection && hasData) {
        summarySection.innerHTML = `
            <div class="range-summary-title">
                Week ${startWeek} - Week ${endWeek} Summary
            </div>
            <div class="range-summary-stats">
                ${Object.entries(rangeTotals).map(([metric, data]) => {
                    if (data.total > 0) {
                        return `
                            <div class="range-stat">
                                <div class="range-stat-label">${metric}</div>
                                <div class="range-stat-value">
                                    ${data.total.toLocaleString()}
                                </div>
                                <div class="range-stat-avg">
                                    Avg: ${data.average.toLocaleString()}/week
                                </div>
                            </div>
                        `;
                    }
                    return '';
                }).join('')}
            </div>
        `;
        summarySection.style.display = 'block';
    } else if (summarySection) {
        summarySection.style.display = 'none';
    }
}

/**
 * Update week range and refresh charts
 */
function updateWeekRange() {
    const startWeek = parseInt(document.getElementById('weekRangeStart').value);
    const endWeek = parseInt(document.getElementById('weekRangeEnd').value);
    
    // Validate inputs
    if (isNaN(startWeek) || isNaN(endWeek)) {
        alert('Please select valid week numbers');
        return;
    }
    
    if (startWeek < 1 || startWeek > 10 || endWeek < 1 || endWeek > 10) {
        alert('Week numbers must be between 1 and 10');
        return;
    }
    
    if (startWeek > endWeek) {
        alert('Start week must be before or equal to end week');
        return;
    }
    
    // Show loading state
    const chartsGrid = document.getElementById('chartsGrid');
    if (chartsGrid) {
        chartsGrid.innerHTML = '<div class="loading-message">Updating charts...</div>';
    }
    
    // Update charts with new range
    setTimeout(() => {
        createChartsWithRange(startWeek, endWeek);
    }, 100);
}

/**
 * Update chart type
 */
function updateChartType() {
    const chartTypeSelect = document.getElementById('chartType');
    if (chartTypeSelect) {
        currentChartType = chartTypeSelect.value;
        createCharts();
    }
}

/**
 * Update visible metrics based on filter
 */
function updateVisibleMetrics() {
    const metricSelector = document.getElementById('metricSelector');
    if (metricSelector) {
        currentMetricFilter = metricSelector.value;
        createCharts();
    }
}

/**
 * Refresh weekly data
 */
function refreshWeeklyData() {
    loadWeeklyData();
}

/**
 * Update results display with weekly totals
 */
function updateWeeklyResultsDisplay() {
    // This function now only updates the weekly charts
    // It doesn't override the results display stats anymore
    console.log('Weekly charts updated with weekly data');
}

// =============================================================================
// CONTENT FUNCTIONS
// =============================================================================

function sortContentGrid() {
    const dropdown = document.getElementById('contentSortDropdown');
    if (!dropdown) return;
    
    const sortValue = dropdown.value;
    const [sortBy, direction] = sortValue.split('-');
    
    currentContentSortBy = sortBy;
    currentContentSortDirection = direction || 'desc';
    
    // Get current filtered content
    const grid = document.getElementById('contentGrid');
    if (!grid) return;
    
    const cards = Array.from(grid.querySelectorAll('.content-card'));
    
    // Filter visible cards only
    const visibleCards = cards.filter(card => card.style.display !== 'none');
    
    if (sortBy === 'default') {
        // Restore original order by re-rendering
        updateContentGrid();
        // Re-apply current filters
        filterContent();
        return;
    }
    
    // Sort visible cards
    visibleCards.sort((cardA, cardB) => {
        const contentA = getContentDataFromCard(cardA);
        const contentB = getContentDataFromCard(cardB);
        
        if (!contentA || !contentB) return 0;
        
        let valueA, valueB;
        
        switch (sortBy) {
            case 'views':
                valueA = contentA.metrics.views || 0;
                valueB = contentB.metrics.views || 0;
                break;
            case 'likes':
                valueA = contentA.metrics.likes || 0;
                valueB = contentB.metrics.likes || 0;
                break;
            case 'engagement':
                valueA = getContentER(contentA);
                valueB = getContentER(contentB);
                break;
            case 'date':
                valueA = new Date(contentA.postedDate || '1970-01-01');
                valueB = new Date(contentB.postedDate || '1970-01-01');
                break;
            default:
                return 0;
        }
        
        // Sort based on direction
        if (direction === 'asc') {
            return valueA < valueB ? -1 : valueA > valueB ? 1 : 0;
        } else {
            return valueB < valueA ? -1 : valueB > valueA ? 1 : 0;
        }
    });
    
    // Clear grid and re-append sorted cards
    grid.innerHTML = '';
    
    // First add the sort dropdown
    const sortContainer = document.createElement('div');
    sortContainer.innerHTML = getContentSortDropdownHTML();
    sortContainer.style.gridColumn = '1 / -1';
    grid.appendChild(sortContainer);
    
    // Add sorted visible cards first
    visibleCards.forEach(card => {
        grid.appendChild(card);
    });
    
    // Add back filtered out cards at the end (hidden)
    cards.forEach(card => {
        if (card.style.display === 'none') {
            grid.appendChild(card);
        }
    });
    
    // Restore dropdown value
    const newDropdown = document.getElementById('contentSortDropdown');
    if (newDropdown) {
        newDropdown.value = sortValue;
    }
}

/**
 * Get content data from a card element
 */
function getContentDataFromCard(card) {
    const contentId = card.dataset.contentId;
    const title = card.dataset.title;
    
    // Use ID for lookup if available, fallback to title
    return contentId 
        ? contentData.find(content => content.id == contentId)
        : contentData.find(content => content.title.toLowerCase() === title.toLowerCase());
}

/**
 * Get the ER (Engagement Rate) for a content item using existing calculation
 */
function getContentER(content) {
    // Use the same ER calculation logic from your existing code
    const isInstagramPosts = content.platform.toLowerCase() === 'instagram' && 
                            content.mediaType === 'post';
    const isInstagramStories = content.platform.toLowerCase() === 'instagram' && 
                              content.mediaType === 'story';
    
    // Return 0 for Instagram Posts and Instagram Stories (as per your existing logic)
    if (isInstagramPosts || isInstagramStories) {
        return 0;
    }
    
    // Calculate ER using the same formula: ((N + O + P + Q + U) / R) * 100
    if (content.metrics.views > 0) {
        const totalEngagement = (content.metrics.likes || 0) + 
                               (content.metrics.comments || 0) + 
                               (content.metrics.shares || 0) + 
                               (content.metrics.saves || 0) + 
                               (content.analytics.stickerTaps || 0);
        return (totalEngagement / content.metrics.views) * 100;
    }
    
    return 0;
}

/**
 * Get the HTML for the sort dropdown
 */
function getContentSortDropdownHTML() {
    return `
        <div class="content-sort-container">
            <label for="contentSortDropdown">Sort by:</label>
            <select id="contentSortDropdown" onchange="sortContentGrid()">
                <option value="default">Default Order</option>
                <option value="views-desc">Most Views</option>
                <option value="views-asc">Least Views</option>
                <option value="likes-desc">Most Likes</option>
                <option value="likes-asc">Least Likes</option>
                <option value="engagement-desc">Highest ER</option>
                <option value="engagement-asc">Lowest ER</option>
                <option value="date-desc">Newest First</option>
                <option value="date-asc">Oldest First</option>
            </select>
        </div>
    `;
}

// =============================================================================
// ANALYSIS FUNCTIONS
// =============================================================================

/**
 * Load analysis data from Info sheet columns E-L
 */
async function loadAnalysisData() {
    try {
        console.log('Loading analysis data from sheet:', currentCampaignId);
        
        // Fetch data from Info sheet - expanded range to include E1:L3
        const data = await fetchSheetData(currentCampaignId, 'Info', 'E1:L3');
        
        console.log('Analysis data received:', data);
        
        if (data && data.length > 2) {
            // Extract titles (row 2) and content (row 3)
            const titles = data[1] || [];  // Row 2 (index 1)
            const content = data[2] || []; // Row 3 (index 2)
            
            console.log('Analysis titles (row 2):', titles);
            console.log('Analysis content (row 3):', content);
            
            // Check if all analysis columns (E-L) are empty
            const allAnalysisData = [...titles, ...content];
            const hasAnyAnalysisContent = allAnalysisData.some(cell => 
                cell && cell.toString().trim() !== ''
            );
            
            if (!hasAnyAnalysisContent) {
                // Hide the entire analysis tab if all columns are empty
                hideAnalysisTab();
                return;
            }
            
            // Show the analysis tab (in case it was hidden)
            showAnalysisTab();
            
            // Key Findings (columns E, F, G, H)
            const findingTitles = [
                titles[0] || '', // E2
                titles[1] || '', // F2
                titles[2] || '', // G2
                titles[3] || ''  // H2
            ];
            
            const findingContents = [
                content[0] || '', // E3
                content[1] || '', // F3
                content[2] || '', // G3
                content[3] || ''  // H3
            ];
            
            // Recommendations (columns I, J, K, L)
            const recommendationTitles = [
                titles[4] || '', // I2
                titles[5] || '', // J2
                titles[6] || '', // K2
                titles[7] || ''  // L2
            ];
            
            const recommendationContents = [
                content[4] || '', // I3
                content[5] || '', // J3
                content[6] || '', // K3
                content[7] || ''  // L3
            ];
            
            console.log('Extracted analysis data:', {
                findingTitles,
                findingContents,
                recommendationTitles,
                recommendationContents
            });
            
            // Update Key Findings - only show cards with content
            updateAnalysisSection('findings', findingTitles, findingContents);
            
            // Update Recommendations - only show cards with content
            updateAnalysisSection('recommendations', recommendationTitles, recommendationContents);
            
        } else {
            console.warn('No analysis data found or insufficient rows in Info sheet');
            hideAnalysisTab();
        }
    } catch (error) {
        console.error('Error loading analysis data:', error);
        hideAnalysisTab();
    }
}

function updateAnalysisSection(sectionType, titles, contents) {
    const sectionClass = sectionType === 'findings' ? 'key-findings' : 'recommendations';
    const cardPrefix = sectionType === 'findings' ? 'finding' : 'recommendation';
    
    // Find the cards grid container
    const sectionElement = document.querySelector(`.${sectionClass}`);
    if (!sectionElement) return;
    
    const cardsGrid = sectionElement.querySelector('.analysis-cards-grid');
    if (!cardsGrid) return;
    
    // Clear existing cards
    cardsGrid.innerHTML = '';
    
    // Check if any content exists for this section
    const hasAnyContent = titles.some((title, index) => 
        (title && title.trim() !== '') || (contents[index] && contents[index].trim() !== '')
    );
    
    if (!hasAnyContent) {
        // Hide the entire section if no content
        sectionElement.style.display = 'none';
        return;
    }
    
    // Show the section
    sectionElement.style.display = 'block';
    
    // Create cards only for non-empty content
    titles.forEach((title, index) => {
        const content = contents[index] || '';
        
        // Only create card if either title or content has data
        if ((title && title.trim() !== '') || (content && content.trim() !== '')) {
            const card = createAnalysisCard(cardPrefix, index + 1, title, content);
            cardsGrid.appendChild(card);
        }
    });
}

/**
 * Create an individual analysis card
 */
function createAnalysisCard(cardPrefix, cardNumber, title, content) {
    const card = document.createElement('div');
    card.className = `analysis-card ${cardPrefix}-card-${cardNumber}`;
    
    // Default titles and icons if empty
    const defaultTitles = {
        finding: [`Key Finding ${cardNumber}`, `Performance Insight ${cardNumber}`, `Data Point ${cardNumber}`, `Research Finding ${cardNumber}`],
        recommendation: [`Recommendation ${cardNumber}`, `Strategy Suggestion ${cardNumber}`, `Optimization Tip ${cardNumber}`, `Action Item ${cardNumber}`]
    };
    
    const defaultIcons = {
        finding: ['📊', '🎯', '📈', '👥'],
        recommendation: ['💡', '🚀', '🔄', '📱']
    };
    
    const cardType = cardPrefix === 'finding' ? 'finding' : 'recommendation';
    const displayTitle = title && title.trim() !== '' ? title : defaultTitles[cardType][cardNumber - 1];
    const displayContent = content && content.trim() !== '' ? content : `Loading ${cardType}...`;
    const icon = defaultIcons[cardType][cardNumber - 1];
    
    card.innerHTML = `
        <div class="analysis-card-header">
            <div class="analysis-card-icon">${icon}</div>
            <div class="analysis-card-title">${escapeHtml(displayTitle)}</div>
        </div>
        <div class="analysis-card-content">${escapeHtml(displayContent)}</div>
    `;
    
    return card;
}

// Update hideAnalysisTab to use inline styles
function hideAnalysisTab() {
    // Hide the analysis tab content with inline style (higher priority than CSS)
    const analysisTab = document.getElementById('analysis');
    if (analysisTab) {
        analysisTab.style.display = 'none';
        analysisTab.classList.remove('active');
    }
    
    // Hide the analysis nav item
    const analysisNavItem = document.querySelector('.nav-item[onclick="showTab(\'analysis\')"]');
    if (analysisNavItem) {
        analysisNavItem.style.display = 'none';
    }
    
    console.log('Analysis tab hidden due to no content');
}

// Add this function to check if a sheet tab exists
async function checkSheetTabExists(sheetId, tabName) {
    try {
        // Try to fetch just one cell from the sheet
        // If the sheet doesn't exist, this will throw an error
        const data = await fetchSheetData(sheetId, tabName, 'A1:A1');
        return true; // Sheet exists
    } catch (error) {
        console.log(`Sheet tab "${tabName}" not found:`, error.message);
        return false; // Sheet doesn't exist
    }
}

/**
 * Show the analysis tab (in case it was previously hidden)
 */
function showAnalysisTab() {
    // Show the analysis tab content
    const analysisTab = document.getElementById('analysis');
    if (analysisTab) {
        // Remove display none that was set inline
        analysisTab.style.display = '';
        // Let CSS handle the display through the active class
        // This will be handled by showTab() when clicked
    }
    
    // Show the analysis nav item
    const analysisNavItem = document.querySelector('.nav-item[onclick="showTab(\'analysis\')"]');
    if (analysisNavItem) {
        analysisNavItem.style.display = '';
    }
    
    console.log('Analysis tab shown with content');
}


// =============================================================================
// CAMPAIGN INFO FUNCTIONS
// =============================================================================

/**
 * Load client and campaign info from Info sheet
 */
async function loadCampaignInfo() {
    try {
        console.log('Loading campaign info from sheet:', currentCampaignId);
        
        // Fetch data from Info sheet - expanded range to include columns U and V
        const data = await fetchSheetData(currentCampaignId, 'Info', 'A1:V3');
        
        console.log('Campaign info data received:', data);
        
        if (data && data.length > 1) {
            const clientName = data[1][0] || 'Client'; // Column A, Row 2
            const campaignName = data[1][1] || 'Campaign'; // Column B, Row 2
            const keyObjectives = data[1][2] || 'Campaign objectives will be displayed here.'; // Column C, Row 2
            const creativeConcept = data[1][3] || 'Creative concept details will be displayed here.'; // Column D, Row 2
            
            console.log('Extracted campaign data:', {
                clientName,
                campaignName,
                keyObjectives,
                creativeConcept
            });
            
            // Update existing elements
            const clientElement = document.getElementById('clientName');
            const campaignElement = document.getElementById('campaignName');
            const objectivesElement = document.getElementById('keyObjectives');
            const conceptElement = document.getElementById('creativeConcept');
            
            if (clientElement) clientElement.textContent = clientName;
            if (campaignElement) campaignElement.textContent = campaignName;
            if (objectivesElement) objectivesElement.textContent = keyObjectives;
            if (conceptElement) conceptElement.textContent = creativeConcept;
            
            // Extract Paid Media Findings and Recommendations from columns U and V
            // These are in row 3 (index 2)
            if (data.length > 2) {
                paidMediaData.findings = data[1][20] || ''; // Column U, Row 2 (index 20)
                paidMediaData.recommendations = data[1][21] || ''; // Column V, Row 2 (index 21)
                
                console.log('Extracted Paid Media analysis:', {
                    findings: paidMediaData.findings,
                    recommendations: paidMediaData.recommendations
                });
                
                // Update Paid Media analysis cards if they exist
                updatePaidMediaAnalysisCards();
            }
        }
        
        // Load analysis data separately with conditional display logic
        await loadAnalysisData();
        
    } catch (error) {
        console.error('Error loading campaign info:', error);
        
        // Set defaults if info can't be loaded
        const clientElement = document.getElementById('clientName');
        const campaignElement = document.getElementById('campaignName');
        const objectivesElement = document.getElementById('keyObjectives');
        const conceptElement = document.getElementById('creativeConcept');
        
        if (clientElement) clientElement.textContent = 'Client';
        if (campaignElement) campaignElement.textContent = 'Campaign';
        if (objectivesElement) objectivesElement.textContent = 'Unable to load campaign objectives. Please check your Google Sheet configuration.';
        if (conceptElement) conceptElement.textContent = 'Unable to load creative concept. Please check your Google Sheet configuration.';
        
        // Hide analysis tab if campaign info fails to load
        hideAnalysisTab();
    }
}


// =============================================================================
// COST CALCULATION FUNCTIONS
// =============================================================================

/**
 * Calculate CPE by content type using sum of costs / sum of engagement
 */
function calculateCPEByContentType(content, filter) {
    let totalCost = 0;
    let totalEngagement = 0;
    
    content.forEach(item => {
        const cost = toNumberFromCurrency(item.cost);
        const engagement = (item.metrics.likes || 0) + 
                          (item.metrics.comments || 0) + 
                          (item.metrics.shares || 0) + 
                          (item.metrics.saves || 0) + 
                          (item.analytics.stickerTaps || 0);
        
        // Include items with cost > 0 OR bonus items (even with 0 cost)
        if ((cost > 0 || item.bonus === true) && engagement > 0) {
            const platform = item.platform.toLowerCase();
            const mediaType = item.mediaType || '';
            
            let includeInCalculation = false;
            
            switch (filter) {
                case 'all':
                    includeInCalculation = true;
                    break;
                case 'tiktok':
                    includeInCalculation = platform === 'tiktok';
                    break;
                case 'instagram':
                    includeInCalculation = platform === 'instagram';
                    break;
                case 'youtube':
                    includeInCalculation = platform === 'youtube';
                    break;
                case 'youtube_shorts':
                    includeInCalculation = platform === 'youtube' && mediaType === 'short';
                    break;
                case 'youtube_integrations':
                    includeInCalculation = platform === 'youtube' && mediaType === 'integration';
                    break;
                case 'instagram_stories':
                    includeInCalculation = platform === 'instagram' && mediaType === 'story';
                    break;
                case 'instagram_reel':
                    includeInCalculation = platform === 'instagram' && mediaType === 'reel';
                    break;
                case 'instagram_posts':
                    includeInCalculation = platform === 'instagram' && mediaType === 'post';
                    break;
                case 'bonus':
                    includeInCalculation = item.bonus === true;
                    break;
                case 'linkedin_post':
                    includeInCalculation = platform === 'linkedin' && mediaType === 'post';
                    break;
                case 'linkedin_video':
                    includeInCalculation = platform === 'linkedin' && mediaType === 'video';
                    break;
                case 'facebook_post':
                    includeInCalculation = platform === 'facebook' && mediaType === 'post';
                    break;
                case 'facebook_reel':
                    includeInCalculation = platform === 'facebook' && mediaType === 'reel';
                    break;
                case 'twitter_post':
                    includeInCalculation = platform === 'twitter' && mediaType === 'post';
                    break;
                case 'twitter_video':
                    includeInCalculation = platform === 'twitter' && mediaType === 'video';
                    break;
                case 'linkedin':
                    includeInCalculation = platform === 'linkedin';
                    break;
                case 'facebook':
                    includeInCalculation = platform === 'facebook';
                    break;
                case 'twitter':
                    includeInCalculation = platform === 'twitter';
                    break;
            }
            
            if (includeInCalculation) {
                totalCost += cost;
                totalEngagement += engagement;
            }
        }
    });
    
    // CPE = total cost / total engagement
    return totalEngagement > 0 ? totalCost / totalEngagement : 0;
}

/**
 * Calculate CPV by content type using sum of costs / sum of views
 */
function calculateCPVByContentType(content, filter) {
    let totalCost = 0;
    let totalViews = 0;
    
    content.forEach(item => {
        const cost = toNumberFromCurrency(item.cost);
        const views = item.metrics.views || 0;
        
        // Include items with cost > 0 OR bonus items (even with 0 cost)
        if ((cost > 0 || item.bonus === true) && views > 0) {
            const platform = item.platform.toLowerCase();
            const mediaType = item.mediaType || '';
            
            let includeInCalculation = false;
            
            switch (filter) {
                case 'all':
                    includeInCalculation = true;
                    break;
                case 'tiktok':
                    includeInCalculation = platform === 'tiktok';
                    break;
                case 'instagram':
                    includeInCalculation = platform === 'instagram';
                    break;
                case 'youtube':
                    includeInCalculation = platform === 'youtube';
                    break;
                case 'youtube_shorts':
                    includeInCalculation = platform === 'youtube' && mediaType === 'short';
                    break;
                case 'youtube_integrations':
                    includeInCalculation = platform === 'youtube' && mediaType === 'integration';
                    break;
                case 'instagram_stories':
                    includeInCalculation = platform === 'instagram' && mediaType === 'story';
                    break;
                case 'instagram_reel':
                    includeInCalculation = platform === 'instagram' && mediaType === 'reel';
                    break;
                case 'instagram_posts':
                    includeInCalculation = platform === 'instagram' && mediaType === 'post';
                    break;
                case 'bonus':
                    includeInCalculation = item.bonus === true;
                    break;
                case 'linkedin_post':
                    includeInCalculation = platform === 'linkedin' && mediaType === 'post';
                    break;
                case 'linkedin_video':
                    includeInCalculation = platform === 'linkedin' && mediaType === 'video';
                    break;
                case 'facebook_post':
                    includeInCalculation = platform === 'facebook' && mediaType === 'post';
                    break;
                case 'facebook_reel':
                    includeInCalculation = platform === 'facebook' && mediaType === 'reel';
                    break;
                case 'twitter_post':
                    includeInCalculation = platform === 'twitter' && mediaType === 'post';
                    break;
                case 'twitter_video':
                    includeInCalculation = platform === 'twitter' && mediaType === 'video';
                    break;
                case 'linkedin':
                    includeInCalculation = platform === 'linkedin';
                    break;
                case 'facebook':
                    includeInCalculation = platform === 'facebook';
                    break;
                case 'twitter':
                    includeInCalculation = platform === 'twitter';
                    break;
            }
            
            if (includeInCalculation) {
                totalCost += cost;
                totalViews += views;
            }
        }
    });
    
    // CPV = total cost / total views
    return totalViews > 0 ? totalCost / totalViews : 0;
}

// =============================================================================
// MAIN FUNCTIONS
// =============================================================================

/**
 * Filter results by platform
 */
function filterResults(platform) {
    currentResultsFilter = platform;
    
    // Update active tab
    document.querySelectorAll('.platform-tab').forEach(tab => {
        tab.classList.remove('active');
    });
    event.target.classList.add('active');
    
    // Update results display
    updateResultsDisplay();
}

/**
 * Update secondary filter options based on primary platform selection
 */
function updateSecondaryFilter() {
    const primaryFilter = document.getElementById('contentPlatformFilter').value;
    const secondaryFilter = document.getElementById('contentMediaTypeFilter');
    
    // Clear existing options
    secondaryFilter.innerHTML = '';
    
    if (!primaryFilter || primaryFilter === '' || primaryFilter === 'bonus') {
        secondaryFilter.style.display = 'none';
        return;
    }
    
    // Show secondary filter
    secondaryFilter.style.display = 'block';
    
    // Add default "All Types" option
    const allOption = document.createElement('option');
    allOption.value = '';
    allOption.textContent = `All ${primaryFilter} Types`;
    secondaryFilter.appendChild(allOption);
    
    // Add platform-specific options
    const platformOptions = getMediaTypeOptions(primaryFilter);
    platformOptions.forEach(option => {
        const optionElement = document.createElement('option');
        optionElement.value = option.value;
        optionElement.textContent = option.label;
        secondaryFilter.appendChild(optionElement);
    });
}

/**
 * Override the existing getMediaTypeOptions function to only show available options
 */
function getMediaTypeOptions(platform) {
    const platformLower = platform.toLowerCase();
    const availableTypes = new Set();
    
    // Check what media types actually exist for this platform
    contentData.forEach(content => {
        if (content.platform.toLowerCase() === platformLower && content.mediaType) {
            availableTypes.add(content.mediaType);
        }
    });
    
    // Define all possible options for each platform
    const allOptions = {
        'instagram': [
            { value: 'post', label: 'Post' },
            { value: 'reel', label: 'Reel' },
            { value: 'story', label: 'Stories' }
        ],
        'tiktok': [],
        'youtube': [
            { value: 'integration', label: 'Integrations' },
            { value: 'short', label: 'Shorts' }
        ],
        'linkedin': [
            { value: 'post', label: 'Post' },
            { value: 'video', label: 'Video' }
        ],
        'facebook': [
            { value: 'post', label: 'Post' },
            { value: 'reel', label: 'Reel' }
        ],
        'twitter': [
            { value: 'post', label: 'Post' },
            { value: 'video', label: 'Video' }
        ]
    };
    
    const platformOptions = allOptions[platformLower] || [];
    
    // Filter to only include types that actually exist in the data
    return platformOptions.filter(option => availableTypes.has(option.value));
}

/**
 * Update results display based on current filter
 */
function updateResultsDisplay() {
  let filteredContent = contentData;
  const currencySymbol = detectCurrencySymbol(filteredContent);

  // Filter by platform/type if not 'all'
  if (currentResultsFilter !== 'all') {
    filteredContent = contentData.filter(content => {
      const platform = (content.platform || '').toLowerCase();
      const mediaType = content.mediaType || '';

      if (currentResultsFilter === 'bonus') return content.bonus === true;

      switch (currentResultsFilter) {
        case 'tiktok':               return platform === 'tiktok';
        case 'instagram':            return platform === 'instagram';
        case 'youtube':              return platform === 'youtube';
        case 'linkedin':             return platform === 'linkedin';
        case 'facebook':             return platform === 'facebook';
        case 'twitter':              return platform === 'twitter';

        // Instagram sub-filters
        case 'instagram_reel':       return platform === 'instagram' && mediaType === 'reel';
        case 'instagram_stories':    return platform === 'instagram' && mediaType === 'story';
        case 'instagram_posts':      return platform === 'instagram' && mediaType === 'post';

        // YouTube sub-filters
        case 'youtube_integrations': return platform === 'youtube' && mediaType === 'integration';
        case 'youtube_shorts':       return platform === 'youtube' && mediaType === 'short';

        // LinkedIn / Facebook / Twitter sub-filters
        case 'linkedin_post':        return platform === 'linkedin' && mediaType === 'post';
        case 'linkedin_video':       return platform === 'linkedin' && mediaType === 'video';
        case 'facebook_post':        return platform === 'facebook' && mediaType === 'post';
        case 'facebook_reel':        return platform === 'facebook' && mediaType === 'reel';
        case 'twitter_post':         return platform === 'twitter'  && mediaType === 'post';
        case 'twitter_video':        return platform === 'twitter'  && mediaType === 'video';
        default:                     return true;
      }
    });
  }

  // Aggregate totals
  const totals = filteredContent.reduce((acc, c) => {
    const m = c.metrics || {};
    acc.views       += m.views    || 0;
    acc.likes       += m.likes    || 0;
    acc.comments    += m.comments || 0;
    acc.shares      += m.shares   || 0;
    acc.saves       += m.saves    || 0;
    acc.linkClicks  += (c.analytics?.linkClicks   || 0);
    acc.stickerTaps += (c.analytics?.stickerTaps  || 0);

    const isIGStory = (c.platform || '').toLowerCase() === 'instagram' && c.mediaType === 'story';
    acc.content += isIGStory ? (c.analytics?.stories || 1) : 1;

    return acc;
  }, { views:0, likes:0, comments:0, shares:0, saves:0, content:0, linkClicks:0, stickerTaps:0 });

  // Helper to set text
  const setText = (id, val) => {
    const el = document.getElementById(id);
    if (el != null) el.textContent = formatNumber(val);
  };

  // Helper to show/hide card based on value
  const toggleCard = (id, value) => {
    const el = document.getElementById(id);
    const box = el?.closest('.result-stat-box');
    if (box) {
      box.style.display = value > 0 ? 'block' : 'none';
    }
    if (el && value > 0) {
      el.textContent = formatNumber(value);
    }
  };

  // Update all cards and hide if zero
  toggleCard('resultsTotalViews', totals.views);
  toggleCard('resultsTotalLikes', totals.likes);
  toggleCard('resultsTotalComments', totals.comments);
  toggleCard('resultsTotalShares', totals.shares);
  toggleCard('resultsTotalContent', totals.content);
  toggleCard('resultsTotalClicks', totals.linkClicks);

  // Saves vs Sticker Taps for IG Stories
  const isIGStories = currentResultsFilter === 'instagram_stories';
  const savesValueForCard = isIGStories ? totals.stickerTaps : totals.saves;

  const savesValueEl = document.getElementById('resultsTotalSaves');
  const savesBox     = savesValueEl?.closest('.result-stat-box');
  const savesLabelEl = savesBox?.querySelector('.result-stat-label');

  if (savesLabelEl) savesLabelEl.textContent = isIGStories ? 'Sticker Taps' : 'Saves';
  if (savesValueEl) savesValueEl.textContent = formatNumber(savesValueForCard);

  // Hide Saves/Shares for YouTube types or when zero
  const isYouTubeFilter = ['youtube','youtube_integrations','youtube_shorts'].includes(currentResultsFilter);
  if (savesBox)  savesBox.style.display  = (savesValueForCard === 0 || isYouTubeFilter) ? 'none' : 'block';

  // Cost cards
  const cpeBox = document.getElementById('resultsTotalCPE')?.closest('.result-stat-box');
  const cpvBox = document.getElementById('resultsTotalCPV')?.closest('.result-stat-box');

  const avgCPE = calculateCPEByContentType(filteredContent, currentResultsFilter);
  const avgCPV = calculateCPVByContentType(filteredContent, currentResultsFilter);

  // Helper function to format cost with appropriate decimal places
  const formatCostMetric = (value) => {
    if (value < 0.01 && value > 0) {
      if (value < 0.001) return value.toFixed(4);
      if (value < 0.01) return value.toFixed(3);
    }
    return value.toFixed(2);
  };

  // CPE - hide if zero
  if (cpeBox) {
    const cpeEl = document.getElementById('resultsTotalCPE');
    if (avgCPE > 0) { 
      if (cpeEl) cpeEl.textContent = `${currencySymbol}${formatCostMetric(avgCPE)}`; 
      cpeBox.style.display = 'block'; 
    } else {
      cpeBox.style.display = 'none';
    }
  }

  // CPV - hide if zero
  if (cpvBox) {
    const cpvEl = document.getElementById('resultsTotalCPV');
    if (avgCPV > 0) { 
      if (cpvEl) cpvEl.textContent = `${currencySymbol}${formatCostMetric(avgCPV)}`; 
      cpvBox.style.display = 'block'; 
    } else {
      cpvBox.style.display = 'none';
    }
  }
}

/**
* Initialize the dashboard application
*/
function initializeDashboard() {
   // IMMEDIATELY hide the analysis tab to prevent flash of content
   const analysisTab = document.getElementById('analysis');
   if (analysisTab) {
       analysisTab.style.display = 'none';
       analysisTab.classList.remove('active'); // Ensure it doesn't have active class
   }
   
   // Also hide the analysis nav item initially
   const analysisNavItem = document.querySelector('.nav-item[onclick="showTab(\'analysis\')"]');
   if (analysisNavItem) {
       analysisNavItem.style.display = 'none';
   }
   
   // Continue with the rest of initialization...
   // Validate configuration
   if (!validateConfig()) {
       showError('Configuration not properly set. Please check config.js file.');
       return;
   }
   
   // Get campaign ID from URL parameters
    const urlParams = new URLSearchParams(window.location.search);
    currentCampaignId = decodeURIComponent(urlParams.get('campaignId') || '');
    const validatedCampaignId = validateSheetId(currentCampaignId);
    if (!validatedCampaignId) {
        showError('Invalid or missing campaignId parameter in URL. Add ?campaignId=YOUR_GOOGLE_SHEET_ID');
        return;
    }
    currentCampaignId = validatedCampaignId;
   
   // Load the data
   loadCampaignInfo();
   loadInfluencerData();
   loadContentData();
   loadWeeklyData();
   loadPaidMediaData();
}

/**
* Load influencer data from Google Sheets Report tab
*/
async function loadInfluencerData() {
   try {
       showLoading();
       
       // Fetch data from the Report sheet
       const data = await fetchSheetData(
           currentCampaignId, 
           'Report',
           'A1:AD1000'
       );
       
       // Process the data for influencers
       const processedData = processReportData(data);
       influencersData = processedData.influencers;
       window.influencersData = influencersData;
       
       // Update the display
       updateInfluencersDisplay();
       showDashboard();

   } catch (error) {
       console.error('Error loading data:', error);
       showError(`Failed to load influencer data: ${error.message}`);
   }
}

/**
* Load content data from Google Sheets Report tab
*/
async function loadContentData() {
   try {
       // Fetch data from the Report sheet
       const data = await fetchSheetData(
           currentCampaignId, 
           'Report',
           'A1:AD1000'
       );
       
       // Process the data for content
       const processedData = processReportData(data);
       contentData = processedData.content;
       window.contentData = contentData;
       
       updateContentDisplay();

   } catch (error) {
       console.error('Error loading content data:', error);
       contentData = [];
       updateContentDisplay();
   }
}

/**
 * Validate and sanitize Google Sheet ID
 */
function validateSheetId(sheetId) {
    if (!sheetId) return null;
    
    // Google Sheet IDs can contain letters, numbers, hyphens, and underscores
    // They are typically 44 characters long but can vary
    const cleanId = sheetId.trim();
    
    // Basic validation - ensure it's not empty and contains valid characters
    if (!/^[a-zA-Z0-9\-_]+$/.test(cleanId)) {
        console.error('Invalid Sheet ID format:', cleanId);
        return null;
    }
    
    return cleanId;
}

/**
* Fetch data from Google Sheets via backend API
*/
async function fetchSheetData(sheetId, sheetName, range) {
   // Properly encode all parameters
   const params = new URLSearchParams({
       sheetId: sheetId,
       sheetName: sheetName,
       range: range
   });
   const url = `${CONFIG.API_BASE_URL}/sheets?${params.toString()}`;
   
   const response = await fetch(url, {
       method: 'GET',
       headers: {
           'x-api-key': CONFIG.API_SECRET_KEY,
           'Content-Type': 'application/json'
       }
   });
   
   if (!response.ok) {
       let errorMessage;
       try {
           const error = await response.json();
           errorMessage = error.error || error.details || 'Unknown error';
       } catch {
           errorMessage = `HTTP ${response.status}: ${response.statusText}`;
       }
       throw new Error(errorMessage);
   }
   
   const result = await response.json();
   return result.data || [];
}

// Parses values like "$1,234.56", "£1.234,56", "€8,24" → number
function parseCurrencyValue(val) {
 if (val === undefined || val === null) return 0;
 let s = String(val).trim().replace(/\s+/g, '');
 if (s.includes(',') && !s.includes('.')) s = s.replace(/\./g, '').replace(',', '.');
 else s = s.replace(/,/g, '');
 s = s.replace(/[^0-9.\-]/g, '');
 const n = parseFloat(s);
 return Number.isFinite(n) ? n : 0;
}


/**
* Process Report data into influencers and content objects
*/
function processReportData(data) {
   if (data.length < 2) return { influencers: [], content: [] };
   
   const rows = data.slice(1); // Skip header row
   const influencersMap = new Map();
   const content = [];
   
   rows.forEach((row, index) => {
       const name = cleanString(row[0]) || '';           // A: Name
       const profileLink = cleanString(row[1]) || '';    // B: Profile Link
       const followers = parseInt(row[2]) || 0;          // C: Followers
       const handle = formatHandle(cleanString(row[3]) || ''); // D: Handle
       const country = cleanString(row[4]) || '';        // E: Country
       const cost = cleanString(row[5]) || '';          // F: Cost
       const socialMedia = formatPlatformName(row[6]) || ''; // G: Social Media
       const profilePic = cleanString(row[7]) || '';     // H: Profile Picture
       const typeOfPost = cleanString(row[8]) || '';     // I: Type of post
       const linkToPost = cleanString(row[9]) || '';     // J: Link to Post
       const dateOfPost = cleanString(row[10]) || '';    // K: Date of post
       const thumbnail = cleanString(row[11]) || '';     // L: Thumbnail of post
       const caption = cleanString(row[12]) || '';       // M: Post Caption
       const bonus = cleanString(row[29]) || '';         // AD: Bonus (column 30, index 29)
       console.log('Row index:', index, 'Bonus value:', bonus, 'Is bonus?:', /^yes$/i.test(bonus.trim()));
       
       // NEW: Capture cost metrics from spreadsheet columns with currency detection
       const cpeRaw = cleanString(row[24]) || '';  // Y: CPE (column 25, index 24)
       const cpvRaw = cleanString(row[25]) || '';  // Z: CPV (column 26, index 25)
       const ecpmRaw = cleanString(row[28]) || ''; // AC: eCPM (column 29, index 28)
       
       const spreadsheetCPE = parseCurrencyValue(cpeRaw);
       const spreadsheetCPV = parseCurrencyValue(cpvRaw);
       const spreadsheetECPM = parseCurrencyValue(ecpmRaw);
       
       // Skip rows without name, social media, or profile link
       if (!name || !socialMedia || !profileLink) return;
       
       // Create/update influencer entry using Profile Link as unique key
       if (!influencersMap.has(profileLink)) {
           influencersMap.set(profileLink, {
               id: influencersMap.size + 1,
               name: name,
               profileLink: profileLink,
               followers: followers,
               handle: handle,
               country: country,
               cost: cost,
               socialMedia: socialMedia,
               profilePic: profilePic,
               followersFormatted: formatNumber(followers)
           });
       } else {
           const existingInfluencer = influencersMap.get(profileLink);
           
           if (!existingInfluencer.profilePic && profilePic) {
               existingInfluencer.profilePic = profilePic;
           }
           
           if (country && country !== existingInfluencer.country) {
               if (!existingInfluencer.country) {
                   existingInfluencer.country = country;
               } else if (!existingInfluencer.country.includes(country)) {
                   existingInfluencer.country = existingInfluencer.country + ', ' + country;
               }
           }
       }
       
       // Create content entry if there's post data
       if (linkToPost || thumbnail || caption || typeOfPost) {
           // Store raw metrics for ER calculation later
           const likes = parseMetricNumber(row[13]);      // N: Likes
           const comments = parseMetricNumber(row[14]);   // O: Comments
           const shares = parseMetricNumber(row[15]);     // P: Shares
           const saves = parseMetricNumber(row[16]);      // Q: Saves
           const views = parseMetricNumber(row[17]);      // R: Views
           const stickerTaps = parseMetricNumber(row[20]); // U: Sticker Taps
           
           content.push({
               id: content.length + 1,
               influencer: name,
               followers: followers,
               title: caption || `${socialMedia} ${typeOfPost || 'post'} by ${name}`,
               thumbnail: thumbnail,
               platform: socialMedia,
               mediaType: getMediaTypeValue(socialMedia, typeOfPost),
               postedDate: dateOfPost,
               videoLink: linkToPost,
               cost: cost,
               bonus: /^yes$/i.test(bonus.trim()),
               metrics: {
                   likes: likes,
                   comments: comments,
                   shares: shares,
                   saves: saves,
                   views: views,
                   reach: parseMetricNumber(row[22]),      // W: Reach
                   impressions: parseMetricNumber(row[23]), // X: Impressions
                   impressions2: parseMetricNumber(row[27]) // AB: Impressions2
               },
               analytics: {
                   er: 0, // Will be calculated in summary functions
                   stories: parseMetricNumber(row[19]),    // T: N. of Stories
                   stickerTaps: stickerTaps,              // U: Sticker Taps
                   linkClicks: parseMetricNumber(row[21])  // V: Link Clicks
               },
               // NEW: Add spreadsheet cost metrics with currency info
               costMetrics: {
                   cpe: spreadsheetCPE,    // Y: CPE from spreadsheet
                   cpv: spreadsheetCPV,    // Z: CPV from spreadsheet
                   ecpm: spreadsheetECPM   // AC: eCPM from spreadsheet
               }
           });
       }
   });
   
   return {
       influencers: Array.from(influencersMap.values()),
       content: content
   };
}

/**
* Map social media and type of post to media type values
*/
function getMediaTypeValue(socialMedia, typeOfPost) {
   const platform = socialMedia.toLowerCase();
   const type = typeOfPost.toLowerCase();
   
   if (platform === 'instagram') {
       if (type.includes('post')) return 'post';
       if (type.includes('reel')) return 'reel';
       if (type.includes('story') || type.includes('stories')) return 'story';
   } else if (platform === 'youtube') {
       if (type.includes('integration')) return 'integration';
       if (type.includes('short')) return 'short';
   } else if (platform === 'linkedin') {
       if (type.includes('video')) return 'video';
       return 'post';
   } else if (platform === 'facebook') {
       if (type.includes('reel')) return 'reel';
       return 'post';
   } else if (platform === 'twitter') {
       if (type.includes('video')) return 'video';
       return 'post';
   }
   
   return type || ''; // Return original type or empty for TikTok
}

// =============================================================================
// DISPLAY FUNCTIONS
// =============================================================================

/**
* Update both influencer displays
*/
function updateInfluencersDisplay() {
   updateInfluencersGrid();
   updateInfluencersTable();
   updateMetricsSummary();
   updateOverviewDisplay();
   updateInfluencersPlatformFilter(); // Added this line
}

/**
* Update the influencers grid display
*/
function updateInfluencersGrid() {
   const grid = document.getElementById('influencersGrid');
   grid.innerHTML = '';
   
   if (influencersData.length === 0) {
       grid.innerHTML = `
           <div class="empty-state">
               <h3>No influencer data found</h3>
               <p>Make sure your Google Sheet has data in the "Report" tab with the correct column structure</p>
           </div>
       `;
       return;
   }
   
   influencersData.forEach(influencer => {
       const card = createInfluencerCard(influencer);
       grid.appendChild(card);
   });
}

function safeImageUrl(url) {
    if (!url || url.trim() === '') return '';
    
    url = url.trim();
    
    // Remove protocol without regex
    let urlWithoutProtocol = url;
    if (url.startsWith('https://')) {
        urlWithoutProtocol = url.substring(8);
    } else if (url.startsWith('http://')) {
        urlWithoutProtocol = url.substring(7);
    }
    
    return `https://images.weserv.nl/?url=${encodeURIComponent(urlWithoutProtocol)}&w=300&h=400&fit=cover`;
}

/**
* Create image element with automatic fallback to proxy
*/
function createImageWithFallback(src, alt, className, fallbackHTML) {
    const img = document.createElement('img');
    img.src = src;
    img.alt = alt;
    img.className = className;
    
    let proxyAttempted = false;
    
    img.onerror = function() {
        if (!proxyAttempted) {
            proxyAttempted = true;
            // Try proxy
            this.src = getProxyImageUrl(src);
        } else {
            // Proxy also failed, replace with fallback
            const fallbackDiv = document.createElement('div');
            fallbackDiv.innerHTML = fallbackHTML;
            const fallbackElement = fallbackDiv.firstChild;
            this.parentNode.replaceChild(fallbackElement, this);
        }
    };
    
    return img.outerHTML;
}

/**
* Create inline SVG avatar (no external requests)
*/
function createAvatarSVG(initials, backgroundColor) {
   return `
       <svg width="60" height="60" xmlns="http://www.w3.org/2000/svg">
           <rect width="60" height="60" fill="${backgroundColor}" rx="30"/>
           <text x="50%" y="50%" dominant-baseline="middle" text-anchor="middle" fill="white" font-size="24" font-weight="bold">${initials}</text>
       </svg>
   `;
}

function createInfluencerCard(influencer) {
    const card = document.createElement('div');
    card.className = 'influencer-card';
    card.dataset.name = influencer.name.toLowerCase();
    card.dataset.platform = influencer.socialMedia.toLowerCase();
    card.dataset.profileLink = influencer.profileLink;
    
    // Handle profile image
    let avatarHTML;
    if (influencer.profilePic && influencer.profilePic.trim() !== '') {
        const proxiedImageUrl = safeImageUrl(influencer.profilePic);
        const initials = getInitials(influencer.name);
        const avatarColor = getAvatarColor(influencer.socialMedia);
        
        avatarHTML = `
            <img src="${proxiedImageUrl}" 
                 alt="${influencer.name}" 
                 class="avatar"
                 onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
            <div class="avatar" style="display:none; background: ${avatarColor}; color: white; font-size: 1.8rem; font-weight: bold; align-items: center; justify-content: center;">
                ${initials}
            </div>
        `;
    } else {
        const initials = getInitials(influencer.name);
        const avatarColor = getAvatarColor(influencer.socialMedia);
        avatarHTML = `
            <div class="avatar" style="background: ${avatarColor}; color: white; font-size: 1.8rem; font-weight: bold; display: flex; align-items: center; justify-content: center;">
                ${initials}
            </div>
        `;
    }
    
    // Rest of the card creation code...
    let handleHTML;
    if (influencer.profileLink && influencer.profileLink.trim() !== '') {
        handleHTML = `
            <a href="${influencer.profileLink}" target="_blank" class="clickable-handle" onclick="event.stopPropagation();">
                ${escapeHtml(influencer.handle)}
                <span class="profile-link-icon">🔗</span>
            </a>
        `;
    } else {
        handleHTML = `<div class="handle">${escapeHtml(influencer.handle)}</div>`;
    }
    
    card.innerHTML = `
        <div class="influencer-header">
            ${avatarHTML}
            <div class="influencer-info">
                <h3>${escapeHtml(influencer.name)}</h3>
                ${handleHTML}
            </div>
        </div>
        
        <div class="influencer-content">
            <div class="metrics-row">
                <div class="metric">
                    <div class="metric-label">Followers</div>
                    <div class="metric-value">${influencer.followersFormatted}</div>
                </div>
                <div class="metric">
                    <div class="metric-label">Country</div>
                    <div class="metric-value" style="font-size: 1.2rem;">${influencer.country || 'N/A'}</div>
                </div>
            </div>
            <div class="platform-badge platform-${influencer.socialMedia.toLowerCase()}">
                ${influencer.socialMedia}
            </div>
        </div>
    `;
    
    return card;
}

/**
* Get a nice avatar color based on platform
*/
function getAvatarColor(platform) {
   const colors = {
       'instagram': 'linear-gradient(45deg, #f09433 0%, #e6683c 25%, #dc2743 50%, #cc2366 75%, #bc1888 100%)',
       'tiktok': '#000000',
       'youtube': '#ff0000',
       'facebook': '#4267B2',
       'twitter': '#1DA1F2',
       'linkedin': '#0077B5'
   };
   return colors[platform.toLowerCase()] || '#fc0060';
}

/**
* Update the influencers table display
*/
function updateInfluencersTable() {
   const tableContent = document.getElementById('tableContent');
   if (!tableContent) return;
   
   tableContent.innerHTML = '';
   
   if (influencersData.length === 0) {
       tableContent.innerHTML = `
           <div class="table-row">
               <div style="grid-column: 1/-1; text-align: center; padding: 2rem; color: #64748b;">
                   <h3>No influencer data found</h3>
                   <p>Make sure your Google Sheet has data in the "Report" tab</p>
               </div>
           </div>
       `;
       return;
   }
   
   influencersData.forEach(influencer => {
       const row = createInfluencerTableRow(influencer);
       tableContent.appendChild(row);
   });
}

/**
* Update metrics summary based on visible influencers
*/
function updateMetricsSummary() {
   let visibleInfluencers = [];
   
   if (currentView === 'grid') {
       const cards = document.querySelectorAll('.influencer-card');
       cards.forEach(card => {
           if (card.style.display !== 'none') {
               const profileLink = card.dataset.profileLink;
               const influencer = influencersData.find(inf => inf.profileLink === profileLink);
               if (influencer) visibleInfluencers.push(influencer);
           }
       });
   } else {
       const rows = document.querySelectorAll('.table-row');
       rows.forEach(tableRow => {
           if (tableRow.dataset.profileLink && tableRow.style.display !== 'none') {
               const profileLink = tableRow.dataset.profileLink;
               const influencer = influencersData.find(inf => inf.profileLink === profileLink);
               if (influencer) visibleInfluencers.push(influencer);
           }
       });
   }
   
   // Calculate metrics
   const uniqueInfluencers = new Set(visibleInfluencers.map(inf => inf.name.toLowerCase())).size;
   const totalAccounts = visibleInfluencers.length;
   const totalFollowers = visibleInfluencers.reduce((sum, inf) => sum + inf.followers, 0);
   
   document.getElementById('influencersCount').textContent = uniqueInfluencers;
   document.getElementById('accountsCount').textContent = totalAccounts;
   document.getElementById('totalFollowers').textContent = formatNumber(totalFollowers);
}

/**
* Update overview display with current data
*/
function updateOverviewDisplay() {
   // Calculate overview metrics
   const uniqueInfluencers = new Set(influencersData.map(inf => inf.name.toLowerCase())).size;
   const totalAccounts = influencersData.length;
   const totalFollowers = influencersData.reduce((sum, inf) => sum + inf.followers, 0);
   const totalLinkClicks = contentData.reduce((sum, content) => sum + (content.analytics.linkClicks || 0), 0);
   
   // Calculate content pieces with stories logic
   const totalContentPieces = calculateTotalContentPieces();
   
   // Get unique countries
   const countries = getUniqueCountries();
   
   // Update main stats
   const overviewInfluencersEl = document.getElementById('overviewInfluencers');
   const overviewAccountsEl = document.getElementById('overviewAccounts');
   const overviewCountriesEl = document.getElementById('overviewCountries');
   const overviewFollowersEl = document.getElementById('overviewFollowers');
   const overviewContentEl = document.getElementById('overviewContent');
   const overviewLinkClicksEl = document.getElementById('overviewLinkClicks');
   
   if (overviewInfluencersEl) overviewInfluencersEl.textContent = uniqueInfluencers;
   if (overviewAccountsEl) overviewAccountsEl.textContent = totalAccounts;
   if (overviewCountriesEl) overviewCountriesEl.textContent = countries;
   if (overviewFollowersEl) overviewFollowersEl.textContent = formatNumber(totalFollowers);
   if (overviewContentEl) overviewContentEl.textContent = totalContentPieces;
    if (overviewLinkClicksEl) {
        // If no link clicks, don't show the Link Clicks card at all
        const linkClicksCard = overviewLinkClicksEl.closest('.overview-stat-card');
        if (linkClicksCard) {
            if (totalLinkClicks === 0) {
                // Remove the card entirely from the DOM
                linkClicksCard.remove();
            } else {
                // Update the value if there are link clicks
                overviewLinkClicksEl.textContent = formatNumber(totalLinkClicks);
            }
        }
    }

   
   // Update platform breakdowns
   updateFollowersBreakdown();
   updateContentBreakdown();
   updateContentTypesBreakdown();
}

/**
* Calculate total content pieces: all rows + sum of column T - count of "story" in column I
*/
function calculateTotalContentPieces() {
   let totalPieces = 0;
   
   // Process each content row
   contentData.forEach(content => {
       const isInstagramStories = content.platform.toLowerCase() === 'instagram' && 
                                 content.mediaType === 'story';
       
       if (isInstagramStories) {
           // For Instagram Stories, add the number of stories (Column T)
           totalPieces += content.analytics.stories || 1;
       } else {
           // For regular content, count as 1 piece
           totalPieces += 1;
       }
   });
   
   return totalPieces;
}

/**
* Get unique countries from influencer data (Column E)
*/
function getUniqueCountries() {
   const allCountries = new Set();
   
   influencersData.forEach(influencer => {
       if (influencer.country && influencer.country.trim()) {
           // Split by comma and clean up each country
           const countries = influencer.country.split(',')
               .map(country => country.trim())
               .filter(country => country.length > 0);
           
           countries.forEach(country => allCountries.add(country));
       }
   });
   
   return Array.from(allCountries).join(', ') || 'N/A';
}

/**
* Update followers breakdown by platform
*/
function updateFollowersBreakdown() {
   const container = document.getElementById('followersBreakdown');
   if (!container) return;
   
   container.innerHTML = '';
   
   // Group followers by platform
   const followersByPlatform = {};
   influencersData.forEach(influencer => {
       const platform = influencer.socialMedia.toLowerCase();
       if (!followersByPlatform[platform]) {
           followersByPlatform[platform] = 0;
       }
       followersByPlatform[platform] += influencer.followers;
   });
   
   // Create breakdown items
   Object.entries(followersByPlatform).forEach(([platform, followers]) => {
       const item = document.createElement('div');
       item.className = 'overview-breakdown-item';
       item.innerHTML = `
           <div class="overview-breakdown-item-label">
               <div class="platform-icon ${platform}">
                   ${platform.charAt(0).toUpperCase()}
               </div>
               ${formatPlatformName(platform)}
           </div>
           <div class="overview-breakdown-item-value">${formatNumber(followers)}</div>
       `;
       container.appendChild(item);
   });
}

/**
* Update content breakdown by platform with stories logic
*/
function updateContentBreakdown() {
   const container = document.getElementById('contentBreakdown');
   if (!container) return;
   
   container.innerHTML = '';
   
   // Group content by platform with special handling for Instagram
   const contentByPlatform = {};
   
   contentData.forEach(content => {
       const platform = content.platform.toLowerCase();
       if (!contentByPlatform[platform]) {
           contentByPlatform[platform] = 0;
       }
       
       const isInstagramStories = platform === 'instagram' && content.mediaType === 'story';
       
       if (isInstagramStories) {
           // For Instagram Stories, add the number of stories
           contentByPlatform[platform] += content.analytics.stories || 1;
       } else {
           // For regular content, count as 1 piece
           contentByPlatform[platform] += 1;
       }
   });
   
   // Create breakdown items
   Object.entries(contentByPlatform).forEach(([platform, count]) => {
       const item = document.createElement('div');
       item.className = 'overview-breakdown-item';
       item.innerHTML = `
           <div class="overview-breakdown-item-label">
               <div class="platform-icon ${platform}">
                   ${platform.charAt(0).toUpperCase()}
               </div>
               ${formatPlatformName(platform)}
           </div>
           <div class="overview-breakdown-item-value">${count}</div>
       `;
       container.appendChild(item);
   });
}

/**
* Update content types breakdown (replaces platforms breakdown)
*/
function updateContentTypesBreakdown() {
   const container = document.getElementById('contentTypesBreakdown');
   if (!container) return;
   
   container.innerHTML = '';
   
   // Group content by platform and media type
   const contentTypes = {};
   
   contentData.forEach(content => {
       const platform = content.platform;
       const mediaType = content.mediaType;
       
       let typeKey;
       let displayName;
       
       if (platform.toLowerCase() === 'instagram') {
           if (mediaType === 'story') {
               typeKey = 'instagram_stories';
               displayName = 'Instagram Stories';
           } else if (mediaType === 'reel') {
               typeKey = 'instagram_reels';
               displayName = 'Instagram Reels';
           } else if (mediaType === 'post') {
               typeKey = 'instagram_posts';
               displayName = 'Instagram Posts';
           } else {
               typeKey = 'instagram_other';
               displayName = 'Instagram Content';
           }
       } else if (platform.toLowerCase() === 'youtube') {
           if (mediaType === 'integration') {
               typeKey = 'youtube_integrations';
               displayName = 'YouTube Integrations';
           } else if (mediaType === 'short') {
               typeKey = 'youtube_shorts';
               displayName = 'YouTube Shorts';
           } else {
               typeKey = 'youtube_other';
               displayName = 'YouTube Content';
           }
       } else if (platform.toLowerCase() === 'tiktok') {
           typeKey = 'tiktok_content';
           displayName = 'TikTok Content';
       } else {
           typeKey = `${platform.toLowerCase()}_content`;
           displayName = `${platform} Content`;
       }
       
       if (!contentTypes[typeKey]) {
           contentTypes[typeKey] = {
               count: 0,
               displayName: displayName,
               platform: platform.toLowerCase()
           };
       }
       
       // Count logic: Instagram Stories use story count, others count as 1
       if (typeKey === 'instagram_stories') {
           contentTypes[typeKey].count += content.analytics.stories || 1;
       } else {
           contentTypes[typeKey].count += 1;
       }
   });
   
   // Create breakdown items
   Object.entries(contentTypes).forEach(([key, data]) => {
       const item = document.createElement('div');
       item.className = 'overview-breakdown-item';
       item.innerHTML = `
           <div class="overview-breakdown-item-label">
               <div class="platform-icon ${data.platform}">
                   ${data.platform.charAt(0).toUpperCase()}
               </div>
               ${data.displayName}
           </div>
           <div class="overview-breakdown-item-value">${data.count}</div>
       `;
       container.appendChild(item);
   });
}

/**
* Create an individual table row for an influencer
*/
function createInfluencerTableRow(influencer) {
    const row = document.createElement('div');
    row.className = 'table-row';
    row.dataset.name = influencer.name.toLowerCase();
    row.dataset.platform = influencer.socialMedia.toLowerCase();
    row.dataset.profileLink = influencer.profileLink;
    
    // Handle profile image with fallback
    let avatarHTML;
    if (influencer.profilePic && influencer.profilePic.trim() !== '') {
        const proxiedImageUrl = safeImageUrl(influencer.profilePic);
        avatarHTML = `
            <img src="${proxiedImageUrl}" 
                 alt="${influencer.name}" 
                 class="table-avatar"
                 onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
            <div class="table-avatar" style="display:none;">
                ${getInitials(influencer.name)}
            </div>
        `;
    } else {
        avatarHTML = `
            <div class="table-avatar">
                ${getInitials(influencer.name)}
            </div>
        `;
    }
    
    row.innerHTML = `
        ${avatarHTML}
        <div class="table-influencer-info">
            <div class="table-name">${escapeHtml(influencer.name)}</div>
            <div class="table-handle">${escapeHtml(influencer.handle)}</div>
        </div>
        <div class="table-metric">${influencer.followersFormatted}</div>
        <div>
            <span class="table-platform-badge platform-${influencer.socialMedia.toLowerCase()}">
                ${influencer.socialMedia}
            </span>
        </div>
        <div class="table-handle">${escapeHtml(influencer.handle)}</div>
    `;
    
    return row;
}

/**
* Update both content displays
*/
function updateContentDisplay() {
   updateContentGrid();
   updateContentTable();
   updateContentMetricsSummary();
   updateResultsDisplay();
   updateWeeklyResultsDisplay();
   updateOverviewDisplay();
   updateContentPlatformFilter();    // Added this line
   renderPlatformFilters();           // Added this line
   renderSubFilters(currentPlatform); // Added this line
}

/**
* Update content grid view
*/
function updateContentGrid() {
   const grid = document.getElementById('contentGrid');
   if (!grid) return;
   
   grid.innerHTML = '';
   
   // Add sort dropdown at the top
   const sortContainer = document.createElement('div');
   sortContainer.innerHTML = getContentSortDropdownHTML();
   sortContainer.style.gridColumn = '1 / -1';
   grid.appendChild(sortContainer);
   
   if (contentData.length === 0) {
       const emptyState = document.createElement('div');
       emptyState.className = 'empty-state';
       emptyState.style.gridColumn = '1 / -1';
       emptyState.innerHTML = `
           <h3>No content data found</h3>
           <p>Make sure your Google Sheet has data in the "Report" tab</p>
       `;
       grid.appendChild(emptyState);
       return;
   }
   
   contentData.forEach(content => {
       const card = createContentCard(content);
       grid.appendChild(card);
   });
   
   // Restore previous sort if any
   const dropdown = document.getElementById('contentSortDropdown');
   if (dropdown && currentContentSortBy !== 'default') {
       dropdown.value = `${currentContentSortBy}-${currentContentSortDirection}`;
       setTimeout(() => sortContentGrid(), 100);
   }
}

/**
* Create content card for grid view with media type and stories support
*/

function createContentCard(content) {
    const card = document.createElement('div');
    card.className = 'content-card';
    card.dataset.contentId = content.id;
    card.dataset.title = content.title.toLowerCase();
    card.dataset.platform = content.platform.toLowerCase();
    card.dataset.influencer = content.influencer.toLowerCase();
    card.dataset.mediaType = content.mediaType || '';
    
    // Check if this is Instagram Stories or Posts
    const isInstagramStories = content.platform.toLowerCase() === 'instagram' && 
                              content.mediaType === 'story';
    const isInstagramPosts = content.platform.toLowerCase() === 'instagram' && 
                            content.mediaType === 'post';
    
    // Calculate individual ER for this content piece
    let individualER = 0;
    if (content.metrics.views > 0 && !isInstagramPosts) {
        const totalEngagement = (content.metrics.likes || 0) + 
                               (content.metrics.comments || 0) + 
                               (content.metrics.shares || 0) + 
                               (content.metrics.saves || 0) + 
                               (content.analytics.stickerTaps || 0);
        individualER = (totalEngagement / content.metrics.views) * 100;
    }
    
    // Make card clickable if video link exists
    if (content.videoLink && content.videoLink.trim() !== '') {
        card.style.cursor = 'pointer';
        card.onclick = () => window.open(content.videoLink, '_blank');
    }
    
    // Thumbnail with fallback and play overlay - different for stories
    let thumbnailHTML;
    if (isInstagramStories) {
        // Instagram Stories - no thumbnail, show stories icon and count
        thumbnailHTML = `
            <div class="stories-thumbnail-container">
                <div class="stories-icon-container">
                    <span class="stories-icon">📖</span>
                    <div class="stories-count">${content.analytics.stories || 1}</div>
                    <div class="stories-label">Stories</div>
                </div>
            </div>
        `;
    } else {
        // Regular content with thumbnail
        if (content.thumbnail && content.thumbnail.trim() !== '') {
            const proxiedThumbnail = safeImageUrl(content.thumbnail);
            thumbnailHTML = `
                <img src="${proxiedThumbnail}" 
                     alt="${content.title}" 
                     class="content-thumbnail"
                     onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                <div class="content-thumbnail-fallback" style="display:none;">
                    <span class="play-icon">▶</span>
                </div>
                ${content.videoLink ? '<div class="video-overlay"><div class="play-button">▶</div></div>' : ''}
            `;
        } else {
            thumbnailHTML = `
                <div class="content-thumbnail-fallback">
                    <span class="play-icon">▶</span>
                </div>
                ${content.videoLink ? '<div class="video-overlay"><div class="play-button">▶</div></div>' : ''}
            `;
        }
    }
    
    // Create metrics HTML - only show non-zero metrics
    const metrics = [
        { key: 'views', icon: '👁', label: 'Views' },
        { key: 'likes', icon: '❤️', label: 'Likes' },
        { key: 'comments', icon: '💬', label: isInstagramStories ? 'Replies' : 'Comments' },
        { key: 'shares', icon: '📤', label: 'Shares' },
        { key: 'saves', icon: '🔖', label: 'Saves' }
    ];
    
    const visibleMetrics = metrics
        .filter(metric => content.metrics[metric.key] > 0)
        .map(metric => `
            <div class="content-metric">
                <span class="metric-icon">${metric.icon}</span>
                <span class="metric-value">${formatNumber(content.metrics[metric.key])}</span>
                <span class="metric-label">${metric.label}</span>
            </div>
        `).join('');
    
    // Add stories-specific metrics if it's Instagram Stories
    let storiesSpecificMetrics = '';
    if (isInstagramStories && content.analytics.stickerTaps > 0) {
        storiesSpecificMetrics = `
            <div class="content-metric">
                <span class="metric-icon">👆</span>
                <span class="metric-value">${formatNumber(content.analytics.stickerTaps)}</span>
                <span class="metric-label">Sticker Taps</span>
            </div>
        `;
    }
    
    // Add link clicks metric for Instagram (all types)
    let linkClicksMetricHTML = '';
    if (content.analytics.linkClicks > 0) {
        linkClicksMetricHTML = `
            <div class="content-metric">
                <span class="metric-icon">🔗</span>
                <span class="metric-value">${formatNumber(content.analytics.linkClicks)}</span>
                <span class="metric-label">Link Clicks</span>
            </div>
        `;
    }
    
    // Add individual ER metric (exclude Instagram Posts and Instagram Stories)
    let erMetricHTML = '';
    if (!isInstagramPosts && !isInstagramStories && individualER > 0) {
        erMetricHTML = `
            <div class="content-metric">
                <span class="metric-icon">📈</span>
                <span class="metric-value">${individualER.toFixed(2)}%</span>
                <span class="metric-label">ER</span>
            </div>
        `;
    }
    
    // Create platform badge with proper content type display
    let platformBadgeText = content.platform;
    if (content.mediaType) {
        if (content.mediaType === 'post') {
            platformBadgeText += ' Post';
        } else if (content.mediaType === 'reel') {
            platformBadgeText += ' Reel';
        } else if (content.mediaType === 'story') {
            platformBadgeText += ' Stories';
        } else if (content.mediaType === 'integration') {
            platformBadgeText += ' Integration';
        } else if (content.mediaType === 'short') {
            platformBadgeText += ' Shorts';
        } else if (content.mediaType === 'video') {
            platformBadgeText += ' Video';
        }
    }
    
    // Add bonus label if applicable
    const bonusLabel = content.bonus ? '<span class="bonus-label">Bonus content</span>' : '';
    
    card.innerHTML = `
        <div class="content-thumbnail-container">
            ${thumbnailHTML}
        </div>
        <div class="content-info">
            <h3 class="content-title">${escapeHtml(content.title)}</h3>
            <div class="content-creator">
                <span>${escapeHtml(content.influencer)}</span>
                <span class="creator-followers">${content.followers > 0 ? `• ${formatNumber(content.followers)} followers` : ''}</span>
            </div>
            <div class="content-date">${content.postedDate}</div>
        </div>
        <div class="content-footer">
            <span class="platform-badge platform-${content.platform.toLowerCase()} content-platform-badge">
                ${platformBadgeText}
            </span>
            ${bonusLabel}
        </div>
        <div class="content-metrics-expanded">
            ${visibleMetrics}
            ${storiesSpecificMetrics}
            ${linkClicksMetricHTML}
            ${erMetricHTML}
        </div>
    `;
    
    return card;
}

// =============================================================================
// TABLE FUNCTIONS
// =============================================================================

/**
* Sort content table by column
*/
function sortContentTable(column) {
   // Toggle sort direction if clicking the same column
   if (currentSortColumn === column) {
       currentSortDirection = currentSortDirection === 'asc' ? 'desc' : 'asc';
   } else {
       currentSortColumn = column;
       currentSortDirection = 'desc';
   }
   
   // Clear all sort indicators
   document.querySelectorAll('.sort-indicator').forEach(indicator => {
       indicator.textContent = '';
   });
   
   // Set current sort indicator
   const indicator = document.getElementById(`sort-${column}`);
   if (indicator) {
       indicator.textContent = currentSortDirection === 'asc' ? '↑' : '↓';
   }
   
   // Get visible content based on current filters
   let visibleContent = getVisibleContentForTable();
   
   // Sort the visible content
   visibleContent.sort((a, b) => {
       let valueA, valueB;
       
       switch (column) {
           case 'title':
               valueA = a.title.toLowerCase();
               valueB = b.title.toLowerCase();
               break;
           case 'influencer':
               valueA = a.influencer.toLowerCase();
               valueB = b.influencer.toLowerCase();
               break;
           case 'platform':
               valueA = a.platform.toLowerCase();
               valueB = b.platform.toLowerCase();
               break;
           case 'postedDate':
               valueA = new Date(a.postedDate);
               valueB = new Date(b.postedDate);
               break;
           case 'views':
               valueA = a.metrics.views || 0;
               valueB = b.metrics.views || 0;
               break;
           case 'likes':
               valueA = a.metrics.likes || 0;
               valueB = b.metrics.likes || 0;
               break;
           case 'comments':
               valueA = a.metrics.comments || 0;
               valueB = b.metrics.comments || 0;
               break;
           case 'shares':
               valueA = a.metrics.shares || 0;
               valueB = b.metrics.shares || 0;
               break;
           case 'saves':
               valueA = a.metrics.saves || 0;
               valueB = b.metrics.saves || 0;
               break;
           case 'reach':
               valueA = a.metrics.reach || 0;
               valueB = b.metrics.reach || 0;
               break;
           case 'stickerTaps':
               valueA = a.analytics.stickerTaps || 0;
               valueB = b.analytics.stickerTaps || 0;
               break;
           case 'linkClicks':
               valueA = a.analytics.linkClicks || 0;
               valueB = b.analytics.linkClicks || 0;
               break;
           default:
               return 0;
       }
       
       // Handle different data types
       if (typeof valueA === 'string') {
           if (currentSortDirection === 'asc') {
               return valueA.localeCompare(valueB);
           } else {
               return valueB.localeCompare(valueA);
           }
       } else {
           if (currentSortDirection === 'asc') {
               return valueA - valueB;
           } else {
               return valueB - valueA;
           }
       }
   });
   
   // Store sorted data
   sortedContentData = visibleContent;
   
   // Re-render the table with sorted data
   renderSortedContentTable(visibleContent);
}

/**
* Get visible content based on current filters
*/
function getVisibleContentForTable() {
   const searchTerm = document.getElementById('contentSearchInput').value.toLowerCase();
   const platformFilter = document.getElementById('contentPlatformFilter').value;
   const mediaTypeFilter = document.getElementById('contentMediaTypeFilter').value;
   
   return contentData.filter(content => {
       const title = content.title.toLowerCase();
       const platform = content.platform.toLowerCase();
       const influencer = content.influencer.toLowerCase();
       const mediaType = content.mediaType || '';
       
       const searchMatch = !searchTerm || title.includes(searchTerm) || influencer.includes(searchTerm);
       const platformMatch = !platformFilter || platform === platformFilter.toLowerCase();
       const mediaTypeMatch = !mediaTypeFilter || mediaType === mediaTypeFilter;
       
       return searchMatch && platformMatch && mediaTypeMatch;
   });
}

/**
* Render sorted content table
*/
function renderSortedContentTable(sortedData) {
   const tableContent = document.getElementById('contentTableContent');
   if (!tableContent) return;
   
   tableContent.innerHTML = '';
   
   if (sortedData.length === 0) {
       tableContent.innerHTML = `
           <div class="content-table-row">
               <div style="grid-column: 1/-1; text-align: center; padding: 2rem; color: #64748b;">
                   <h3>No content data found</h3>
               </div>
           </div>
       `;
       return;
   }
   
   sortedData.forEach(content => {
       const row = createContentTableRowDetailed(content);
       tableContent.appendChild(row);
   });
}

/**
* Create detailed content table row with separate columns for each metric
*/
function createContentTableRowDetailed(content) {
   const row = document.createElement('div');
   row.className = 'content-table-row';
   row.dataset.contentId = content.id; // Add unique ID
   row.dataset.title = content.title.toLowerCase();
   row.dataset.platform = content.platform.toLowerCase();
   row.dataset.influencer = content.influencer.toLowerCase();
   row.dataset.mediaType = content.mediaType || '';
   
   // Create clickable title if video link exists
   let titleHTML;
   if (content.videoLink && content.videoLink.trim() !== '') {
       titleHTML = `
           <a href="${content.videoLink}" target="_blank" class="table-content-title-link">
               ${escapeHtml(content.title)}
               <span class="external-link-icon">🔗</span>
           </a>
       `;
   } else {
       titleHTML = `<div class="table-content-title">${escapeHtml(content.title)}</div>`;
   }
   
   // Create platform badge with media type
   let platformBadgeText = content.platform;
   if (content.mediaType) {
       if (content.mediaType === 'post') {
           platformBadgeText += ' Post';
       } else if (content.mediaType === 'reel') {
           platformBadgeText += ' Reel';
       } else if (content.mediaType === 'story') {
           platformBadgeText += ' Stories';
       } else if (content.mediaType === 'integration') {
           platformBadgeText += ' Integration';
       } else if (content.mediaType === 'short') {
           platformBadgeText += ' Shorts';
       } else if (content.mediaType === 'video') {
           platformBadgeText += ' Video';
       }
   }
   
   // Add bonus label if applicable
   const bonusLabel = content.bonus ? '<span class="bonus-label">Bonus</span>' : '';
   
   // Helper function to format metric values
   const formatMetric = (value) => {
       return value > 0 ? formatNumber(value) : '-';
   };
   
   row.innerHTML = `
       ${titleHTML}
       <div class="table-content-creator">${escapeHtml(content.influencer)}</div>
       <div>
           <span class="table-platform-badge platform-${content.platform.toLowerCase()}">
               ${platformBadgeText}
           </span>
           ${bonusLabel}
       </div>
       <div class="table-content-date">${content.postedDate}</div>
       <div class="table-metric">${formatMetric(content.metrics.views)}</div>
       <div class="table-metric">${formatMetric(content.metrics.likes)}</div>
       <div class="table-metric">${formatMetric(content.metrics.comments)}</div>
       <div class="table-metric">${formatMetric(content.metrics.shares)}</div>
       <div class="table-metric">${formatMetric(content.metrics.saves)}</div>
       <div class="table-metric">${formatMetric(content.analytics.stickerTaps)}</div>
       <div class="table-metric">${formatMetric(content.analytics.linkClicks)}</div>
   `;
   
   return row;
}

/**
* Update content table view
*/
function updateContentTable() {
   // If we have sorted data, use that; otherwise use original data
   const dataToRender = sortedContentData.length > 0 ? sortedContentData : contentData;
   
   const tableContent = document.getElementById('contentTableContent');
   if (!tableContent) return;
   
   tableContent.innerHTML = '';
   
   if (dataToRender.length === 0) {
       tableContent.innerHTML = `
           <div class="content-table-row">
               <div style="grid-column: 1/-1; text-align: center; padding: 2rem; color: #64748b;">
                   <h3>No content data found</h3>
               </div>
           </div>
       `;
       return;
   }
   
   dataToRender.forEach(content => {
       const row = createContentTableRowDetailed(content);
       tableContent.appendChild(row);
   });
}

// =============================================================================
// VIEW SWITCHING FUNCTIONS
// =============================================================================

/**
* Switch to grid view
*/
function switchToGridView() {
   currentView = 'grid';
   
   // Update button states
   document.getElementById('gridViewBtn').classList.add('active');
   document.getElementById('tableViewBtn').classList.remove('active');
   
   // Show/hide views
   document.getElementById('influencersGrid').style.display = 'grid';
   document.getElementById('influencersTable').style.display = 'none';
}

/**
* Switch to table view
*/
function switchToTableView() {
   currentView = 'table';
   
   // Update button states
   document.getElementById('gridViewBtn').classList.remove('active');
   document.getElementById('tableViewBtn').classList.add('active');
   
   // Show/hide views
   document.getElementById('influencersGrid').style.display = 'none';
   document.getElementById('influencersTable').style.display = 'block';
   
   // Update table content
   updateInfluencersTable();
}

/**
* Switch to content grid view
*/
function switchToContentGridView() {
   currentContentView = 'grid';
   
   document.getElementById('contentGridViewBtn').classList.add('active');
   document.getElementById('contentTableViewBtn').classList.remove('active');
   
   document.getElementById('contentGrid').style.display = 'grid';
   document.getElementById('contentTable').style.display = 'none';
}

/**
* Switch to content table view
*/
function switchToContentTableView() {
   currentContentView = 'table';
   
   document.getElementById('contentGridViewBtn').classList.remove('active');
   document.getElementById('contentTableViewBtn').classList.add('active');
   
   document.getElementById('contentGrid').style.display = 'none';
   document.getElementById('contentTable').style.display = 'block';
}

// =============================================================================
// FILTER FUNCTIONS
// =============================================================================

/**
* Filter influencers based on search and platform selection
*/
function filterInfluencers() {
   const searchTerm = document.getElementById('searchInput').value.toLowerCase();
   const platformFilter = document.getElementById('platformFilter').value;
   
   if (currentView === 'grid') {
       // Filter grid view
       const cards = document.querySelectorAll('.influencer-card');
       
       cards.forEach(card => {
           const name = card.dataset.name;
           const platform = card.dataset.platform;
           
           const searchMatch = !searchTerm || name.includes(searchTerm);
           const platformMatch = !platformFilter || platform.toLowerCase() === platformFilter.toLowerCase();
           
           if (searchMatch && platformMatch) {
               card.style.display = 'block';
           } else {
               card.style.display = 'none';
           }
       });
       
   } else {
       // Filter table view
       const rows = document.querySelectorAll('.table-row');
       
       rows.forEach(tableRow => {
           // Skip header row
           if (!tableRow.dataset.profileLink) return;
           
           const name = tableRow.dataset.name;
           const platform = tableRow.dataset.platform;
           
           const searchMatch = !searchTerm || name.includes(searchTerm);
           const platformMatch = !platformFilter || platform.toLowerCase() === platformFilter.toLowerCase();
           
           if (searchMatch && platformMatch) {
               tableRow.style.display = '';
           } else {
               tableRow.style.display = 'none';
           }
       });
   }
   
   // Update metrics summary
   updateMetricsSummary();
}

/**
* Filter content based on search and platform selection
*/
function filterContent() {
   const searchTerm = document.getElementById('contentSearchInput').value.toLowerCase();
   const platformFilter = document.getElementById('contentPlatformFilter').value;
   const mediaTypeFilter = document.getElementById('contentMediaTypeFilter').value;
   
   // Clear sorted data when filtering
   sortedContentData = [];
   currentSortColumn = '';
   currentSortDirection = 'desc';
   
   // Clear sort indicators
   document.querySelectorAll('.sort-indicator').forEach(indicator => {
       indicator.textContent = '';
   });
   
   // Just update the metrics summary - it will handle all the filtering logic
   updateContentMetricsSummary();
   
   // Update visual display based on current view
   if (currentContentView === 'grid') {
       const cards = document.querySelectorAll('.content-card');
       cards.forEach(card => {
           const contentId = card.dataset.contentId;
           const title = card.dataset.title;
           const platform = card.dataset.platform;
           const influencer = card.dataset.influencer;
           const mediaType = card.dataset.mediaType || '';
           
           // Use ID for lookup if available, fallback to title
           const contentItem = contentId 
               ? contentData.find(c => c.id == contentId)
               : contentData.find(c => c.title.toLowerCase() === title);
           const isBonus = contentItem ? contentItem.bonus === true : false;
                      
           const searchMatch = !searchTerm || title.includes(searchTerm) || influencer.includes(searchTerm);
           
           let platformMatch = true;
           if (platformFilter === 'bonus') {
               platformMatch = isBonus === true;
           } else if (platformFilter) {
               platformMatch = platform.toLowerCase() === platformFilter.toLowerCase();
           }
           
           const mediaTypeMatch = !mediaTypeFilter || mediaType === mediaTypeFilter;
           
           card.style.display = (searchMatch && platformMatch && mediaTypeMatch) ? 'block' : 'none';
       });
   } else {
       // Similar logic for table view
       const rows = document.querySelectorAll('.content-table-row');
       rows.forEach(row => {
           const contentId = row.dataset.contentId;
           const title = row.dataset.title;
           const platform = row.dataset.platform;
           const influencer = row.dataset.influencer;
           const mediaType = row.dataset.mediaType || '';
           
           // Use ID for lookup if available, fallback to title
           const contentItem = contentId 
               ? contentData.find(c => c.id == contentId)
               : contentData.find(c => c.title.toLowerCase() === title);
           const isBonus = contentItem ? contentItem.bonus === true : false;
           
           const searchMatch = !searchTerm || title.includes(searchTerm) || influencer.includes(searchTerm);
           
           let platformMatch = true;
           if (platformFilter === 'bonus') {
               platformMatch = isBonus === true;
           } else if (platformFilter) {
               platformMatch = platform.toLowerCase() === platformFilter.toLowerCase();
           }
           
           const mediaTypeMatch = !mediaTypeFilter || mediaType === mediaTypeFilter;
           
           row.style.display = (searchMatch && platformMatch && mediaTypeMatch) ? '' : 'none';
       });
   }
}

/**
* Update content metrics summary with platform-specific conditional logic
* ER calculation: Sum all engagement metrics, then divide by sum of views for ALL platforms
*/
function updateContentMetricsSummary() {
   // Get filter values directly
   const searchTerm = document.getElementById('contentSearchInput').value.toLowerCase();
   const platformFilter = document.getElementById('contentPlatformFilter').value;
   const mediaTypeFilter = document.getElementById('contentMediaTypeFilter').value;
   
   // Filter content data based on current filters
   let visibleContent = contentData.filter(content => {
       const title = content.title.toLowerCase();
       const platform = content.platform.toLowerCase();
       const influencer = content.influencer.toLowerCase();
       const mediaType = content.mediaType || '';
       
       // Search match
       const searchMatch = !searchTerm || 
           title.includes(searchTerm) || 
           influencer.includes(searchTerm);
       
       // Platform match
       let platformMatch = true;
       if (platformFilter === 'bonus') {
           platformMatch = content.bonus === true;
       } else if (platformFilter) {
           platformMatch = platform === platformFilter.toLowerCase();
       }
       
       // Media type match
       const mediaTypeMatch = !mediaTypeFilter || mediaType === mediaTypeFilter;
       
       return searchMatch && platformMatch && mediaTypeMatch;
   });
   
   // Determine platform-specific display logic
   const isInstagramStoriesOnly = platformFilter === 'Instagram' && mediaTypeFilter === 'story';
   const isInstagramPostsOnly = platformFilter === 'Instagram' && mediaTypeFilter === 'post';
   const isYouTubeOnly = platformFilter === 'YouTube';
   const isTikTokOnly = platformFilter === 'TikTok';
   const isInstagramOnly = platformFilter === 'Instagram' && !mediaTypeFilter;
   const isAllPlatforms = !platformFilter || platformFilter === '';
   
   // Calculate pieces of content with special logic for Instagram Stories
   let piecesOfContent = 0;
   
   if (isYouTubeOnly || isTikTokOnly) {
       piecesOfContent = visibleContent.length;
   } else {
       visibleContent.forEach(content => {
           const isInstagramStories = content.platform.toLowerCase() === 'instagram' && 
                                     content.mediaType === 'story';
           
           if (isInstagramStories) {
               piecesOfContent += content.analytics.stories || 1;
           } else {
               piecesOfContent += 1;
           }
       });
   }
   
   // Calculate totals with ER calculation for ALL platforms using sum of metrics
   const totals = visibleContent.reduce((acc, content) => {
       acc.views += content.metrics.views || 0;
       acc.likes += content.metrics.likes || 0;
       acc.comments += content.metrics.comments || 0;
       acc.shares += content.metrics.shares || 0;
       acc.saves += content.metrics.saves || 0;
       acc.reach += content.metrics.reach || 0;
       acc.linkClicks += content.analytics.linkClicks || 0; // Add link clicks to totals
       
       // For Instagram Stories: collect sticker taps instead of saves
       if (isInstagramStoriesOnly) {
           acc.stickerTaps += content.analytics.stickerTaps || 0;
       }
       
       // Calculate total engagement for ER (exclude only Instagram Posts from ER calculation)
       const isInstagramPosts = content.platform.toLowerCase() === 'instagram' && 
                                content.mediaType === 'post';
       
       if (!isInstagramPosts) {
           acc.totalEngagement += (content.metrics.likes || 0) + 
                                 (content.metrics.comments || 0) + 
                                 (content.metrics.shares || 0) + 
                                 (content.metrics.saves || 0) + 
                                 (content.analytics.stickerTaps || 0);
           acc.totalViewsForER += content.metrics.views || 0;
       }
       
       return acc;
   }, {
       views: 0,
       likes: 0,
       comments: 0,
       shares: 0,
       saves: 0,
       reach: 0,
       linkClicks: 0, // Initialize link clicks
       stickerTaps: 0,
       totalEngagement: 0,
       totalViewsForER: 0
   });
   
   // Calculate ER using total engagement / total views for ALL platforms
   let calculatedER = 0;
   if (totals.totalViewsForER > 0) {
       calculatedER = (totals.totalEngagement / totals.totalViewsForER) * 100;
   }
   
   // Update display elements
   const updateElement = (id, value) => {
       const element = document.getElementById(id);
       if (element) element.textContent = formatNumber(value);
   };
   
   // Update labels and show/hide metrics based on platform selection
   
   // 1. Comments/Replies label for Instagram Stories
   const commentsLabel = document.getElementById('commentsLabel');
   if (commentsLabel) {
       commentsLabel.textContent = isInstagramStoriesOnly ? 'Total Replies' : 'Total Comments';
   }
   
   // 2. Saves/Sticker Taps label and value for Instagram Stories
   const savesLabel = document.getElementById('savesLabel');
   const savesElement = document.getElementById('contentTotalSaves');
   if (savesLabel && savesElement) {
       if (isInstagramStoriesOnly) {
           savesLabel.textContent = 'Sticker Taps';
           savesElement.textContent = formatNumber(totals.stickerTaps);
       } else {
           savesLabel.textContent = 'Total Saves';
           savesElement.textContent = formatNumber(totals.saves);
       }
   }
   
   // 3. Show/hide metrics based on platform and conditions
   
   // Total Views: Hide for Instagram Posts only
   const totalViewsElement = document.querySelector('[data-metric="totalViews"]') || document.getElementById('contentTotalViews')?.parentElement;
   if (totalViewsElement) {
       totalViewsElement.style.display = isInstagramPostsOnly ? 'none' : 'block';
   }
   
   // Total Reach: Only show for Instagram Stories
   const totalReachElement = document.querySelector('[data-metric="totalReach"]');
   if (totalReachElement) {
       totalReachElement.style.display = isInstagramStoriesOnly ? 'block' : 'none';
   }
   
   // Total Impressions: Hide completely
   const totalImpressionsElement = document.querySelector('[data-metric="totalImpressions"]');
   if (totalImpressionsElement) {
       totalImpressionsElement.style.display = 'none';
   }
   
   // Total Shares: Hide if value is 0 or for YouTube when empty
   const totalSharesElement = document.querySelector('[data-metric="totalShares"]');
   if (totalSharesElement) {
       if (totals.shares === 0 || (isYouTubeOnly && totals.shares === 0)) {
           totalSharesElement.style.display = 'none';
       } else {
           totalSharesElement.style.display = 'block';
       }
   }
   
   // Total Saves: Hide if value is 0 or for YouTube
   const totalSavesElement = document.querySelector('[data-metric="totalSaves"]');
   if (totalSavesElement) {
       const hideCondition = isInstagramStoriesOnly ? false : (totals.saves === 0 || isYouTubeOnly);
       totalSavesElement.style.display = hideCondition ? 'none' : 'block';
   }
   
   // Total Link Clicks: Show when there are link clicks for ALL platforms, Instagram, or TikTok
   const totalLinkClicksElement = document.querySelector('[data-metric="totalLinkClicks"]');
   if (totalLinkClicksElement) {
       totalLinkClicksElement.style.display = totals.linkClicks > 0 ? 'block' : 'none';
   }
   
   // ER: Hide for Instagram Stories only and Instagram Posts only, show for Instagram overall (reels + stories)
   const totalERElement = document.querySelector('[data-metric="totalER"]');
   if (totalERElement) {
       if (isInstagramStoriesOnly || isInstagramPostsOnly) {
           totalERElement.style.display = 'none';
       } else {
           totalERElement.style.display = totals.totalViewsForER > 0 ? 'block' : 'none';
       }
   }
   
   // Update all metric values
   updateElement('contentCount', piecesOfContent);
   updateElement('contentTotalViews', totals.views);
   updateElement('contentTotalLikes', totals.likes);
   updateElement('contentTotalComments', totals.comments);
   updateElement('contentTotalShares', totals.shares);
   updateElement('contentTotalReach', totals.reach);
   updateElement('contentTotalLinkClicks', totals.linkClicks); // Update link clicks value
   
   // Update ER with proper formatting
   const erElement = document.getElementById('contentTotalER');
   if (erElement) {
       erElement.textContent = calculatedER > 0 ? `${calculatedER.toFixed(2)}%` : '0%';
   }
}

// =============================================================================
// PAID MEDIA TAB FUNCTIONS
// =============================================================================

// Global variables for Paid Media
let paidMediaData = {
    tiktok: {},
    instagram: {},
    youtube: {},
    findings: '',
    recommendations: ''
};
let currentPaidMediaFilter = null;

/**
 * Load Paid Media data from Google Sheets Paid tab
 */
async function loadPaidMediaData() {
    try {
        console.log('Loading Paid Media data from sheet:', currentCampaignId);
        
        // Check if the Paid tab exists
        const paidTabExists = await checkSheetTabExists(currentCampaignId, 'Paid');
        
        if (!paidTabExists) {
            console.log('Paid tab not found - hiding Paid Media section');
            hidePaidMediaTab();
            return;
        }
        
        console.log('Paid tab found - loading paid media data');
        showPaidMediaTab();
        
        // Fetch data from the Paid sheet - columns A to BH (60 columns)
        const data = await fetchSheetData(currentCampaignId, 'Paid', 'A1:BH3');
        
        if (data && data.length >= 3) {
            // Process the data
            processPaidMediaData(data);
            
            // Display the data
            updatePaidMediaDisplay();
        } else {
            console.warn('No paid media data found or insufficient rows in Paid sheet');
            hidePaidMediaTab();
        }
        
    } catch (error) {
        console.error('Error loading paid media data:', error);
        hidePaidMediaTab();
    }
}

/**
 * Process Paid Media data from the sheet
 */
function processPaidMediaData(data) {
    const headers = data[1] || [];  // Row 2 (index 1)
    const values = data[2] || [];   // Row 3 (index 2)
    
    console.log('Processing paid media data');
    
    // TikTok data: columns A-T (indices 0-19)
    for (let i = 0; i <= 19; i++) {
        if (headers[i] && values[i] !== undefined && values[i] !== '') {
            const key = headers[i].toString().trim();
            paidMediaData.tiktok[key] = values[i];
        }
    }
    
    // Instagram data: columns U-AN (indices 20-39)
    for (let i = 20; i <= 39; i++) {
        if (headers[i] && values[i] !== undefined && values[i] !== '') {
            const key = headers[i].toString().trim();
            paidMediaData.instagram[key] = values[i];
        }
    }
    
    // YouTube data: columns AO-BH (indices 40-59)
    for (let i = 40; i <= 59; i++) {
        if (headers[i] && values[i] !== undefined && values[i] !== '') {
            const key = headers[i].toString().trim();
            paidMediaData.youtube[key] = values[i];
        }
    }
    
    // Set initial filter to first available platform with data
    if (!currentPaidMediaFilter) {
        if (Object.keys(paidMediaData.tiktok).length > 0) {
            currentPaidMediaFilter = 'tiktok';
        } else if (Object.keys(paidMediaData.instagram).length > 0) {
            currentPaidMediaFilter = 'instagram';
        } else if (Object.keys(paidMediaData.youtube).length > 0) {
            currentPaidMediaFilter = 'youtube';
        }
    }
    
    console.log('Paid media data processed:', paidMediaData);
    console.log('Initial filter set to:', currentPaidMediaFilter);
}


/**
 * Update Paid Media display based on current filter
 */
function updatePaidMediaDisplay() {
    // First, update the filter buttons to show only platforms with data
    updatePaidMediaFilterButtons();
    
    const container = document.getElementById('paidMediaMetrics');
    if (!container) return;
    
    container.innerHTML = '';
    
    // Use specific platform data only
    const dataToDisplay = paidMediaData[currentPaidMediaFilter] || {};
    
    // Check if there's any data to display
    if (Object.keys(dataToDisplay).length === 0) {
        container.innerHTML = `
            <div class="empty-state" style="grid-column: 1/-1;">
                <h3>No paid media data available</h3>
                <p>No data found for the selected platform</p>
            </div>
        `;
        // Still update analysis cards even if no metrics
        updatePaidMediaAnalysisCards();
        return;
    }
    
    // Create cards for each metric, but only add non-null cards
    let hasVisibleCards = false;
    Object.entries(dataToDisplay).forEach(([metric, value]) => {
        const card = createPaidMediaCard(metric, value);
        if (card !== null) {
            container.appendChild(card);
            hasVisibleCards = true;
        }
    });
    
    // If no cards were added (all were zero), show empty state
    if (!hasVisibleCards) {
        container.innerHTML = `
            <div class="empty-state" style="grid-column: 1/-1;">
                <h3>No paid media data available</h3>
                <p>All metrics are zero for the selected platform</p>
            </div>
        `;
    }
    
    // Update the analysis cards (findings and recommendations)
    updatePaidMediaAnalysisCards();
}

/**
 * Update Paid Media analysis cards (Findings and Recommendations)
 */
function updatePaidMediaAnalysisCards() {
    const findingsElement = document.getElementById('paidMediaFindings');
    const recommendationsElement = document.getElementById('paidMediaRecommendations');
    
    if (findingsElement) {
        const findings = paidMediaData.findings || 'No findings available.';
        findingsElement.textContent = findings;
    }
    
    if (recommendationsElement) {
        const recommendations = paidMediaData.recommendations || 'No recommendations available.';
        recommendationsElement.textContent = recommendations;
    }
    
    console.log('Updated Paid Media analysis cards');
}

/**
 * Update paid media filter buttons to show only platforms with data
 */
function updatePaidMediaFilterButtons() {
    const filtersContainer = document.getElementById('paidMediaFilters');
    if (!filtersContainer) return;
    
    filtersContainer.innerHTML = '';
    
    // Check which platforms have data
    const platformsWithData = [];
    
    if (Object.keys(paidMediaData.tiktok).length > 0) {
        platformsWithData.push({ id: 'tiktok', label: 'TikTok' });
    }
    if (Object.keys(paidMediaData.instagram).length > 0) {
        platformsWithData.push({ id: 'instagram', label: 'Instagram' });
    }
    if (Object.keys(paidMediaData.youtube).length > 0) {
        platformsWithData.push({ id: 'youtube', label: 'YouTube' });
    }
    
    // Create buttons for platforms with data
    platformsWithData.forEach((platform, index) => {
        const button = document.createElement('button');
        button.className = 'platform-tab';
        button.textContent = platform.label;
        button.onclick = () => filterPaidMedia(platform.id);
        
        // Set first button as active if current filter matches or is not set
        if (platform.id === currentPaidMediaFilter) {
            button.classList.add('active');
        }
        
        filtersContainer.appendChild(button);
    });
}

/**
/**
 * Create a paid media metric card (returns null if value is 0)
 */
function createPaidMediaCard(metric, value) {
    // Parse the value to check if it's zero
    const numericValue = parseFloat(value.toString().replace(/[^0-9.\-]/g, '')) || 0;
    
    // Return null if value is 0
    if (numericValue === 0) {
        return null;
    }
    
    const card = document.createElement('div');
    card.className = 'paid-media-stat-box';
    
    // Choose an appropriate icon based on metric name
    const icon = getPaidMediaIcon(metric);
    
    // Format the label
    const label = formatPaidMediaLabel(metric);
    
    card.innerHTML = `
        <div class="paid-media-stat-icon">${icon}</div>
        <div class="paid-media-stat-label">${label}</div>
        <div class="paid-media-stat-value">${value}</div>
    `;
    
    return card;
}

/**
 * Get icon for paid media metric
 */
function getPaidMediaIcon(metric) {
    const metricLower = metric.toLowerCase();
    
    if (metricLower.includes('spend')) return '💰';
    if (metricLower.includes('cpc')) return '💲';
    if (metricLower.includes('impression')) return '👁';
    if (metricLower.includes('click')) return '🖱️';
    if (metricLower.includes('ctr')) return '📊';
    if (metricLower.includes('engagement')) return '💬';
    if (metricLower.includes('er')) return '📈';
    if (metricLower.includes('conversion')) return '🎯';
    if (metricLower.includes('cpa')) return '💸';
    if (metricLower.includes('cvr')) return '🔄';
    if (metricLower.includes('view')) return '👀';
    if (metricLower.includes('reach')) return '📡';
    if (metricLower.includes('frequency')) return '🔁';
    if (metricLower.includes('roi')) return '📊';
    if (metricLower.includes('roas')) return '💹';
    
    return '📊'; // Default icon
}

/**
 * Format paid media label for display
 */
function formatPaidMediaLabel(metric) {
    // Return the metric as-is since it's from the header
    return metric;
}

/**
 * Format paid media numbers
 */
function formatPaidMediaNumber(num) {
    if (typeof num !== 'number' || isNaN(num)) return '0';
    
    // For very large numbers
    if (num >= 1000000) {
        return (num / 1000000).toFixed(1) + 'M';
    }
    if (num >= 1000) {
        return (num / 1000).toFixed(1) + 'K';
    }
    
    // For decimal numbers
    if (num < 1 && num > 0) {
        return num.toFixed(2);
    }
    
    // For regular numbers
    return Math.round(num).toLocaleString();
}

/**
 * Filter paid media by platform
 */
function filterPaidMedia(platform) {
    currentPaidMediaFilter = platform;
    
    // Update active tab
    const buttons = document.querySelectorAll('#paidMediaFilters .platform-tab');
    buttons.forEach(btn => {
        btn.classList.remove('active');
        if (btn.textContent.toLowerCase().includes(platform)) {
            btn.classList.add('active');
        }
    });
    
    // Update display
    updatePaidMediaDisplay();
}

/**
 * Hide the Paid Media tab
 */
function hidePaidMediaTab() {
    const paidMediaTab = document.getElementById('paidMedia');
    if (paidMediaTab) {
        paidMediaTab.style.display = 'none';
        paidMediaTab.classList.remove('active');
    }
    
    const paidMediaNavItem = document.querySelector('.nav-item[onclick="showTab(\'paidMedia\')"]');
    if (paidMediaNavItem) {
        paidMediaNavItem.style.display = 'none';
    }
    
    console.log('Paid Media tab hidden due to no content');
}

/**
 * Show the Paid Media tab
 */
function showPaidMediaTab() {
    const paidMediaTab = document.getElementById('paidMedia');
    if (paidMediaTab) {
        paidMediaTab.style.display = '';
    }
    
    const paidMediaNavItem = document.querySelector('.nav-item[onclick="showTab(\'paidMedia\')"]');
    if (paidMediaNavItem) {
        paidMediaNavItem.style.display = '';
    }
    
    console.log('Paid Media tab shown with content');
}

/**
* Show specific tab
*/
function showTab(tabName) {
   // Hide all tab content
   document.querySelectorAll('.tab-content').forEach(tab => {
       tab.classList.remove('active');
       tab.style.display = 'none';
   });
   
   // Remove active class from all nav items
   document.querySelectorAll('.nav-item').forEach(item => {
       item.classList.remove('active');
   });
   
   // Show selected tab
   const selectedTab = document.getElementById(tabName);
   if (selectedTab) {
       selectedTab.classList.add('active');
       selectedTab.style.display = 'block';
   }
   
   // Add active class to clicked nav item
   if (event && event.target) {
       event.target.classList.add('active');
   }
   
   // Update displays when tabs are shown
   if (tabName === 'overview') {
       updateOverviewDisplay();
   } else if (tabName === 'results') {
       // Initialize results filters if needed
       renderPlatformFilters();
       renderSubFilters(currentPlatform);
       updateResultsDisplay();
   }
}

/**
* Refresh the dashboard data
*/
function refreshData() {
   loadInfluencerData();
   loadContentData();
   loadWeeklyData();
}

// =============================================================================
// UTILITY FUNCTIONS
// =============================================================================

/**
* Format numbers for display (e.g., 1000 -> 1K, 1000000 -> 1M)
*/
function formatNumber(num) {
   if (typeof num !== 'number' || isNaN(num)) return '0';
   if (num >= 1000000) return (num / 1000000).toFixed(1) + 'M';
   if (num >= 1000) return (num / 1000).toFixed(1) + 'K';
   return num.toLocaleString();
}

/**
* Parse metric numbers that might have commas
*/
function parseMetricNumber(value) {
   if (!value) return 0;
   const cleaned = value.toString().replace(/,/g, '');
   return parseInt(cleaned) || 0;
}

/**
* Clean and trim string input
*/
function cleanString(str) {
   if (!str) return '';
   return str.toString().trim();
}

/**
* Format handle to include @ if missing
*/
function formatHandle(handle) {
   if (!handle) return '';
   return handle.startsWith('@') ? handle : `@${handle}`;
}

/**
* Format platform names to proper case
*/
function formatPlatformName(platform) {
   const platformMap = {
       'tiktok': 'TikTok',
       'instagram': 'Instagram', 
       'youtube': 'YouTube',
       'linkedin': 'LinkedIn',
       'facebook': 'Facebook',
       'twitter': 'Twitter'
   };
   
   const normalized = platform ? platform.toLowerCase() : '';
   return platformMap[normalized] || platform;
}

/**
* Get initials from a name
*/
function getInitials(name) {
   if (!name) return '?';
   return name.split(' ')
              .map(word => word.charAt(0))
              .join('')
              .toUpperCase()
              .substring(0, 2);
}

/**
* Escape HTML to prevent XSS
*/
function escapeHtml(text) {
   const div = document.createElement('div');
   div.textContent = text;
   return div.innerHTML;
}

// =============================================================================
// UI STATE MANAGEMENT
// =============================================================================

/**
* Show loading state
*/
function showLoading() {
   document.getElementById('loading').style.display = 'flex';
   document.getElementById('dashboard').style.display = 'none';
   document.getElementById('error').style.display = 'none';
}

/**
* Show dashboard state
*/
function showDashboard() {
   document.getElementById('loading').style.display = 'none';
   document.getElementById('dashboard').style.display = 'block';
   document.getElementById('error').style.display = 'none';
}

/**
* Show error state
*/
function showError(message) {
   document.getElementById('loading').style.display = 'none';
   document.getElementById('dashboard').style.display = 'none';
   document.getElementById('error').style.display = 'flex';
   document.getElementById('errorMessage').textContent = message;
}

/**
* Validate configuration
*/
function validateConfig() {
   return typeof CONFIG !== 'undefined' && CONFIG.API_BASE_URL && CONFIG.API_SECRET_KEY;
}

/**
 * Wait for DOM to be ready then initialize
 */
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeDashboard);
} else {
    // DOM is already ready
    initializeDashboard();
}