// Main application logic for Social Media Marketing Dashboard

// Global variables
let currentCampaignId = '';
let recapData = [];
let filteredData = [];
let currentPlatform = 'all';
let currentViewType = 'daily'; // 'daily', 'weekly', 'monthly'
let startDate = null;
let endDate = null;
let charts = {};

// Initialize the dashboard
window.addEventListener('DOMContentLoaded', initDashboard);

/**
 * Initialize dashboard on page load
 */
async function initDashboard() {
    console.log('Initializing dashboard...');
    
    // Get campaign ID from URL
    const urlParams = new URLSearchParams(window.location.search);
    currentCampaignId = urlParams.get('campaignId');
    
    console.log('Campaign ID from URL:', currentCampaignId);
    
    if (!currentCampaignId) {
        showError('No campaign ID provided. Please add ?campaignId=YOUR_SHEET_ID to the URL');
        return;
    }
    
    // Validate sheet ID format
    if (currentCampaignId.length < 10) {
        showError('Invalid campaign ID format. Please check your Google Sheet ID.');
        return;
    }
    
    try {
        // Load data
        await loadRecapData();
        
        // Check if we got any data
        if (recapData.length === 0) {
            showError('No data found in the Recap sheet. Please check that your sheet has data and the columns are named correctly (Date, Network, Profile, Audience, Impressions, Video Views, Engagements).');
            return;
        }
        
        // Set default date range (last 30 days or full range)
        setDefaultDateRange();
        
        // Initialize platform filters
        renderPlatformFilters();
        
        // Update overview
        updateOverviewStats();
        
        // Initial render of results
        filterAndRenderResults();
        
        // Hide loading, show dashboard
        hideLoading();
    } catch (error) {
        console.error('Error initializing dashboard:', error);
        showError(`Failed to load data: ${error.message}`);
    }
}

/**
 * Load data from Google Sheets Recap tab
 */
async function loadRecapData() {
    try {
        console.log('Loading data from Recap sheet...');
        
        // Try to fetch just the first row to verify the tab exists
        let data;
        try {
            data = await fetchSheetData(currentCampaignId, 'Recap', 'A1:Z1000');
        } catch (error) {
            // If Recap doesn't exist, provide helpful error
            if (error.message.includes('not found') || error.message.includes('Unable to parse')) {
                throw new Error('The "Recap" tab was not found in your Google Sheet. Please make sure you have a tab named exactly "Recap" (case-sensitive).');
            }
            // If it's a permissions error
            if (error.message.includes('permission') || error.message.includes('403')) {
                throw new Error('Permission denied. Please make sure your Google Sheet is shared with "Anyone with the link" as Viewer.');
            }
            throw error;
        }
        
        if (!data || data.length < 2) {
            throw new Error('The Recap sheet exists but has no data. Please add your data with headers: Date, Network, Profile, Audience, Impressions, Video Views, Engagements');
        }
        
        recapData = parseRecapData(data);
        console.log('Loaded recap data:', recapData);
        
    } catch (error) {
        console.error('Error loading recap data:', error);
        throw error;
    }
}

/**
 * Parse Recap sheet data
 */
function parseRecapData(data) {
    const headers = data[0];
    const parsed = [];
    
    console.log('Sheet headers:', headers);
    
    // Find column indices - matching exact column names from the Recap sheet
    const dateIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'date');
    const networkIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'network');
    const profileIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'profile');
    const audienceIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'audience');
    const impressionsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'impressions');
    const videoViewsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'video views');
    const engagementsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'engagements');
    const netAudienceGrowthIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'net audience growth');
    
    console.log('Column indices:', { 
        dateIdx, 
        networkIdx, 
        profileIdx, 
        audienceIdx, 
        impressionsIdx, 
        videoViewsIdx, 
        engagementsIdx,
        netAudienceGrowthIdx
    });
    
    // Check if required columns are found
    if (dateIdx === -1) {
        console.error('Date column not found! Available headers:', headers);
        throw new Error('Required "Date" column not found in Recap sheet');
    }
    if (networkIdx === -1) {
        console.error('Network column not found! Available headers:', headers);
        throw new Error('Required "Network" column not found in Recap sheet');
    }
    
    // Parse each row
    for (let i = 1; i < data.length; i++) {
        const row = data[i];
        
        if (!row[dateIdx]) continue; // Skip rows without date
        
        const record = {
            date: parseDate(row[dateIdx]),
            network: (row[networkIdx] || '').trim(),
            profile: (row[profileIdx] || '').trim(),
            audience: parseNumber(row[audienceIdx]),
            impressions: parseNumber(row[impressionsIdx]),
            videoViews: parseNumber(row[videoViewsIdx]),
            engagements: parseNumber(row[engagementsIdx]),
            audienceGrowth: parseNumber(row[netAudienceGrowthIdx])
        };
        
        if (record.date && record.network) {
            parsed.push(record);
        }
    }
    
    console.log(`Parsed ${parsed.length} records from ${data.length - 1} rows`);
    
    return parsed;
}

/**
 * Parse date string to Date object
 */
function parseDate(dateStr) {
    if (!dateStr) return null;
    
    // Handle MM-DD-YYYY format (e.g., "07-01-2025")
    if (typeof dateStr === 'string' && dateStr.includes('-')) {
        const parts = dateStr.split('-');
        if (parts.length === 3) {
            const month = parseInt(parts[0]) - 1; // Month is 0-indexed
            const day = parseInt(parts[1]);
            const year = parseInt(parts[2]);
            const date = new Date(year, month, day);
            if (!isNaN(date.getTime())) {
                return date;
            }
        }
    }
    
    // Try standard date parsing
    const date = new Date(dateStr);
    if (!isNaN(date.getTime())) {
        return date;
    }
    
    return null;
}

/**
 * Parse number from string (handles commas, etc.)
 */
function parseNumber(val) {
    if (val === null || val === undefined || val === '') return 0;
    const num = parseFloat(val.toString().replace(/,/g, ''));
    return isNaN(num) ? 0 : num;
}

/**
 * Fetch data from Google Sheets via API
 */
async function fetchSheetData(sheetId, tabName, range = '') {
    try {
        // Use the same format as the original working app
        const params = new URLSearchParams({
            sheetId: sheetId,
            sheetName: tabName,
            range: range || ''
        });
        const url = `${CONFIG.API_BASE_URL}/sheets?${params.toString()}`;
        
        console.log('Fetching data from:', url);
        console.log('Using API key:', CONFIG.API_SECRET_KEY.substring(0, 8) + '...');
        
        const response = await fetch(url, {
            method: 'GET',
            headers: {
                'x-api-key': CONFIG.API_SECRET_KEY,
                'Content-Type': 'application/json'
            }
        });
        
        console.log('Response status:', response.status);
        
        if (!response.ok) {
            let errorMessage;
            try {
                const error = await response.json();
                errorMessage = error.error || error.details || 'Unknown error';
            } catch {
                errorMessage = `HTTP ${response.status}: ${response.statusText}`;
            }
            console.error('API error:', errorMessage);
            throw new Error(errorMessage);
        }
        
        const result = await response.json();
        const data = result.data || [];
        console.log('Received data rows:', data.length);
        
        return data;
    } catch (error) {
        console.error('Error fetching sheet data:', error);
        if (error.message.includes('Failed to fetch')) {
            throw new Error('Network error: Could not connect to the API. Please check your internet connection and API URL.');
        }
        throw error;
    }
}

/**
 * Get unique networks/platforms from data
 */
function getAvailablePlatforms() {
    const platforms = new Set();
    recapData.forEach(record => {
        if (record.network) {
            platforms.add(record.network);
        }
    });
    return Array.from(platforms).sort();
}

/**
 * Render platform filter buttons
 */
function renderPlatformFilters() {
    const container = document.getElementById('platformFilters');
    if (!container) return;
    
    container.innerHTML = '';
    
    const platforms = ['All', ...getAvailablePlatforms()];
    
    platforms.forEach(platform => {
        const button = document.createElement('button');
        button.className = 'platform-tab' + (currentPlatform === platform.toLowerCase() ? ' active' : '');
        button.textContent = platform;
        button.onclick = () => selectPlatform(platform);
        container.appendChild(button);
    });
}

/**
 * Select platform filter
 */
function selectPlatform(platform) {
    currentPlatform = platform.toLowerCase();
    renderPlatformFilters();
    filterAndRenderResults();
}

/**
 * Set default date range
 */
function setDefaultDateRange() {
    if (recapData.length === 0) return;
    
    // Get min and max dates from data
    const dates = recapData.map(r => r.date).filter(d => d);
    const minDate = new Date(Math.min(...dates));
    const maxDate = new Date(Math.max(...dates));
    
    // Set date inputs
    document.getElementById('startDate').valueAsDate = minDate;
    document.getElementById('endDate').valueAsDate = maxDate;
    
    startDate = minDate;
    endDate = maxDate;
}

/**
 * Apply date range filter
 */
function applyDateRange() {
    const startInput = document.getElementById('startDate').value;
    const endInput = document.getElementById('endDate').value;
    
    if (startInput) startDate = new Date(startInput);
    if (endInput) endDate = new Date(endInput);
    
    filterAndRenderResults();
}

/**
 * Change view type (daily/weekly/monthly)
 */
function changeViewType(viewType) {
    currentViewType = viewType;
    
    // Update active button
    document.querySelectorAll('.view-toggle-btn').forEach(btn => {
        btn.classList.remove('active');
        if (btn.dataset.view === viewType) {
            btn.classList.add('active');
        }
    });
    
    filterAndRenderResults();
}

/**
 * Filter data based on current selections
 */
function filterAndRenderResults() {
    // Filter by platform
    filteredData = recapData.filter(record => {
        if (currentPlatform !== 'all' && record.network.toLowerCase() !== currentPlatform) {
            return false;
        }
        return true;
    });
    
    // Filter by date range
    if (startDate && endDate) {
        filteredData = filteredData.filter(record => {
            return record.date >= startDate && record.date <= endDate;
        });
    }
    
    console.log('Filtered data:', filteredData);
    
    // Update summary stats
    updateResultsStats();
    
    // Update charts
    renderCharts();
}

/**
 * Update results summary statistics
 */
function updateResultsStats() {
    const stats = calculateStats(filteredData);
    
    document.getElementById('resultEngagements').textContent = formatNumber(stats.totalEngagements);
    document.getElementById('resultImpressions').textContent = formatNumber(stats.totalImpressions);
    document.getElementById('resultAudience').textContent = formatNumber(stats.maxAudience);
    document.getElementById('resultVideoViews').textContent = formatNumber(stats.totalVideoViews);
    document.getElementById('resultAudienceGrowth').textContent = formatNumber(stats.audienceGrowth);
    
    // Update the audience label based on current platform
    const audienceLabel = document.getElementById('audienceLabel');
    if (audienceLabel) {
        if (currentPlatform === 'all') {
            audienceLabel.textContent = 'Total Audience';
        } else {
            audienceLabel.textContent = 'Audience';
        }
    }
}

/**
 * Calculate statistics from filtered data
 */
function calculateStats(data) {
    const stats = {
        totalEngagements: 0,
        totalImpressions: 0,
        maxAudience: 0,
        totalVideoViews: 0,
        audienceGrowth: 0
    };
    
    if (data.length === 0) return stats;
    
    // If we're viewing "All" platforms, sum the max audience from each network
    if (currentPlatform === 'all') {
        const networkMaxAudience = {};
        
        // Find max audience for each network
        data.forEach(record => {
            const network = record.network.toLowerCase();
            if (!networkMaxAudience[network]) {
                networkMaxAudience[network] = 0;
            }
            networkMaxAudience[network] = Math.max(networkMaxAudience[network], record.audience);
        });
        
        // Sum up all network max audiences
        stats.maxAudience = Object.values(networkMaxAudience).reduce((sum, val) => sum + val, 0);
    } else {
        // For specific platform, just get the max audience
        data.forEach(record => {
            stats.maxAudience = Math.max(stats.maxAudience, record.audience);
        });
    }
    
    // Calculate other totals
    data.forEach(record => {
        stats.totalEngagements += record.engagements;
        stats.totalImpressions += record.impressions;
        stats.totalVideoViews += record.videoViews;
        // Sum up the daily audience growth values
        stats.audienceGrowth += record.audienceGrowth;
    });
    
    return stats;
}

/**
 * Aggregate data based on view type
 */
function aggregateData(data, viewType) {
    if (viewType === 'daily') {
        return aggregateByDay(data);
    } else if (viewType === 'weekly') {
        return aggregateByWeek(data);
    } else if (viewType === 'monthly') {
        return aggregateByMonth(data);
    }
    return aggregateByDay(data);
}

/**
 * Aggregate data by day
 */
function aggregateByDay(data) {
    const grouped = {};
    
    data.forEach(record => {
        const dateKey = record.date.toISOString().split('T')[0];
        
        if (!grouped[dateKey]) {
            grouped[dateKey] = {
                date: record.date,
                label: formatDate(record.date),
                audience: 0,
                impressions: 0,
                videoViews: 0,
                engagements: 0,
                count: 0
            };
        }
        
        grouped[dateKey].audience = Math.max(grouped[dateKey].audience, record.audience);
        grouped[dateKey].impressions += record.impressions;
        grouped[dateKey].videoViews += record.videoViews;
        grouped[dateKey].engagements += record.engagements;
        grouped[dateKey].count++;
    });
    
    return Object.values(grouped).sort((a, b) => a.date - b.date);
}

/**
 * Aggregate data by week
 */
function aggregateByWeek(data) {
    const grouped = {};
    
    data.forEach(record => {
        const weekKey = getWeekKey(record.date);
        
        if (!grouped[weekKey]) {
            grouped[weekKey] = {
                date: getWeekStart(record.date),
                label: `Week of ${formatDate(getWeekStart(record.date))}`,
                audience: 0,
                impressions: 0,
                videoViews: 0,
                engagements: 0,
                count: 0
            };
        }
        
        grouped[weekKey].audience = Math.max(grouped[weekKey].audience, record.audience);
        grouped[weekKey].impressions += record.impressions;
        grouped[weekKey].videoViews += record.videoViews;
        grouped[weekKey].engagements += record.engagements;
        grouped[weekKey].count++;
    });
    
    return Object.values(grouped).sort((a, b) => a.date - b.date);
}

/**
 * Aggregate data by month
 */
function aggregateByMonth(data) {
    const grouped = {};
    
    data.forEach(record => {
        const monthKey = `${record.date.getFullYear()}-${String(record.date.getMonth() + 1).padStart(2, '0')}`;
        
        if (!grouped[monthKey]) {
            grouped[monthKey] = {
                date: new Date(record.date.getFullYear(), record.date.getMonth(), 1),
                label: formatMonthYear(record.date),
                audience: 0,
                impressions: 0,
                videoViews: 0,
                engagements: 0,
                count: 0
            };
        }
        
        grouped[monthKey].audience = Math.max(grouped[monthKey].audience, record.audience);
        grouped[monthKey].impressions += record.impressions;
        grouped[monthKey].videoViews += record.videoViews;
        grouped[monthKey].engagements += record.engagements;
        grouped[monthKey].count++;
    });
    
    return Object.values(grouped).sort((a, b) => a.date - b.date);
}

/**
 * Get week key for grouping
 */
function getWeekKey(date) {
    const weekStart = getWeekStart(date);
    return `${weekStart.getFullYear()}-W${getWeekNumber(weekStart)}`;
}

/**
 * Get start of week (Monday)
 */
function getWeekStart(date) {
    const d = new Date(date);
    const day = d.getDay();
    const diff = d.getDate() - day + (day === 0 ? -6 : 1);
    return new Date(d.setDate(diff));
}

/**
 * Get week number
 */
function getWeekNumber(date) {
    const d = new Date(Date.UTC(date.getFullYear(), date.getMonth(), date.getDate()));
    const dayNum = d.getUTCDay() || 7;
    d.setUTCDate(d.getUTCDate() + 4 - dayNum);
    const yearStart = new Date(Date.UTC(d.getUTCFullYear(), 0, 1));
    return Math.ceil((((d - yearStart) / 86400000) + 1) / 7);
}

/**
 * Render all charts
 */
function renderCharts() {
    const aggregated = aggregateData(filteredData, currentViewType);
    
    renderChart('audienceChart', aggregated, 'audience', '👥 Audience', '#4ecdc4');
    renderChart('impressionsChart', aggregated, 'impressions', '👁 Impressions', '#45b7d1');
    renderChart('videoViewsChart', aggregated, 'videoViews', '📹 Video Views', '#96ceb4');
    renderChart('engagementsChart', aggregated, 'engagements', '🤝 Engagements', '#ff6b8a');
}

/**
 * Render individual chart
 */
function renderChart(canvasId, data, metric, label, color) {
    const canvas = document.getElementById(canvasId);
    if (!canvas) return;
    
    const ctx = canvas.getContext('2d');
    
    // Destroy existing chart
    if (charts[canvasId]) {
        charts[canvasId].destroy();
    }
    
    const labels = data.map(d => d.label);
    const values = data.map(d => d[metric]);
    
    // Calculate min and max values
    const minValue = Math.min(...values);
    const maxValue = Math.max(...values);
    const range = maxValue - minValue;
    
    // Determine if we should use dynamic Y-axis
    // If the range is less than 10% of the maximum value, use dynamic scaling
    const useDynamicAxis = range > 0 && (range / maxValue) < 0.15;
    
    // Calculate Y-axis bounds for better visualization
    let yAxisMin = 0;
    let yAxisMax = maxValue;
    
    if (useDynamicAxis) {
        // Add 10% padding above and below the data range
        const padding = range * 0.1;
        yAxisMin = Math.max(0, minValue - padding);
        yAxisMax = maxValue + padding;
    }
    
    charts[canvasId] = new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [{
                label: label,
                data: values,
                borderColor: color,
                backgroundColor: hexToRgba(color, 0.1),
                borderWidth: 3,
                fill: true,
                tension: 0.4,
                pointRadius: 4,
                pointHoverRadius: 6,
                pointBackgroundColor: color,
                pointBorderColor: '#fff',
                pointBorderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    padding: 12,
                    titleFont: {
                        size: 14,
                        weight: 'bold'
                    },
                    bodyFont: {
                        size: 13
                    },
                    callbacks: {
                        label: function(context) {
                            return `${label}: ${formatNumber(context.parsed.y)}`;
                        }
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: !useDynamicAxis,
                    min: yAxisMin,
                    max: yAxisMax,
                    ticks: {
                        callback: function(value) {
                            return formatNumber(value);
                        }
                    },
                    grid: {
                        color: 'rgba(0, 0, 0, 0.05)'
                    }
                },
                x: {
                    grid: {
                        display: false
                    },
                    ticks: {
                        maxRotation: 45,
                        minRotation: 45
                    }
                }
            }
        }
    });
}

/**
 * Update overview statistics
 */
function updateOverviewStats() {
    // Calculate overview stats from all data
    let totalViews = 0;
    let totalLikes = 0;
    let totalComments = 0;
    let totalShares = 0;
    let totalSaves = 0;
    let contentCount = 0;
    
    // For now, set placeholder values
    // These will be populated from other sheets (Content tab) in future
    document.getElementById('overviewViews').textContent = '0';
    document.getElementById('overviewLikes').textContent = '0';
    document.getElementById('overviewComments').textContent = '0';
    document.getElementById('overviewShares').textContent = '0';
    document.getElementById('overviewSaves').textContent = '0';
    document.getElementById('overviewContent').textContent = '0';
    document.getElementById('cpeValue').textContent = '$0.00';
    document.getElementById('cpvValue').textContent = '$0.00';
    document.getElementById('linkClicksValue').textContent = '0';
    document.getElementById('totalEngagementValue').textContent = '0';
}

/**
 * Format number with K, M suffixes
 */
function formatNumber(num) {
    if (num >= 1000000) {
        return (num / 1000000).toFixed(1) + 'M';
    }
    if (num >= 1000) {
        return (num / 1000).toFixed(1) + 'K';
    }
    return num.toLocaleString();
}

/**
 * Format date
 */
function formatDate(date) {
    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
}

/**
 * Format month and year
 */
function formatMonthYear(date) {
    return date.toLocaleDateString('en-US', { month: 'short', year: 'numeric' });
}

/**
 * Convert hex color to rgba
 */
function hexToRgba(hex, alpha) {
    const r = parseInt(hex.slice(1, 3), 16);
    const g = parseInt(hex.slice(3, 5), 16);
    const b = parseInt(hex.slice(5, 7), 16);
    return `rgba(${r}, ${g}, ${b}, ${alpha})`;
}

/**
 * Show/hide tabs
 */
function showTab(tabName) {
    // Update nav items
    document.querySelectorAll('.nav-item').forEach(item => {
        item.classList.remove('active');
    });
    event.target.classList.add('active');
    
    // Update tab content
    document.querySelectorAll('.tab-content').forEach(content => {
        content.classList.remove('active');
    });
    document.getElementById(tabName).classList.add('active');
}

/**
 * Show error message
 */
function showError(message) {
    document.getElementById('loading').style.display = 'none';
    document.getElementById('error').style.display = 'flex';
    document.getElementById('errorMessage').textContent = message;
}

/**
 * Hide loading screen
 */
function hideLoading() {
    document.getElementById('loading').style.display = 'none';
    document.getElementById('dashboard').style.display = 'block';
}

/**
 * Show loading screen
 */
function showLoading() {
    document.getElementById('loading').style.display = 'flex';
    document.getElementById('dashboard').style.display = 'none';
}

/**
 * Refresh data
 */
async function refreshData() {
    showLoading();
    try {
        await loadRecapData();
        filterAndRenderResults();
        hideLoading();
    } catch (error) {
        showError('Failed to refresh data: ' + error.message);
    }
}