// ============================================================================
// SOCIAL MEDIA MARKETING DASHBOARD - COMPLETE APP.JS
// ============================================================================

// ============================================================================
// GLOBAL VARIABLES & CONFIGURATION
// ============================================================================
let currentCampaignId = '';
let recapData = [];
let postsData = [];
let commentAnalysisData = []; // Comment analysis data
let paidMediaData = []; // Paid media data
let paidMediaHeaders = []; // Dynamic headers from PaidMedia sheet
let infoData = {}; // Info cards data from Info sheet
let currentPaidMediaPlatform = 'all'; // Filter for paid media
let contentAllViews = 0; // Total from Content tab column AN
let filteredData = [];
let filteredPostsData = [];
let currentPlatform = 'all';
let currentViewType = 'daily'; // 'daily', 'weekly', 'monthly'
let currentContentView = 'grid'; // 'grid', 'table'
let currentContentSortBy = 'default';
let startDate = null;
let endDate = null;
let charts = {};

// Pagination variables
let currentPage = 1;
const postsPerPage = 12;


// ============================================================================
// INITIALIZATION
// ============================================================================

/**
 * Initialize the dashboard on page load
 */
window.addEventListener('DOMContentLoaded', initDashboard);

async function initDashboard() {
    console.log('Initializing dashboard...');
    
    // Get campaign ID from URL
    const urlParams = new URLSearchParams(window.location.search);
    currentCampaignId = urlParams.get('campaignId');
    
    console.log('Campaign ID from URL:', currentCampaignId);
    
    if (!currentCampaignId) {
        showError('No campaign ID provided. Please add ?campaignId=YOUR_SHEET_ID to the URL');
        return;
    }
    
    // Validate sheet ID format
    if (currentCampaignId.length < 10) {
        showError('Invalid campaign ID format. Please check your Google Sheet ID.');
        return;
    }
    
    try {
        // Load data from both sheets
        await loadRecapData();
        await loadPostsData();
        await loadCommentAnalysisData();
        await loadPaidMediaData();
        await loadInfoData();
        contentAllViews = await loadContentViewsData();
        
        // Check if we got any data
        if (recapData.length === 0) {
            showError('No data found in the Recap sheet. Please check that your sheet has data and the columns are named correctly.');
            return;
        }
        
        // Initialize all sections
        setDefaultDateRange();
        renderPlatformFilters();
        updateOverviewStats();
        filterAndRenderResults();
        
        // Initialize content section
        populateContentFilters();
        filterContent();
        
        // Initialize comment analysis
        renderCommentAnalysis();
        
        // Initialize paid media section
        renderPaidMedia();
        
        // Hide loading, show dashboard
        hideLoading();
        
    } catch (error) {
        console.error('Error initializing dashboard:', error);
        showError(`Failed to load data: ${error.message}`);
    }
}

// ============================================================================
// SHARED UTILITY FUNCTIONS
// ============================================================================

/**
 * Parse date string to Date object
 */
function parseDate(dateStr) {
    if (!dateStr) return null;
    
    // Handle MM-DD-YYYY format (e.g., "07-01-2025")
    if (typeof dateStr === 'string' && dateStr.includes('-')) {
        const parts = dateStr.split('-');
        if (parts.length === 3) {
            const month = parseInt(parts[0]) - 1;
            const day = parseInt(parts[1]);
            const year = parseInt(parts[2]);
            const date = new Date(year, month, day);
            if (!isNaN(date.getTime())) {
                return date;
            }
        }
    }
    
    // Try standard date parsing
    const date = new Date(dateStr);
    if (!isNaN(date.getTime())) {
        return date;
    }
    
    return null;
}

/**
 * Parse post date (handles "7/21/2025 1:44 PM" format)
 */
function parsePostDate(dateStr) {
    if (!dateStr) return null;
    
    // Remove time portion if present
    const datePart = dateStr.split(' ')[0];
    
    // Handle M/D/YYYY format
    if (datePart.includes('/')) {
        const parts = datePart.split('/');
        if (parts.length === 3) {
            const month = parseInt(parts[0]) - 1;
            const day = parseInt(parts[1]);
            const year = parseInt(parts[2]);
            return new Date(year, month, day);
        }
    }
    
    return parseDate(dateStr);
}

/**
 * Parse number from string (handles commas, dollar signs, etc.)
 */
function parseNumber(val) {
    if (val === null || val === undefined || val === '') return 0;
    // Remove commas, dollar signs, and other common currency symbols
    const cleaned = val.toString().replace(/[$,€£¥]/g, '').trim();
    const num = parseFloat(cleaned);
    return isNaN(num) ? 0 : num;
}

/**
 * Format number with K, M suffixes
 */
function formatNumber(num) {
    const isNegative = num < 0;
    const absNum = Math.abs(num);
    
    let formatted;
    if (absNum >= 1000000) {
        formatted = (absNum / 1000000).toFixed(1) + 'M';
    } else if (absNum >= 1000) {
        formatted = (absNum / 1000).toFixed(1) + 'K';
    } else {
        formatted = absNum.toLocaleString();
    }
    
    return isNegative ? '-' + formatted : formatted;
}

/**
 * Format date
 */
function formatDate(date) {
    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
}

/**
 * Format month and year
 */
function formatMonthYear(date) {
    return date.toLocaleDateString('en-US', { month: 'short', year: 'numeric' });
}

/**
 * Convert hex color to rgba
 */
function hexToRgba(hex, alpha) {
    const r = parseInt(hex.slice(1, 3), 16);
    const g = parseInt(hex.slice(3, 5), 16);
    const b = parseInt(hex.slice(5, 7), 16);
    return `rgba(${r}, ${g}, ${b}, ${alpha})`;
}

/**
 * Show/hide tabs
 */
function showTab(tabName) {
    // Update nav items
    document.querySelectorAll('.nav-item').forEach(item => {
        item.classList.remove('active');
    });
    event.target.classList.add('active');
    
    // Update tab content
    document.querySelectorAll('.tab-content').forEach(content => {
        content.classList.remove('active');
    });
    document.getElementById(tabName).classList.add('active');
    
    // If switching to comment analysis tab, ensure it's rendered
    if (tabName === 'commentAnalysis' && commentAnalysisData.length > 0) {
        // Re-render if needed
        const content = document.getElementById('commentAnalysisContent');
        if (!content.children.length) {
            renderCommentAnalysis();
        }
    }
}

/**
 * Show error message
 */
function showError(message) {
    document.getElementById('loading').style.display = 'none';
    document.getElementById('error').style.display = 'flex';
    document.getElementById('errorMessage').textContent = message;
}

/**
 * Hide loading screen
 */
function hideLoading() {
    document.getElementById('loading').style.display = 'none';
    document.getElementById('dashboard').style.display = 'block';
}

/**
 * Show loading screen
 */
function showLoading() {
    document.getElementById('loading').style.display = 'flex';
    document.getElementById('dashboard').style.display = 'none';
}

// ============================================================================
// API & DATA LOADING
// ============================================================================

/**
 * Fetch data from Google Sheets via API
 */
async function fetchSheetData(sheetId, tabName, range = '') {
    try {
        const params = new URLSearchParams({
            sheetId: sheetId,
            sheetName: tabName,
            range: range || ''
        });
        const url = `${CONFIG.API_BASE_URL}/sheets?${params.toString()}`;
        
        console.log('Fetching data from:', url);
        console.log('Using API key:', CONFIG.API_SECRET_KEY.substring(0, 8) + '...');
        
        const response = await fetch(url, {
            method: 'GET',
            headers: {
                'x-api-key': CONFIG.API_SECRET_KEY,
                'Content-Type': 'application/json'
            }
        });
        
        console.log('Response status:', response.status);
        
        if (!response.ok) {
            let errorMessage;
            try {
                const error = await response.json();
                errorMessage = error.error || error.details || 'Unknown error';
            } catch {
                errorMessage = `HTTP ${response.status}: ${response.statusText}`;
            }
            console.error('API error:', errorMessage);
            throw new Error(errorMessage);
        }
        
        const result = await response.json();
        const data = result.data || [];
        console.log('Received data rows:', data.length);
        
        return data;
    } catch (error) {
        console.error('Error fetching sheet data:', error);
        if (error.message.includes('Failed to fetch')) {
            throw new Error('Network error: Could not connect to the API. Please check your internet connection and API URL.');
        }
        throw error;
    }
}

/**
 * Load data from Google Sheets Recap tab
 */
async function loadRecapData() {
    try {
        console.log('Loading data from Recap sheet...');
        
        let data;
        try {
            data = await fetchSheetData(currentCampaignId, 'Recap', 'A1:AZ1000');
        } catch (error) {
            if (error.message.includes('not found') || error.message.includes('Unable to parse')) {
                throw new Error('The "Recap" tab was not found in your Google Sheet.');
            }
            if (error.message.includes('permission') || error.message.includes('403')) {
                throw new Error('Permission denied. Please share your sheet with the service account.');
            }
            throw error;
        }
        
        if (!data || data.length < 2) {
            throw new Error('The Recap sheet exists but has no data.');
        }
        
        recapData = parseRecapData(data);
        console.log('Loaded recap data:', recapData);
        
    } catch (error) {
        console.error('Error loading recap data:', error);
        throw error;
    }
}

/**
 * Parse Recap sheet data
 */
function parseRecapData(data) {
    const headers = data[0];
    const parsed = [];
    
    console.log('Sheet headers:', headers);
    
    // Find column indices
    const dateIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'date');
    const networkIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'network');
    const profileIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'profile');
    const audienceIdx = 4; // Column E (hardcoded to index 4)
    const impressionsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'impressions');
    const reachIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'reach'); // Column Q
    const videoViewsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'video views');
    const engagementsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'engagements');
    const netAudienceGrowthIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'net audience growth');
    const postClicksAllIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'post clicks (all)'); // Column AI
    
    console.log('Column indices:', { 
        dateIdx, networkIdx, profileIdx, audienceIdx, 
        impressionsIdx, reachIdx, videoViewsIdx,
        engagementsIdx, netAudienceGrowthIdx, postClicksAllIdx
    });
    
    if (dateIdx === -1) {
        throw new Error('Required "Date" column not found in Recap sheet');
    }
    if (networkIdx === -1) {
        throw new Error('Required "Network" column not found in Recap sheet');
    }
    
    // Parse each row
    for (let i = 1; i < data.length; i++) {
        const row = data[i];
        
        if (!row[dateIdx]) continue;
        
        const record = {
            date: parseDate(row[dateIdx]),
            network: (row[networkIdx] || '').trim(),
            profile: (row[profileIdx] || '').trim(),
            audience: parseNumber(row[audienceIdx]),
            impressions: parseNumber(row[impressionsIdx]),
            reach: parseNumber(row[reachIdx]),
            videoViews: parseNumber(row[videoViewsIdx]),
            engagements: parseNumber(row[engagementsIdx]),
            audienceGrowth: parseNumber(row[netAudienceGrowthIdx]),
            postClicksAll: parseNumber(row[postClicksAllIdx])
        };
        
        if (record.date && record.network) {
            parsed.push(record);
        }
    }
    
    console.log(`Parsed ${parsed.length} records from ${data.length - 1} rows`);
    
    return parsed;
}

/**
 * Load posts data from Google Sheets Posts tab
 */
async function loadPostsData() {
    try {
        console.log('Loading posts data from Posts sheet...');
        const data = await fetchSheetData(currentCampaignId, 'Posts', 'A1:AZ1000');
        
        if (!data || data.length < 2) {
            console.warn('No posts data found');
            postsData = [];
            return;
        }
        
        postsData = parsePostsData(data);
        console.log('Loaded posts data:', postsData.length, 'posts');
        
    } catch (error) {
        console.error('Error loading posts data:', error);
        postsData = [];
    }
}

/**
 * Load All Views data from Content tab (or Posts tab as fallback)
 */
async function loadContentViewsData() {
    try {
        console.log('Attempting to load All Views data from Content sheet...');
        
        // Try to fetch Content sheet data
        let data;
        try {
            data = await fetchSheetData(currentCampaignId, 'Posts', 'A1:AZ1000');
        } catch (fetchError) {
            // Content sheet doesn't exist or error fetching - use fallback
            console.log('Content sheet not accessible, using Posts tab as fallback');
            return await loadAllViewsFromPosts();
        }
        
        if (!data || data.length < 2) {
            console.log('No data in Content sheet, using Posts tab as fallback');
            return await loadAllViewsFromPosts();
        }
        
        const headers = data[0];
        const allViewsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'all views'); // Column AN
        
        console.log('Content sheet - All Views column index:', allViewsIdx);
        
        if (allViewsIdx === -1) {
            console.log('All Views column not found in Content sheet, using Posts tab as fallback');
            return await loadAllViewsFromPosts();
        }
        
        // Sum all views from all rows
        let totalViews = 0;
        for (let i = 1; i < data.length; i++) {
            const row = data[i];
            if (row[allViewsIdx]) {
                totalViews += parseNumber(row[allViewsIdx]);
            }
        }
        
        console.log('✓ Total All Views from Content sheet:', totalViews);
        return totalViews;
        
    } catch (error) {
        console.log('Error with Content sheet, using Posts tab as fallback');
        return await loadAllViewsFromPosts();
    }
}

/**
 * Load All Views from Posts tab as fallback
 */
async function loadAllViewsFromPosts() {
    try {
        console.log('→ Using Posts sheet for All Views (fallback mode)');
        
        // If postsData is already loaded, sum Video Views from there
        if (postsData && postsData.length > 0) {
            const totalViews = postsData.reduce((sum, post) => sum + (post.videoViews || 0), 0);
            console.log('✓ Total All Views from Posts data (Video Views, Column AN):', totalViews);
            return totalViews;
        }
        
        // Otherwise fetch Posts data directly and parse
        console.log('Posts data not loaded yet, fetching directly...');
        const data = await fetchSheetData(currentCampaignId, 'Posts', 'A1:AZ1000');
        
        if (!data || data.length < 2) {
            console.warn('No Posts data found');
            return 0;
        }
        
        const headers = data[0];
        console.log('Posts sheet headers:', headers);
        
        // Look for Video Views column (should be Column AN)
        const videoViewsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'video views');
        
        console.log('Video Views column index:', videoViewsIdx);
        
        if (videoViewsIdx === -1) {
            console.warn('Video Views column not found in Posts sheet');
            return 0;
        }
        
        // Sum video views from all rows
        let totalViews = 0;
        for (let i = 1; i < data.length; i++) {
            const row = data[i];
            if (row[videoViewsIdx]) {
                totalViews += parseNumber(row[videoViewsIdx]);
            }
        }
        
        console.log('✓ Total Views from Posts sheet (Video Views, Column AN):', totalViews);
        return totalViews;
        
    } catch (error) {
        console.error('Error loading views from Posts fallback:', error);
        return 0;
    }
}

/**
 * Convert Google Drive sharing URL to direct image URL
 */
function convertGoogleDriveUrl(url) {
    if (!url || typeof url !== 'string') return '';
    
    // Check if it's a Google Drive URL
    if (url.includes('drive.google.com')) {
        // Extract file ID from various Google Drive URL formats
        let fileId = null;
        
        // Format: https://drive.google.com/file/d/FILE_ID/view
        const match1 = url.match(/\/file\/d\/([a-zA-Z0-9_-]+)/);
        if (match1) {
            fileId = match1[1];
        }
        
        // Format: https://drive.google.com/open?id=FILE_ID
        const match2 = url.match(/[?&]id=([a-zA-Z0-9_-]+)/);
        if (match2) {
            fileId = match2[1];
        }
        
        // If we found a file ID, convert to direct image URL
        if (fileId) {
            return `https://drive.google.com/thumbnail?id=${fileId}&sz=w400`;
        }
    }
    
    // If not a Google Drive URL or couldn't parse, return as-is
    return url;
}

/**
 * Parse Posts sheet data
 */
function parsePostsData(data) {
    const headers = data[0];
    const parsed = [];
    
    console.log('Posts headers:', headers);
    
    // Find column indices
    const dateIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'date');
    const networkIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'network');
    const postTypeIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'post type');
    const contentTypeIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'content type');
    const linkIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'link');
    const thumbnailIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'thumbnail');
    const postIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'post');
    const impressionsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'impressions');
    const reachIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'reach');
    const engagementsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'engagements');
    const reactionsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'reactions');
    const likesIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'likes');
    const commentsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'comments');
    const sharesIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'shares');
    const savesIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'saves');
    const postClicksIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'post clicks (all)');
    const videoViewsIdx = headers.findIndex(h => h && h.toLowerCase().trim() === 'video views');
    
    // Parse each row
    for (let i = 1; i < data.length; i++) {
        const row = data[i];
        
        if (!row[dateIdx]) continue;
        
        const post = {
            date: parsePostDate(row[dateIdx]),
            network: (row[networkIdx] || '').trim(),
            postType: (row[postTypeIdx] || '').trim(),
            contentType: (row[contentTypeIdx] || '').trim(),
            link: (row[linkIdx] || '').trim(),
            thumbnail: convertGoogleDriveUrl((row[thumbnailIdx] || '').trim()),
            title: (row[postIdx] || '').trim(),
            impressions: parseNumber(row[impressionsIdx]),
            reach: parseNumber(row[reachIdx]),
            engagements: parseNumber(row[engagementsIdx]),
            reactions: parseNumber(row[reactionsIdx]),
            likes: parseNumber(row[likesIdx]),
            comments: parseNumber(row[commentsIdx]),
            shares: parseNumber(row[sharesIdx]),
            saves: parseNumber(row[savesIdx]),
            postClicks: parseNumber(row[postClicksIdx]),
            videoViews: parseNumber(row[videoViewsIdx])
        };
        
        // Calculate ER
        post.er = post.impressions > 0 ? (post.engagements / post.impressions * 100) : 0;
        
        if (post.date && post.network) {
            parsed.push(post);
        }
    }
    
    return parsed;
}

// ============================================================================
// OVERVIEW TAB
// ============================================================================

/**
 * Update overview statistics
 */
function updateOverviewStats() {
    // Calculate totals from recapData
    const totals = {
        videoViews: 0,
        reach: 0,
        engagements: 0,
        postClicksAll: 0
    };
    
    recapData.forEach(record => {
        totals.videoViews += record.videoViews || 0;
        totals.reach += record.reach || 0;
        totals.engagements += record.engagements || 0;
        totals.postClicksAll += record.postClicksAll || 0;
    });
    
    // Calculate from postsData if available
    const postTotals = {
        impressions: 0,
        reach: 0,
        likes: 0,
        comments: 0,
        shares: 0,
        saves: 0,
        reactions: 0,
        postClicks: 0,
        contentPieces: postsData.length
    };
    
    postsData.forEach(post => {
        postTotals.impressions += post.impressions || 0;
        postTotals.reach += post.reach || 0;
        postTotals.likes += post.likes || 0;
        postTotals.comments += post.comments || 0;
        postTotals.shares += post.shares || 0;
        postTotals.saves += post.saves || 0;
        postTotals.reactions += post.reactions || 0;
        postTotals.postClicks += post.postClicks || 0;
    });
    
    // Update display - use contentAllViews from Content tab and postTotals from Posts
    document.getElementById('overviewImpressions').textContent = formatNumber(postTotals.impressions);
    document.getElementById('overviewViews').textContent = formatNumber(contentAllViews);
    document.getElementById('overviewReach').textContent = formatNumber(postTotals.reach);
    document.getElementById('overviewReactions').textContent = formatNumber(postTotals.reactions);
    document.getElementById('overviewLikes').textContent = formatNumber(postTotals.likes);
    document.getElementById('overviewComments').textContent = formatNumber(postTotals.comments);
    document.getElementById('overviewShares').textContent = formatNumber(postTotals.shares);
    document.getElementById('overviewSaves').textContent = formatNumber(postTotals.saves);
    document.getElementById('overviewPostClicks').textContent = formatNumber(postTotals.postClicks);
    document.getElementById('overviewContent').textContent = formatNumber(postTotals.contentPieces);
    
    // Render info cards
    renderInfoCards();
    
    // Update platform breakdown
    updatePlatformBreakdown();
}

/**
 * Update platform breakdown sections
 */
function updatePlatformBreakdown() {
    // Calculate followers by platform from recapData
    const followersByPlatform = {};
    recapData.forEach(record => {
        const platform = record.network || 'Unknown';
        if (!followersByPlatform[platform]) {
            followersByPlatform[platform] = 0;
        }
        // Use the max audience for each platform (most recent count)
        if (record.audience > followersByPlatform[platform]) {
            followersByPlatform[platform] = record.audience;
        }
    });
    
    // Calculate content by platform from postsData
    const contentByPlatform = {};
    const contentTypes = {};
    
    postsData.forEach(post => {
        const platform = post.network || 'Unknown';
        const type = post.contentType || 'Unknown';
        
        // Count posts by platform
        contentByPlatform[platform] = (contentByPlatform[platform] || 0) + 1;
        
        // Count by content type
        contentTypes[type] = (contentTypes[type] || 0) + 1;
    });
    
    // Render followers by platform
    renderPlatformList('followersByPlatform', followersByPlatform, true);
    
    // Render content by platform
    renderPlatformList('contentByPlatform', contentByPlatform, false);
    
    // Render content types
    renderPlatformList('contentTypes', contentTypes, false);
}

/**
 * Render platform breakdown list
 */
function renderPlatformList(containerId, data, isFollowers) {
    const container = document.getElementById(containerId);
    if (!container) return;
    
    // Sort by value descending
    const sorted = Object.entries(data).sort((a, b) => b[1] - a[1]);
    
    container.innerHTML = sorted.map(([name, value]) => {
        const platformClass = name.toLowerCase().replace(/\s+/g, '');
        const iconUrl = getPlatformIconUrl(name);
        const useImage = iconUrl !== null;
        const iconContent = useImage 
            ? `<img src="${iconUrl}" alt="${name}" class="platform-icon-img" />` 
            : getPlatformIcon(name);
        
        return `
            <div class="platform-breakdown-item">
                <div class="platform-breakdown-item-left">
                    <div class="platform-breakdown-item-icon ${platformClass} ${useImage ? 'has-image' : ''}">
                        ${iconContent}
                    </div>
                    <div class="platform-breakdown-item-name">${name}</div>
                </div>
                <div class="platform-breakdown-item-value">${formatNumber(value)}</div>
            </div>
        `;
    }).join('');
}

/**
 * Get platform icon URL for actual images
 */
function getPlatformIconUrl(name) {
    const iconMap = {
        'Facebook': 'icons8-facebook-100.png',
        'Meta': 'icons8-facebook-100.png', // Meta Ads uses Facebook icon
        'LinkedIn': 'icons8-linkedin-100.png',
        'Twitter': 'icons8-x-100.png',
        'X': 'icons8-x-100.png',
        'TikTok': 'icons8-tiktok-100.png',
        'Instagram': 'icons8-instagram-100.png',
        'YouTube': 'icons8-youtube-100.png'
    };
    
    // Normalize the name for matching
    const nameLower = name.toLowerCase();
    
    // Check if the name contains any of these platforms (case-insensitive)
    for (const [platform, icon] of Object.entries(iconMap)) {
        if (nameLower.includes(platform.toLowerCase())) {
            return icon;
        }
    }
    
    return null;
}

/**
 * Get platform icon emoji (fallback for content types without images)
 */
function getPlatformIcon(name) {
    const icons = {
        'Facebook': '📘',
        'Meta': '📘', // Meta uses Facebook icon
        'LinkedIn': '💼',
        'Twitter': '🐦',
        'X': '✖️',
        'TikTok': '🎵',
        'Instagram': '📷',
        'YouTube': '▶️',
        // Content types
        'Facebook Content': '📘',
        'LinkedIn Content': '💼',
        'Twitter Content': '🐦',
        'TikTok Content': '🎵',
        'Instagram Reels': '🎬',
        'Instagram Stories': '📖',
        'Instagram Content': '📷',
        'YouTube Integrations': '▶️',
        'YouTube Content': '▶️',
        'Video': '🎥',
        'Photo': '📸',
        'Carousel': '🎠',
        'Story': '📖',
        'Reel': '🎬',
        'Short': '⚡',
        'Post': '📝'
    };
    
    // Try exact match first
    if (icons[name]) {
        return icons[name];
    }
    
    // Try case-insensitive partial match
    const nameLower = name.toLowerCase();
    for (const [key, icon] of Object.entries(icons)) {
        if (nameLower.includes(key.toLowerCase())) {
            return icon;
        }
    }
    
    return '📱'; // Default icon
}

/**
 * Get icon for info card based on title
 */
function getInfoCardIcon(title) {
    const titleLower = title.toLowerCase();
    
    const iconMap = {
        'campaign': '📣',
        'objective': '🎯',
        'goal': '🎯',
        'target': '🎯',
        'budget': '💰',
        'spend': '💸',
        'cost': '💵',
        'date': '📅',
        'duration': '⏱️',
        'period': '📆',
        'time': '⏰',
        'deadline': '⏳',
        'start': '▶️',
        'end': '⏹️',
        'audience': '👥',
        'people': '👨‍👩‍👧‍👦',
        'demographic': '📊',
        'location': '📍',
        'geography': '🌍',
        'region': '🗺️',
        'age': '🎂',
        'gender': '👤',
        'brand': '🏷️',
        'product': '📦',
        'service': '🛎️',
        'client': '🤝',
        'partner': '🤝',
        'agency': '🏢',
        'company': '🏢',
        'industry': '🏭',
        'category': '📋',
        'type': '📝',
        'format': '🎨',
        'creative': '🎨',
        'message': '💬',
        'tagline': '💬',
        'description': '📄',
        'overview': '📋',
        'summary': '📝',
        'note': '📌',
        'info': 'ℹ️',
        'detail': '📋',
        'status': '✅',
        'stage': '🔄',
        'phase': '🔄',
        'kpi': '📈',
        'metric': '📊',
        'performance': '📈'
    };
    
    // Try to match keywords in title
    for (const [keyword, icon] of Object.entries(iconMap)) {
        if (titleLower.includes(keyword)) {
            return icon;
        }
    }
    
    return '📋'; // Default icon
}

/**
 * Render info cards from Info sheet data
 */
function renderInfoCards() {
    const container = document.getElementById('infoCardsContainer');
    const section = document.getElementById('infoCardsSection');
    
    if (!container) {
        console.warn('Info cards container not found');
        return;
    }
    
    // Filter out headerA and headerB - only show cards from columns C, D, E
    const cardData = Object.entries(infoData).filter(([key]) => 
        key !== 'headerA' && key !== 'headerB'
    );
    
    if (cardData.length === 0) {
        container.innerHTML = '';
        if (section) section.style.display = 'none';
        return;
    }
    
    if (section) section.style.display = 'block';
    
    const cardsHtml = cardData.map(([title, content]) => {
        const icon = getInfoCardIcon(title);
        return `
            <div class="campaign-info-card">
                <div class="campaign-info-header">
                    <div class="campaign-info-icon">${icon}</div>
                    <div class="campaign-info-card-title">${title}</div>
                </div>
                <div class="campaign-info-content">
                    <div class="info-text">${content}</div>
                </div>
            </div>
        `;
    }).join('');
    
    container.innerHTML = cardsHtml;
}

// ============================================================================
// CONTENT TAB
// ============================================================================

/**
 * Populate content filter dropdowns
 */
function populateContentFilters() {
    // Platforms
    const platforms = [...new Set(postsData.map(p => p.network))].sort();
    const platformFilter = document.getElementById('contentPlatformFilter');
    if (platformFilter) {
        platformFilter.innerHTML = '<option value="">All Platforms</option>';
        platforms.forEach(platform => {
            const option = document.createElement('option');
            option.value = platform;
            option.textContent = platform;
            platformFilter.appendChild(option);
        });
    }
    
    // Initial population of content types and post types (all)
    updateContentTypeDropdown('');
    updatePostTypeDropdown('');
    
    // Set default date range based on posts data
    setContentDateRange();
}

/**
 * Set content date range to oldest and newest posts
 */
function setContentDateRange() {
    if (postsData.length === 0) return;
    
    const dates = postsData
        .map(p => p.date)
        .filter(d => d && !isNaN(d.getTime()));
    
    if (dates.length === 0) return;
    
    const minDate = new Date(Math.min(...dates));
    const maxDate = new Date(Math.max(...dates));
    
    const startDateInput = document.getElementById('contentStartDate');
    const endDateInput = document.getElementById('contentEndDate');
    
    if (startDateInput) {
        startDateInput.valueAsDate = minDate;
    }
    if (endDateInput) {
        endDateInput.valueAsDate = maxDate;
    }
    
    console.log('Content date range set:', { minDate, maxDate });
}

/**
 * Update Content Type dropdown based on selected platform
 */
function updateContentTypeDropdown(selectedPlatform) {
    const contentTypeFilter = document.getElementById('contentTypeFilter');
    if (!contentTypeFilter) return;
    
    const currentValue = contentTypeFilter.value;
    
    // Filter posts by platform if one is selected
    const filteredPosts = selectedPlatform 
        ? postsData.filter(p => p.network === selectedPlatform)
        : postsData;
    
    // Get unique content types from filtered posts
    const contentTypes = [...new Set(filteredPosts.map(p => p.contentType).filter(c => c))].sort();
    
    contentTypeFilter.innerHTML = '<option value="">All Content Types</option>';
    contentTypes.forEach(type => {
        const option = document.createElement('option');
        option.value = type;
        option.textContent = type;
        contentTypeFilter.appendChild(option);
    });
    
    // Restore previous selection if it still exists
    if (currentValue && contentTypes.includes(currentValue)) {
        contentTypeFilter.value = currentValue;
    }
}

/**
 * Update Post Type dropdown based on selected platform
 */
function updatePostTypeDropdown(selectedPlatform) {
    const postTypeFilter = document.getElementById('postTypeFilter');
    if (!postTypeFilter) return;
    
    const currentValue = postTypeFilter.value;
    
    // Filter posts by platform if one is selected
    const filteredPosts = selectedPlatform 
        ? postsData.filter(p => p.network === selectedPlatform)
        : postsData;
    
    // Get unique post types from filtered posts
    const postTypes = [...new Set(filteredPosts.map(p => p.postType).filter(p => p))].sort();
    
    postTypeFilter.innerHTML = '<option value="">All Post Types</option>';
    postTypes.forEach(type => {
        const option = document.createElement('option');
        option.value = type;
        option.textContent = type;
        postTypeFilter.appendChild(option);
    });
    
    // Restore previous selection if it still exists
    if (currentValue && postTypes.includes(currentValue)) {
        postTypeFilter.value = currentValue;
    }
}

/**
 * Filter content based on selections
 */
function filterContent() {
    const searchTerm = document.getElementById('contentSearchInput')?.value.toLowerCase() || '';
    const platformFilter = document.getElementById('contentPlatformFilter')?.value || '';
    const contentTypeFilter = document.getElementById('contentTypeFilter')?.value || '';
    const postTypeFilter = document.getElementById('postTypeFilter')?.value || '';
    const startDateInput = document.getElementById('contentStartDate')?.value || '';
    const endDateInput = document.getElementById('contentEndDate')?.value || '';
    
    console.log('Filtering content with dates:', { startDateInput, endDateInput });
    
    // Update dependent dropdowns when platform changes
    updateContentTypeDropdown(platformFilter);
    updatePostTypeDropdown(platformFilter);
    
    filteredPostsData = postsData.filter(post => {
        if (platformFilter && post.network !== platformFilter) return false;
        if (contentTypeFilter && post.contentType !== contentTypeFilter) return false;
        if (postTypeFilter && post.postType !== postTypeFilter) return false;
        if (searchTerm && !post.title.toLowerCase().includes(searchTerm)) return false;
        
        // Date filtering
        if (startDateInput || endDateInput) {
            const postDate = post.date; // Already a Date object from parsePostsData
            
            if (!postDate || isNaN(postDate.getTime())) {
                console.warn('Invalid post date:', postDate);
                return true; // Include posts with invalid dates
            }
            
            if (startDateInput) {
                const startDate = new Date(startDateInput);
                startDate.setHours(0, 0, 0, 0);
                const postDateOnly = new Date(postDate);
                postDateOnly.setHours(0, 0, 0, 0);
                if (postDateOnly < startDate) return false;
            }
            if (endDateInput) {
                const endDate = new Date(endDateInput);
                endDate.setHours(23, 59, 59, 999);
                const postDateOnly = new Date(postDate);
                postDateOnly.setHours(23, 59, 59, 999);
                if (postDateOnly > endDate) return false;
            }
        }
        
        return true;
    });
    
    console.log('Filtered posts:', filteredPostsData.length, 'of', postsData.length);
    
    sortContent();
}

/**
 * Sort content based on selection
 */
function sortContent() {
    const sortBy = document.getElementById('contentSortBy')?.value || 'default';
    
    let sorted = [...filteredPostsData];
    
    switch(sortBy) {
        case 'views':
            sorted.sort((a, b) => b.videoViews - a.videoViews);
            break;
        case 'views_asc':
            sorted.sort((a, b) => a.videoViews - b.videoViews);
            break;
        case 'likes':
            sorted.sort((a, b) => b.likes - a.likes);
            break;
        case 'likes_asc':
            sorted.sort((a, b) => a.likes - b.likes);
            break;
        case 'engagements':
            sorted.sort((a, b) => b.engagements - a.engagements);
            break;
        case 'engagements_asc':
            sorted.sort((a, b) => a.engagements - b.engagements);
            break;
        case 'er':
            sorted.sort((a, b) => b.er - a.er);
            break;
        case 'er_asc':
            sorted.sort((a, b) => a.er - b.er);
            break;
        case 'newest':
            sorted.sort((a, b) => b.date - a.date);
            break;
        case 'oldest':
            sorted.sort((a, b) => a.date - b.date);
            break;
        default:
            break;
    }
    
    filteredPostsData = sorted;
    updateContentDisplay();
}

/**
 * Update content display
 */
function updateContentDisplay() {
    updateContentStats();

    const totalPages = Math.ceil(filteredPostsData.length / postsPerPage);
    currentPage = Math.min(currentPage, totalPages || 1);

    const startIndex = (currentPage - 1) * postsPerPage;
    const endIndex = startIndex + postsPerPage;
    const paginatedData = filteredPostsData.slice(startIndex, endIndex);

    if (currentContentView === 'grid') {
        renderContentGrid(paginatedData);
    } else {
        renderContentTable(paginatedData);
    }

    renderPaginationControls(totalPages);
}


/**
 * Update content summary statistics
 */
function updateContentStats() {
    const stats = {
        pieces: filteredPostsData.length,
        views: filteredPostsData.reduce((sum, p) => sum + p.videoViews, 0),
        impressions: filteredPostsData.reduce((sum, p) => sum + p.impressions, 0),
        reach: filteredPostsData.reduce((sum, p) => sum + (p.reach || 0), 0),
        reactions: filteredPostsData.reduce((sum, p) => sum + p.reactions, 0),
        likes: filteredPostsData.reduce((sum, p) => sum + p.likes, 0),
        postClicks: filteredPostsData.reduce((sum, p) => sum + p.postClicks, 0)
    };
    
    document.getElementById('contentPieces').textContent = formatNumber(stats.pieces);
    document.getElementById('contentTotalViews').textContent = formatNumber(stats.views);
    document.getElementById('contentTotalImpressions').textContent = formatNumber(stats.impressions);
    document.getElementById('contentTotalReach').textContent = formatNumber(stats.reach);
    document.getElementById('contentTotalReactions').textContent = formatNumber(stats.reactions);
    document.getElementById('contentTotalLikes').textContent = formatNumber(stats.likes);
    document.getElementById('contentTotalPostClicks').textContent = formatNumber(stats.postClicks);
}

/**
 * Render content grid view
 */
function renderContentGrid(posts) {
    const gridContainer = document.getElementById('contentGridView');
    if (!gridContainer) return;

    gridContainer.innerHTML = '';

    posts.forEach(post => {
        const card = createContentCard(post);
        gridContainer.appendChild(card);
    });
}

/**
 * Create content card for grid view
 */
function createContentCard(post) {
    const card = document.createElement('div');
    card.className = 'content-card';
    
    const thumbnail = post.thumbnail || 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="400" height="300"%3E%3Crect fill="%23ddd" width="400" height="300"/%3E%3Ctext x="50%25" y="50%25" text-anchor="middle" fill="%23999" font-size="18"%3ENo Thumbnail%3C/text%3E%3C/svg%3E';
    
    const title = post.title.substring(0, 80) + (post.title.length > 80 ? '...' : '');
    
    // Build stats array based on rules
    const stats = [];
    
    // Views (always) - use impressions if views not available
    const viewsValue = post.videoViews > 0 ? post.videoViews : post.impressions;
    const viewsLabel = post.videoViews > 0 ? 'Views' : 'Impressions';
    stats.push(`
        <div class="content-stat-item">
            <div class="content-stat-value">${formatNumber(viewsValue)}</div>
            <div class="content-stat-label">${viewsLabel}</div>
        </div>
    `);
    
    // Likes (always)
    stats.push(`
        <div class="content-stat-item">
            <div class="content-stat-value">${formatNumber(post.likes)}</div>
            <div class="content-stat-label">Likes</div>
        </div>
    `);
    
    // Comments (if more than 20)
    if (post.comments > 20) {
        stats.push(`
            <div class="content-stat-item">
                <div class="content-stat-value">${formatNumber(post.comments)}</div>
                <div class="content-stat-label">Comments</div>
            </div>
        `);
    }
    
    // Shares (if more than 10)
    if (post.shares > 10) {
        stats.push(`
            <div class="content-stat-item">
                <div class="content-stat-value">${formatNumber(post.shares)}</div>
                <div class="content-stat-label">Shares</div>
            </div>
        `);
    }
    
    // Saves (if more than 10)
    if (post.saves > 10) {
        stats.push(`
            <div class="content-stat-item">
                <div class="content-stat-value">${formatNumber(post.saves)}</div>
                <div class="content-stat-label">Saves</div>
            </div>
        `);
    }
    
    card.innerHTML = `
        <div class="content-card-thumbnail">
            <img src="${thumbnail}" alt="Post thumbnail" onerror="this.src='data:image/svg+xml,%3Csvg xmlns=%22http://www.w3.org/2000/svg%22 width=%22400%22 height=%22300%22%3E%3Crect fill=%22%23ddd%22 width=%22400%22 height=%22300%22/%3E%3Ctext x=%2250%25%22 y=%2250%25%22 text-anchor=%22middle%22 fill=%22%23999%22 font-size=%2218%22%3ENo Image%3C/text%3E%3C/svg%3E'">
            <div class="content-card-platform">${post.network}</div>
        </div>
        <div class="content-card-body">
            <div class="content-card-title">${title}</div>
            <div class="content-card-meta">
                <span>${post.contentType || 'Content'}</span>
                <span>${post.date ? post.date.toLocaleDateString() : ''}</span>
            </div>
            <div class="content-card-stats">
                ${stats.join('')}
            </div>
            ${post.link ? `<a href="${post.link}" target="_blank" class="content-card-link">View Post →</a>` : ''}
        </div>
    `;
    
    return card;
}

/**
 * Render content table view
 */
function renderContentTable(posts) {
    const tableContainer = document.getElementById('contentTableView');
    if (!tableContainer) return;

    let tableHTML = `
        <table class="content-data-table">
            <thead>
                <tr>
                    <th>Thumbnail</th>
                    <th>Post</th>
                    <th>Network</th>
                    <th>Type</th>
                    <th>Date</th>
                    <th>Impressions</th>
                    <th>Reach</th>
                    <th>Views</th>
                    <th>Engagements</th>
                    <th>Reactions</th>
                    <th>Likes</th>
                    <th>Comments</th>
                    <th>Shares</th>
                    <th>Saves</th>
                    <th>Post Clicks</th>
                    <th>Link</th>
                </tr>
            </thead>
            <tbody>
    `;

    posts.forEach(post => {
        const thumbnail = post.thumbnail || '';
        const title = post.title.substring(0, 60) + (post.title.length > 60 ? '...' : '');
        tableHTML += `
            <tr>
                <td><img src="${thumbnail}" alt="" class="table-thumbnail" onerror="this.style.display='none'"></td>
                <td class="table-post-title">${title}</td>
                <td>${post.network}</td>
                <td>${post.contentType || '-'}</td>
                <td>${post.date ? post.date.toLocaleDateString() : '-'}</td>
                <td>${formatNumber(post.impressions)}</td>
                <td>${formatNumber(post.reach)}</td>
                <td>${formatNumber(post.videoViews)}</td>
                <td>${formatNumber(post.engagements)}</td>
                <td>${formatNumber(post.reactions)}</td>
                <td>${formatNumber(post.likes)}</td>
                <td>${formatNumber(post.comments)}</td>
                <td>${formatNumber(post.shares)}</td>
                <td>${formatNumber(post.saves)}</td>
                <td>${formatNumber(post.postClicks)}</td>
                <td>${post.link ? `<a href="${post.link}" target="_blank" class="table-link">🔗</a>` : '-'}</td>
            </tr>
        `;
    });

    tableHTML += `
            </tbody>
        </table>
    `;
    
    tableContainer.innerHTML = tableHTML;
}

function renderPaginationControls(totalPages) {
    // Always append pagination to a unified container below both views
    let container = document.getElementById('paginationContainer');
    if (!container) {
        container = document.createElement('div');
        container.id = 'paginationContainer';
        const parent = document.getElementById('content');
        parent.appendChild(container);
    }
    container.innerHTML = ''; // clear old pagination
    if (totalPages <= 1) return; // no pagination needed
    
    const paginationDiv = document.createElement('div');
    paginationDiv.className = 'pagination-controls';
    
    let html = '';
    
    // Prev button
    html += `<button class="page-btn" ${currentPage === 1 ? 'disabled' : ''} onclick="changePage(${currentPage - 1})">⬅ Prev</button>`;
    
    // Page numbers
    const maxVisible = 5;
    let startPage = Math.max(1, currentPage - Math.floor(maxVisible / 2));
    let endPage = startPage + maxVisible - 1;
    if (endPage > totalPages) {
        endPage = totalPages;
        startPage = Math.max(1, endPage - maxVisible + 1);
    }
    
    if (startPage > 1) {
        html += `<button class="page-btn" onclick="changePage(1)">1</button>`;
        if (startPage > 2) html += `<span class="dots">...</span>`;
    }
    
    for (let i = startPage; i <= endPage; i++) {
        html += `
            <button class="page-btn ${i === currentPage ? 'active' : ''}" onclick="changePage(${i})">${i}</button>
        `;
    }
    
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) html += `<span class="dots">...</span>`;
        html += `<button class="page-btn" onclick="changePage(${totalPages})">${totalPages}</button>`;
    }
    
    // Next button
    html += `<button class="page-btn" ${currentPage === totalPages ? 'disabled' : ''} onclick="changePage(${currentPage + 1})">Next ➡</button>`;
    
    paginationDiv.innerHTML = html;
    container.appendChild(paginationDiv);
} // ← You missed this closing brace!

function changePage(page) {
    currentPage = page;
    updateContentDisplay();

    // Scroll to top of content
    const topElement = document.querySelector('.content-controls-section');
    if (topElement) topElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
}


/**
 * Switch content view (grid/table)
 */
function switchContentView(view) {
    currentContentView = view;
    
    const gridView = document.getElementById('contentGridView');
    const tableView = document.getElementById('contentTableView');
    const gridBtn = document.getElementById('contentGridViewBtn');
    const tableBtn = document.getElementById('contentTableViewBtn');
    
    if (view === 'grid') {
        gridView.style.display = 'grid';
        tableView.style.display = 'none';
        gridBtn.classList.add('active');
        tableBtn.classList.remove('active');
    } else {
        gridView.style.display = 'none';
        tableView.style.display = 'block';
        gridBtn.classList.remove('active');
        tableBtn.classList.add('active');
    }
    
    updateContentDisplay();
}

// ============================================================================
// RESULTS TAB
// ============================================================================

/**
 * Get unique networks/platforms from data
 */
function getAvailablePlatforms() {
    const platforms = new Set();
    recapData.forEach(record => {
        if (record.network) {
            platforms.add(record.network);
        }
    });
    return Array.from(platforms).sort();
}

/**
 * Render platform filter buttons
 */
function renderPlatformFilters() {
    const container = document.getElementById('platformFilters');
    if (!container) return;
    
    container.innerHTML = '';
    
    const platforms = ['All', ...getAvailablePlatforms()];
    
    platforms.forEach(platform => {
        const button = document.createElement('button');
        button.className = 'platform-tab' + (currentPlatform === platform.toLowerCase() ? ' active' : '');
        button.textContent = platform;
        button.onclick = () => selectPlatform(platform);
        container.appendChild(button);
    });
}

/**
 * Select platform filter
 */
function selectPlatform(platform) {
    currentPlatform = platform.toLowerCase();
    renderPlatformFilters();
    filterAndRenderResults();
}

/**
 * Set default date range
 */
function setDefaultDateRange() {
    if (recapData.length === 0) return;
    
    const dates = recapData.map(r => r.date).filter(d => d);
    const minDate = new Date(Math.min(...dates));
    const maxDate = new Date(Math.max(...dates));
    
    document.getElementById('startDate').valueAsDate = minDate;
    document.getElementById('endDate').valueAsDate = maxDate;
    
    startDate = minDate;
    endDate = maxDate;
}

/**
 * Apply date range filter
 */
function applyDateRange() {
    const startInput = document.getElementById('startDate').value;
    const endInput = document.getElementById('endDate').value;
    
    if (startInput) startDate = new Date(startInput);
    if (endInput) endDate = new Date(endInput);
    
    filterAndRenderResults();
}

/**
 * Change view type (daily/weekly/monthly)
 */
function changeViewType(viewType) {
    currentViewType = viewType;
    
    document.querySelectorAll('.view-toggle-btn').forEach(btn => {
        btn.classList.remove('active');
        if (btn.dataset.view === viewType) {
            btn.classList.add('active');
        }
    });
    
    filterAndRenderResults();
}

/**
 * Filter data based on current selections
 */
function filterAndRenderResults() {
    // Filter by platform
    filteredData = recapData.filter(record => {
        if (currentPlatform !== 'all' && record.network.toLowerCase() !== currentPlatform) {
            return false;
        }
        return true;
    });
    
    // Filter by date range
    if (startDate && endDate) {
        filteredData = filteredData.filter(record => {
            return record.date >= startDate && record.date <= endDate;
        });
    }
    
    console.log('Filtered data:', filteredData);
    
    updateResultsStats();
    renderCharts();
}

/**
 * Update results summary statistics
 */
function updateResultsStats() {
    const stats = calculateStats(filteredData);
    
    document.getElementById('resultEngagements').textContent = formatNumber(stats.totalEngagements);
    document.getElementById('resultImpressions').textContent = formatNumber(stats.totalImpressions);
    document.getElementById('resultAudience').textContent = formatNumber(stats.maxAudience);
    document.getElementById('resultVideoViews').textContent = formatNumber(stats.totalVideoViews);
    document.getElementById('resultAudienceGrowth').textContent = formatNumber(stats.audienceGrowth);
    
    // Update audience label
    const audienceLabel = document.getElementById('audienceLabel');
    if (audienceLabel) {
        if (currentPlatform === 'all') {
            audienceLabel.textContent = 'Total Audience';
        } else {
            audienceLabel.textContent = 'Audience';
        }
    }
}

/**
 * Calculate statistics from filtered data
 */
function calculateStats(data) {
    const stats = {
        totalEngagements: 0,
        totalImpressions: 0,
        totalReach: 0,
        maxAudience: 0,
        totalVideoViews: 0,
        totalPostClicksAll: 0,
        audienceGrowth: 0
    };
    
    if (data.length === 0) return stats;
    
    // If viewing "All", sum max audience from each network
    if (currentPlatform === 'all') {
        const networkMaxAudience = {};
        
        data.forEach(record => {
            const network = record.network.toLowerCase();
            if (!networkMaxAudience[network]) {
                networkMaxAudience[network] = 0;
            }
            networkMaxAudience[network] = Math.max(networkMaxAudience[network], record.audience);
        });
        
        stats.maxAudience = Object.values(networkMaxAudience).reduce((sum, val) => sum + val, 0);
    } else {
        data.forEach(record => {
            stats.maxAudience = Math.max(stats.maxAudience, record.audience);
        });
    }
    
    // Calculate other totals
    data.forEach(record => {
        stats.totalEngagements += record.engagements;
        stats.totalImpressions += record.impressions;
        stats.totalReach += record.reach || 0;
        stats.totalVideoViews += record.videoViews;
        stats.totalPostClicksAll += record.postClicksAll || 0;
        stats.audienceGrowth += record.audienceGrowth;
    });
    
    return stats;
}

/**
 * Aggregate data based on view type
 */
function aggregateData(data, viewType) {
    if (viewType === 'daily') {
        return aggregateByDay(data);
    } else if (viewType === 'weekly') {
        return aggregateByWeek(data);
    } else if (viewType === 'monthly') {
        return aggregateByMonth(data);
    }
    return aggregateByDay(data);
}

/**
 * Aggregate data by day
 */
function aggregateByDay(data) {
    const grouped = {};
    
    data.forEach(record => {
        const dateKey = record.date.toISOString().split('T')[0];
        
        if (!grouped[dateKey]) {
            grouped[dateKey] = {
                date: record.date,
                label: formatDate(record.date),
                audience: 0,
                impressions: 0,
                reach: 0,
                videoViews: 0,
                engagements: 0,
                postClicksAll: 0,
                count: 0
            };
        }
        
        grouped[dateKey].audience += record.audience; // Sum instead of Math.max to handle negatives
        grouped[dateKey].impressions += record.impressions;
        grouped[dateKey].reach += record.reach || 0;
        grouped[dateKey].videoViews += record.videoViews;
        grouped[dateKey].engagements += record.engagements;
        grouped[dateKey].postClicksAll += record.postClicksAll || 0;
        grouped[dateKey].count++;
    });
    
    return Object.values(grouped).sort((a, b) => a.date - b.date);
}

/**
 * Aggregate data by week
 */
function aggregateByWeek(data) {
    const grouped = {};
    
    data.forEach(record => {
        const weekKey = getWeekKey(record.date);
        
        if (!grouped[weekKey]) {
            grouped[weekKey] = {
                date: getWeekStart(record.date),
                label: `Week of ${formatDate(getWeekStart(record.date))}`,
                audience: 0,
                impressions: 0,
                reach: 0,
                videoViews: 0,
                engagements: 0,
                postClicksAll: 0,
                count: 0
            };
        }
        
        grouped[weekKey].audience += record.audience; // Sum instead of Math.max to handle negatives
        grouped[weekKey].impressions += record.impressions;
        grouped[weekKey].reach += record.reach || 0;
        grouped[weekKey].videoViews += record.videoViews;
        grouped[weekKey].engagements += record.engagements;
        grouped[weekKey].postClicksAll += record.postClicksAll || 0;
        grouped[weekKey].count++;
    });
    
    return Object.values(grouped).sort((a, b) => a.date - b.date);
}

/**
 * Aggregate data by month
 */
function aggregateByMonth(data) {
    const grouped = {};
    
    data.forEach(record => {
        const monthKey = `${record.date.getFullYear()}-${String(record.date.getMonth() + 1).padStart(2, '0')}`;
        
        if (!grouped[monthKey]) {
            grouped[monthKey] = {
                date: new Date(record.date.getFullYear(), record.date.getMonth(), 1),
                label: formatMonthYear(record.date),
                audience: 0,
                impressions: 0,
                reach: 0,
                videoViews: 0,
                engagements: 0,
                postClicksAll: 0,
                count: 0
            };
        }
        
        grouped[monthKey].audience += record.audience; // Sum instead of Math.max to handle negatives
        grouped[monthKey].impressions += record.impressions;
        grouped[monthKey].reach += record.reach || 0;
        grouped[monthKey].videoViews += record.videoViews;
        grouped[monthKey].engagements += record.engagements;
        grouped[monthKey].postClicksAll += record.postClicksAll || 0;
        grouped[monthKey].count++;
    });
    
    return Object.values(grouped).sort((a, b) => a.date - b.date);
}

/**
 * Get week key for grouping
 */
function getWeekKey(date) {
    const weekStart = getWeekStart(date);
    return `${weekStart.getFullYear()}-W${getWeekNumber(weekStart)}`;
}

/**
 * Get start of week (Monday)
 */
function getWeekStart(date) {
    const d = new Date(date);
    const day = d.getDay();
    const diff = d.getDate() - day + (day === 0 ? -6 : 1);
    return new Date(d.setDate(diff));
}

/**
 * Get week number
 */
function getWeekNumber(date) {
    const d = new Date(Date.UTC(date.getFullYear(), date.getMonth(), date.getDate()));
    const dayNum = d.getUTCDay() || 7;
    d.setUTCDate(d.getUTCDate() + 4 - dayNum);
    const yearStart = new Date(Date.UTC(d.getUTCFullYear(), 0, 1));
    return Math.ceil((((d - yearStart) / 86400000) + 1) / 7);
}

/**
 * Render all charts
 */
function renderCharts() {
    const aggregated = aggregateData(filteredData, currentViewType);
    
    renderChart('audienceChart', aggregated, 'audience', '👥 Audience', '#4ecdc4');
    renderChart('impressionsChart', aggregated, 'impressions', '👁 Impressions', '#45b7d1');
    renderChart('videoViewsChart', aggregated, 'videoViews', '📹 Video Views', '#96ceb4');
    renderChart('engagementsChart', aggregated, 'engagements', '🤝 Engagements', '#ff6b8a');
}

/**
 * Render individual chart with dynamic Y-axis
 */
function renderChart(canvasId, data, metric, label, color) {
    const canvas = document.getElementById(canvasId);
    if (!canvas) return;
    
    const ctx = canvas.getContext('2d');
    
    // Destroy existing chart
    if (charts[canvasId]) {
        charts[canvasId].destroy();
    }
    
    const labels = data.map(d => d.label);
    const values = data.map(d => d[metric]);
    
    // Calculate min and max values
    const minValue = Math.min(...values);
    const maxValue = Math.max(...values);
    const range = maxValue - minValue;
    
    // Use dynamic Y-axis if range is less than 15% of max value
    const useDynamicAxis = range > 0 && (range / Math.max(Math.abs(maxValue), Math.abs(minValue))) < 0.15;
    
    let yAxisMin = 0;
    let yAxisMax = maxValue;
    
    if (useDynamicAxis) {
        const padding = range * 0.1;
        // Allow negative values - don't force min to 0
        yAxisMin = minValue - padding;
        yAxisMax = maxValue + padding;
    } else if (minValue < 0) {
        // If there are negative values, set axis min to accommodate them
        yAxisMin = minValue * 1.1; // Add 10% padding for negative values
    }
    
    charts[canvasId] = new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [{
                label: label,
                data: values,
                borderColor: color,
                backgroundColor: hexToRgba(color, 0.1),
                borderWidth: 3,
                fill: true,
                tension: 0.4,
                pointRadius: 4,
                pointHoverRadius: 6,
                pointBackgroundColor: color,
                pointBorderColor: '#fff',
                pointBorderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    padding: 12,
                    titleFont: {
                        size: 14,
                        weight: 'bold'
                    },
                    bodyFont: {
                        size: 13
                    },
                    callbacks: {
                        label: function(context) {
                            return `${label}: ${formatNumber(context.parsed.y)}`;
                        }
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: !useDynamicAxis && minValue >= 0,
                    min: yAxisMin,
                    max: yAxisMax,
                    ticks: {
                        callback: function(value) {
                            return formatNumber(value);
                        }
                    },
                    grid: {
                        color: 'rgba(0, 0, 0, 0.05)'
                    }
                },
                x: {
                    grid: {
                        display: false
                    },
                    ticks: {
                        maxRotation: 45,
                        minRotation: 45
                    }
                }
            }
        }
    });
}

/**
 * Refresh data
 */
async function refreshData() {
    showLoading();
    try {
        await loadRecapData();
        await loadPostsData();
        await loadCommentAnalysisData();
        filterAndRenderResults();
        populateContentFilters();
        filterContent();
        renderCommentAnalysis();
        hideLoading();
    } catch (error) {
        showError('Failed to refresh data: ' + error.message);
    }
}
// ============================================================================
// COMMENT ANALYSIS FUNCTIONS
// ============================================================================

/**
 * Load Comment Analysis data from Google Sheets
 */
/**
 * Load Comment Analysis data from Google Sheets
 */
async function loadCommentAnalysisData() {
    try {
        console.log('Loading comment analysis data from CommentAnalysis sheet...');
        const data = await fetchSheetData(currentCampaignId, 'CommentAnalysis', 'A1:F100');
        
        if (!data || data.length < 2) {
            console.warn('No comment analysis data found');
            commentAnalysisData = [];
            return;
        }
        
        console.log('Raw comment analysis data:', data);
        
        // Parse the data (skip header row)
        commentAnalysisData = [];
        let currentCategory = '';
        
        for (let i = 1; i < data.length; i++) {
            const row = data[i];
            let category = row[0];
            const metric = row[1];
            const value = row[2];
            const imageUrl = row[4]; // Column E (index 4)
            const imageTitle = row[5]; // Column F (index 5) - CommentSection
            
            // Handle merged cells - if category is empty, use the last non-empty category
            if (category && category.trim() !== '') {
                currentCategory = category;
            } else {
                category = currentCategory;
            }
            
            console.log(`Row ${i}:`, { category, metric, value, imageUrl, imageTitle });
            
            if (category && metric && value !== null && value !== undefined && value !== '') {
                commentAnalysisData.push({
                    category: category,
                    metric: metric,
                    value: value,
                    imageUrl: imageUrl || null,
                    imageTitle: imageTitle || null
                });
            } else {
                console.log(`Row ${i} skipped - missing data:`, { category, metric, value });
            }
        }
        
        console.log('Loaded comment analysis data:', commentAnalysisData.length, 'rows');
        console.log('Comment analysis data:', commentAnalysisData);
        
    } catch (error) {
        console.error('Error loading comment analysis data:', error);
        commentAnalysisData = [];
    }
}

/**
 * Load Paid Media data from PaidMedia sheet
 */
async function loadPaidMediaData() {
    try {
        console.log('Loading paid media data from PaidMedia sheet...');
        const data = await fetchSheetData(currentCampaignId, 'PaidMedia', 'A1:Z100');
        
        if (!data || data.length < 2) {
            console.warn('No paid media data found');
            paidMediaData = [];
            paidMediaHeaders = [];
            return;
        }
        
        console.log('Raw paid media data:', data);
        
        // Store headers (first row)
        paidMediaHeaders = data[0];
        console.log('Paid media headers:', paidMediaHeaders);
        
        // Parse data rows
        paidMediaData = [];
        for (let i = 1; i < data.length; i++) {
            const row = data[i];
            
            // Skip empty rows
            if (!row[0]) continue;
            
            // Log raw values for debugging
            console.log(`Row ${i} raw values:`, {
                socialMedia: row[0],
                spend: row[1],
                impressions: row[2],
                cpm: row[3]
            });
            
            const record = {
                socialMedia: (row[0] || '').trim(), // Column A
                spend: parseNumber(row[1]), // Column B - handles $ sign
                impressions: parseNumber(row[2]), // Column C
                cpm: parseNumber(row[3]), // Column D - handles $ sign
                additionalMetrics: {} // For dynamic columns
            };
            
            console.log(`Row ${i} parsed values:`, {
                socialMedia: record.socialMedia,
                spend: record.spend,
                impressions: record.impressions,
                cpm: record.cpm
            });
            
            // Add additional metrics from remaining columns (E onwards)
            for (let j = 4; j < row.length && j < paidMediaHeaders.length; j++) {
                const header = paidMediaHeaders[j];
                const value = row[j];
                
                if (header && value !== null && value !== undefined && value !== '') {
                    // Store both the raw value and whether it contains currency symbol
                    const rawValue = value.toString();
                    const hasCurrency = rawValue.includes('$') || rawValue.includes('€') || rawValue.includes('£') || rawValue.includes('¥');
                    
                    record.additionalMetrics[header] = {
                        value: value,
                        isCurrency: hasCurrency
                    };
                }
            }
            
            if (record.socialMedia) {
                paidMediaData.push(record);
            }
        }
        
        console.log('Loaded paid media data:', paidMediaData.length, 'records');
        console.log('Paid media data:', paidMediaData);
        
    } catch (error) {
        console.error('Error loading paid media data:', error);
        paidMediaData = [];
        paidMediaHeaders = [];
    }
}

/**
 * Load Info data from Info sheet (columns A-E)
 */
async function loadInfoData() {
    try {
        console.log('Loading info data from Info sheet...');
        const data = await fetchSheetData(currentCampaignId, 'Info', 'A1:E2');
        
        if (!data || data.length < 2) {
            console.warn('No info data found');
            infoData = {};
            return;
        }
        
        console.log('Raw info data:', data);
        
        // Row 0 contains headers, Row 1 contains values
        const headers = data[0];
        const values = data[1];
        
        infoData = {
            headerA: values[0] ? values[0].toString().trim() : '', // Column A
            headerB: values[1] ? values[1].toString().trim() : ''  // Column B
        };
        
        // Process columns C, D, E (indices 2, 3, 4 in the fetched range)
        for (let i = 2; i < headers.length && i < 5; i++) {
            const header = headers[i];
            const value = values[i];
            
            if (header && header.trim() !== '') {
                infoData[header.trim()] = (value || '').toString().trim();
            }
        }
        
        console.log('Loaded info data:', infoData);
        
        // Update header subtitle
        updateHeaderSubtitle();
        
    } catch (error) {
        console.error('Error loading info data:', error);
        infoData = {};
    }
}

/**
 * Update header subtitle with info from columns A and B
 */
function updateHeaderSubtitle() {
    const subtitleEl = document.getElementById('headerSubtitle');
    if (!subtitleEl) return;
    
    const partA = infoData.headerA || '';
    const partB = infoData.headerB || '';
    
    if (partA && partB) {
        subtitleEl.textContent = `${partA} - ${partB}`;
    } else if (partA) {
        subtitleEl.textContent = partA;
    } else if (partB) {
        subtitleEl.textContent = partB;
    } else {
        subtitleEl.textContent = 'Performance Overview & Analytics';
    }
}

/**
 * Get score rating text and color based on score value
 */
function getScoreRating(score) {
    if (score < 3) return { text: 'Poor', color: '#c62828', class: 'score-poor' };
    if (score < 4.5) return { text: 'Fair', color: '#f57c00', class: 'score-fair' };
    if (score < 6) return { text: 'Average', color: '#fdd835', class: 'score-average' };
    if (score < 7.5) return { text: 'Good', color: '#43a047', class: 'score-good' };
    if (score < 9) return { text: 'Very Good', color: '#1976d2', class: 'score-very-good' };
    return { text: 'Excellent', color: '#6a1b9a', class: 'score-excellent' };
}

/**
 * Render Comment Analysis section
 */
function renderCommentAnalysis() {
    console.log('Rendering comment analysis...');
    
    const scoresContainer = document.getElementById('commentAnalysisScores');
    const contentContainer = document.getElementById('commentAnalysisContent');
    const overallScoreValueEl = document.getElementById('overallScoreValue');
    const overallScoreTextEl = document.getElementById('overallScoreText');
    
    if (!contentContainer || !scoresContainer) {
        console.warn('Comment analysis containers not found');
        return;
    }
    
    if (!commentAnalysisData || commentAnalysisData.length === 0) {
        scoresContainer.innerHTML = '';
        contentContainer.innerHTML = `
            <div class="comment-analysis-empty">
                <div class="comment-analysis-empty-icon">💬</div>
                <h3>No Comment Analysis Data Available</h3>
                <p>Make sure your Google Sheet has a "CommentAnalysis" tab with the proper structure.</p>
            </div>
        `;
        
        // Reset overall score
        if (overallScoreValueEl) overallScoreValueEl.textContent = '-';
        if (overallScoreTextEl) overallScoreTextEl.textContent = 'No Data';
        return;
    }
    
    // Parse the data into categories
    const sentimentData = {};
    const relevanceData = {};
    const consensusData = {};
    let overallScore = 0;
    let overallRating = 'Good';
    let sentimentScore = 0;
    let relevanceScore = 0;
    let consensusScore = 0;
    
    console.log('Parsing comment analysis data...');
    
    commentAnalysisData.forEach(item => {
        const { category, metric, value } = item;
        
        console.log(`Category: ${category}, Metric: ${metric}, Value: ${value}`);
        
        if (category === 'Sentiment') {
            sentimentData[metric] = value;
            if (metric === 'Sentiment Score') {
                sentimentScore = parseFloat(value);
            }
        } else if (category === 'Relevance') {
            relevanceData[metric] = value;
            if (metric === 'Relevance Score') {
                relevanceScore = parseFloat(value);
            }
        } else if (category === 'Consensus') {
            consensusData[metric] = value;
            if (metric === 'Consensus Score') {
                consensusScore = parseFloat(value);
            }
        } else if (category === 'Overall') {
            if (metric === 'Total Weighted Score') {
                overallScore = parseFloat(value);
                // If individual scores weren't found, estimate them from overall
                if (sentimentScore === 0 && relevanceScore === 0 && consensusScore === 0) {
                    sentimentScore = overallScore;
                    relevanceScore = overallScore;
                    consensusScore = overallScore;
                }
            } else if (metric === 'Overall Rating') {
                overallRating = value;
            }
        }
    });
    
    console.log('Parsed sentimentData:', sentimentData);
    console.log('Parsed relevanceData:', relevanceData);
    console.log('Parsed consensusData:', consensusData);
    console.log('Overall score:', overallScore, 'Rating:', overallRating);
    console.log('Extracted scores:', { sentimentScore, relevanceScore, consensusScore });
    
    // Update overall score in header
    const ratingInfo = getScoreRating(overallScore);
    if (overallScoreValueEl) {
        overallScoreValueEl.textContent = overallScore.toFixed(1);
        overallScoreValueEl.className = ratingInfo.class;
    }
    if (overallScoreTextEl) {
        overallScoreTextEl.textContent = overallRating;
        overallScoreTextEl.className = ratingInfo.class;
    }
    
    // Build score cards HTML
    let scoresHtml = '';
    
    // Sentiment Score Card
    if (Object.keys(sentimentData).length > 0) {
        const sentimentRating = getScoreRating(sentimentScore);
        const sentimentDescription = getSentimentDescription(sentimentScore);
        
        scoresHtml += `
            <div class="score-card">
                <div class="score-card-header">
                    <span class="score-card-icon">💭</span>
                    <h4 class="score-card-title">Sentiment</h4>
                </div>
                <div class="score-card-description">${sentimentDescription}</div>
                <div class="score-display">
                    <div class="score-bar-container">
                        <div class="score-bar-fill" style="width: ${(sentimentScore / 10) * 100}%; background: ${sentimentRating.color};"></div>
                    </div>
                    <div class="score-number" style="color: ${sentimentRating.color};">${sentimentScore.toFixed(1)}</div>
                </div>
                <p class="score-explanation">Sentiment score measures how positive or negative the comments are.</p>
            </div>
        `;
    }
    
    // Relevance Score Card
    if (Object.keys(relevanceData).length > 0) {
        const relevanceRating = getScoreRating(relevanceScore);
        const relevanceDescription = getRelevanceDescription(relevanceScore);
        
        scoresHtml += `
            <div class="score-card">
                <div class="score-card-header">
                    <span class="score-card-icon">🎯</span>
                    <h4 class="score-card-title">Relevance</h4>
                </div>
                <div class="score-card-description">${relevanceDescription}</div>
                <div class="score-display">
                    <div class="score-bar-container">
                        <div class="score-bar-fill" style="width: ${(relevanceScore / 10) * 100}%; background: ${relevanceRating.color};"></div>
                    </div>
                    <div class="score-number" style="color: ${relevanceRating.color};">${relevanceScore.toFixed(1)}</div>
                </div>
                <p class="score-explanation">Relevance measures how related the comments are to the post.</p>
            </div>
        `;
    }
    
    // Consensus Score Card
    if (Object.keys(consensusData).length > 0) {
        const consensusRating = getScoreRating(consensusScore);
        const consensusDescription = getConsensusDescription(consensusScore);
        
        scoresHtml += `
            <div class="score-card">
                <div class="score-card-header">
                    <span class="score-card-icon">🤝</span>
                    <h4 class="score-card-title">Consensus</h4>
                </div>
                <div class="score-card-description">${consensusDescription}</div>
                <div class="score-display">
                    <div class="score-bar-container">
                        <div class="score-bar-fill" style="width: ${(consensusScore / 10) * 100}%; background: ${consensusRating.color};"></div>
                    </div>
                    <div class="score-number" style="color: ${consensusRating.color};">${consensusScore.toFixed(1)}</div>
                </div>
                <p class="score-explanation">Consensus is a measure of how much the comments agree with each other.</p>
            </div>
        `;
    }
    
    scoresContainer.innerHTML = scoresHtml;
    
    // Build detail sections HTML
    let detailsHtml = '';
    
    // Sentiment Detail Section
    if (Object.keys(sentimentData).length > 0) {
        detailsHtml += `
            <div class="comment-section">
                <h3>Sentiment Breakdown</h3>
                <div class="metric-breakdown">
                    ${sentimentData['Very Positive'] ? `
                        <div class="metric-item" style="background: linear-gradient(135deg, #43a047 0%, #66bb6a 100%);">
                            <div class="metric-label">Very Positive</div>
                            <div class="metric-value">${formatPercent(sentimentData['Very Positive'])}</div>
                        </div>
                    ` : ''}
                    ${sentimentData['Positive Leaning'] ? `
                        <div class="metric-item" style="background: linear-gradient(135deg, #7cb342 0%, #9ccc65 100%);">
                            <div class="metric-label">Positive Leaning</div>
                            <div class="metric-value">${formatPercent(sentimentData['Positive Leaning'])}</div>
                        </div>
                    ` : ''}
                    ${sentimentData['Negative Leaning'] || sentimentData['Negative Leanin'] ? `
                        <div class="metric-item" style="background: linear-gradient(135deg, #fb8c00 0%, #ffa726 100%);">
                            <div class="metric-label">Negative Leaning</div>
                            <div class="metric-value">${formatPercent(sentimentData['Negative Leaning'] || sentimentData['Negative Leanin'])}</div>
                        </div>
                    ` : ''}
                    ${sentimentData['Negative'] ? `
                        <div class="metric-item" style="background: linear-gradient(135deg, #e53935 0%, #ef5350 100%);">
                            <div class="metric-label">Negative</div>
                            <div class="metric-value">${formatPercent(sentimentData['Negative'])}</div>
                        </div>
                    ` : ''}
                </div>
            </div>
        `;
    }
    
    // Relevance Detail Section
    if (Object.keys(relevanceData).length > 0) {
        detailsHtml += `
            <div class="comment-section">
                <h3>Relevance Breakdown</h3>
                <div class="metric-breakdown">
                    ${relevanceData['On-Topic'] ? `
                        <div class="metric-item" style="background: linear-gradient(135deg, #1976d2 0%, #42a5f5 100%);">
                            <div class="metric-label">On-Topic</div>
                            <div class="metric-value">${formatPercent(relevanceData['On-Topic'])}</div>
                        </div>
                    ` : ''}
                    ${relevanceData['Relevant'] ? `
                        <div class="metric-item" style="background: linear-gradient(135deg, #00897b 0%, #26a69a 100%);">
                            <div class="metric-label">Relevant</div>
                            <div class="metric-value">${formatPercent(relevanceData['Relevant'])}</div>
                        </div>
                    ` : ''}
                    ${relevanceData['General'] ? `
                        <div class="metric-item" style="background: linear-gradient(135deg, #fdd835 0%, #ffeb3b 100%);">
                            <div class="metric-label">General</div>
                            <div class="metric-value">${formatPercent(relevanceData['General'])}</div>
                        </div>
                    ` : ''}
                    ${relevanceData['Off-Topic'] ? `
                        <div class="metric-item" style="background: linear-gradient(135deg, #f4511e 0%, #ff7043 100%);">
                            <div class="metric-label">Off-Topic</div>
                            <div class="metric-value">${formatPercent(relevanceData['Off-Topic'])}</div>
                        </div>
                    ` : ''}
                </div>
            </div>
        `;
    }
    
    // Consensus Detail Section
    if (Object.keys(consensusData).length > 0) {
        detailsHtml += `
            <div class="comment-section">
                <h3>Consensus Breakdown</h3>
                <div class="metric-breakdown">
                    ${consensusData['Agreeing'] ? `
                        <div class="metric-item" style="background: linear-gradient(135deg, #388e3c 0%, #4caf50 100%);">
                            <div class="metric-label">Agreeing</div>
                            <div class="metric-value">${formatPercent(consensusData['Agreeing'])}</div>
                        </div>
                    ` : ''}
                    ${consensusData['Discussing'] ? `
                        <div class="metric-item" style="background: linear-gradient(135deg, #0288d1 0%, #03a9f4 100%);">
                            <div class="metric-label">Discussing</div>
                            <div class="metric-value">${formatPercent(consensusData['Discussing'])}</div>
                        </div>
                    ` : ''}
                    ${consensusData['Debating'] ? `
                        <div class="metric-item" style="background: linear-gradient(135deg, #f57c00 0%, #ff9800 100%);">
                            <div class="metric-label">Debating</div>
                            <div class="metric-value">${formatPercent(consensusData['Debating'])}</div>
                        </div>
                    ` : ''}
                    ${consensusData['Fighting'] ? `
                        <div class="metric-item" style="background: linear-gradient(135deg, #c62828 0%, #e53935 100%);">
                            <div class="metric-label">Fighting</div>
                            <div class="metric-value">${formatPercent(consensusData['Fighting'])}</div>
                        </div>
                    ` : ''}
                </div>
            </div>
        `;
    }
    
    contentContainer.innerHTML = detailsHtml;
    
    // Render Best Comments Gallery
    renderBestCommentsGallery();
}

/**
 * Render Best Comments Gallery from image URLs
 */
function renderBestCommentsGallery() {
    const galleryContainer = document.getElementById('bestCommentsGallery');
    if (!galleryContainer) return;
    
    // Extract unique image URLs and titles from commentAnalysisData
    const imageData = commentAnalysisData
        .filter(item => item.imageUrl && item.imageUrl.trim() !== '')
        .map(item => ({
            url: convertGoogleDriveUrl(item.imageUrl).replace('sz=w400', 'sz=w2000'),
            title: item.imageTitle || ''
        }));
    
    // Remove duplicates based on URL
    const uniqueImageData = [];
    const seenUrls = new Set();
    imageData.forEach(item => {
        if (!seenUrls.has(item.url)) {
            seenUrls.add(item.url);
            uniqueImageData.push(item);
        }
    });
    
    console.log('Best comment images found:', uniqueImageData);
    
    if (uniqueImageData.length === 0) {
        galleryContainer.innerHTML = '<div class="best-comments-empty">No comment screenshots available</div>';
        return;
    }
    
    // Render images with titles and click handlers
    galleryContainer.innerHTML = uniqueImageData.map((item, index) => {
        const titleHtml = item.title && item.title.trim() !== '' 
            ? `<h3 style="margin: 0 0 1rem 0; font-size: 1.125rem; font-weight: 600; color: #1d1d1f;">${item.title}</h3>`
            : '';
        
        return `
        <div class="best-comment-item-wrapper" style="margin-bottom: 2rem;">
            ${titleHtml}
            <div class="best-comment-item" onclick="openLightbox('${item.url}', '${item.title || 'Comment Screenshot ' + (index + 1)}')">
                <img src="${item.url}" alt="${item.title || 'Comment screenshot ' + (index + 1)}" loading="lazy" 
                     onerror="console.error('Failed to load image:', this.src); this.parentElement.innerHTML = '<div class=\\'best-comments-empty\\'>Image failed to load</div>'">
            </div>
        </div>
        `;
    }).join('');
    
    // Log each image attempt
    uniqueImageData.forEach((item, i) => {
        console.log(`Image ${i + 1}:`, item.url, 'Title:', item.title);
    });
}

/**
 * Open lightbox with image
 */
function openLightbox(imageSrc, caption) {
    const lightbox = document.getElementById('imageLightbox');
    const lightboxImg = document.getElementById('lightboxImage');
    const lightboxCaption = document.getElementById('lightboxCaption');
    
    if (lightbox && lightboxImg) {
        lightbox.style.display = 'block';
        lightboxImg.src = imageSrc;
        if (lightboxCaption) {
            lightboxCaption.textContent = caption;
        }
    }
}

/**
 * Close lightbox
 */
function closeLightbox() {
    const lightbox = document.getElementById('imageLightbox');
    if (lightbox) {
        lightbox.style.display = 'none';
    }
}

function getSentimentDescription(score) {
    if (score < 3) return 'The comments are very negative';
    if (score < 4.5) return 'The comments are somewhat negative';
    if (score < 6) return 'The comments are mixed';
    if (score < 7.5) return 'The comments are mostly positive';
    if (score < 9) return 'The comments are overwhelmingly positive';
    return 'The comments are exceptionally positive';
}

/**
 * Get relevance description based on score
 */
function getRelevanceDescription(score) {
    if (score < 3) return 'The comments are mostly off-topic';
    if (score < 4.5) return 'The comments are somewhat off-topic';
    if (score < 6) return 'The comments are moderately relevant';
    if (score < 7.5) return 'The comments are mostly relevant';
    if (score < 9) return 'The comments are highly relevant to the post';
    return 'The comments are extremely relevant to the post';
}

/**
 * Get consensus description based on score
 */
function getConsensusDescription(score) {
    if (score < 3) return 'Commenters are highly divided';
    if (score < 4.5) return 'Commenters have varied opinions';
    if (score < 6) return 'Commenters show some agreement';
    if (score < 7.5) return 'Commenters are strongly in agreement';
    if (score < 9) return 'Commenters are in overwhelming agreement';
    return 'Commenters are in complete consensus';
}

/**
 * Format value as percentage
 */
function formatPercent(value) {
    if (typeof value === 'string' && value.includes('%')) {
        return value;
    }
    const num = parseFloat(value);
    return isNaN(num) ? '0%' : `${(num * 100).toFixed(1)}%`;
}

/**
 * Render Overall Score Chart (Doughnut)
 */
// This function is no longer used - overall score is now displayed inline in header
// function renderOverallScoreChart(score, rating) {
//     // ... removed for brevity
// }

/**
 * Filter content by date range
 */
function filterContentByDate() {
    filterContent();
}

/**
 * Reset content date range filter
 */
function resetContentDateRange() {
    document.getElementById('contentStartDate').value = '';
    document.getElementById('contentEndDate').value = '';
    filterContent();
}

// ============================================================================
// PAID MEDIA TAB
// ============================================================================

/**
 * Get emoji/icon for paid media metrics
 */
function getPaidMediaIcon(metricName) {
    const lowerMetric = metricName.toLowerCase();
    const iconMap = {
        'spend': '💰',
        'budget': '💵',
        'cost': '💸',
        'impressions': '👁️',
        'views': '📊',
        'cpm': '📈',
        'cpc': '💳',
        'ctr': '🖱️',
        'clicks': '👆',
        'conversions': '✅',
        'roas': '📊',
        'roi': '💹',
        'reach': '🎯',
        'engagement': '❤️',
        'frequency': '🔄',
        'results': '🎯',
        'link clicks': '🔗',
        'landing page views': '🏠',
        'purchases': '🛒',
        'revenue': '💰',
        'leads': '📋',
        'add to cart': '🛒',
        'video views': '📹',
        'saves': '🔖',
        'shares': '📤',
        'comments': '💬',
        'likes': '👍',
        'reactions': '⭐'
    };
    
    // Check for partial matches
    for (const [key, icon] of Object.entries(iconMap)) {
        if (lowerMetric.includes(key)) {
            return icon;
        }
    }
    
    return '📊'; // Default icon
}

/**
 * Render Paid Media section
 */
function renderPaidMedia() {
    console.log('Rendering paid media section...');
    
    const filtersContainer = document.getElementById('paidMediaFilters');
    const metricsGridContainer = document.getElementById('paidMediaMetricsGrid');
    
    if (!filtersContainer || !metricsGridContainer) {
        console.warn('Paid media containers not found');
        return;
    }
    
    if (!paidMediaData || paidMediaData.length === 0) {
        filtersContainer.innerHTML = '<div class="platform-filters-empty">No paid media data available</div>';
        metricsGridContainer.innerHTML = '<div class="paid-media-empty">No paid media data available</div>';
        return;
    }
    
    // Render platform filters
    const platforms = ['all', ...new Set(paidMediaData.map(item => item.socialMedia))];
    filtersContainer.innerHTML = platforms.map(platform => {
        const displayName = platform === 'all' ? 'All Platforms' : platform;
        const activeClass = currentPaidMediaPlatform === platform ? 'active' : '';
        
        // Get icon for the platform
        let iconHtml = '';
        if (platform !== 'all') {
            const iconUrl = getPlatformIconUrl(platform);
            if (iconUrl) {
                // Use actual platform icon image
                iconHtml = `<img src="${iconUrl}" alt="${platform}" style="width: 20px; height: 20px; margin-right: 0.5rem; vertical-align: middle;">`;
            } else {
                // Use emoji fallback
                const icon = getPlatformIcon(platform);
                iconHtml = `<span style="margin-right: 0.5rem;">${icon}</span>`;
            }
        }
        
        return `<button class="platform-tab ${activeClass}" onclick="filterPaidMedia('${platform}')">${iconHtml}${displayName}</button>`;
    }).join('');
    
    // Filter data based on selected platform
    const filteredData = currentPaidMediaPlatform === 'all' 
        ? paidMediaData 
        : paidMediaData.filter(item => item.socialMedia.toLowerCase() === currentPaidMediaPlatform.toLowerCase());
    
    if (filteredData.length === 0) {
        metricsGridContainer.innerHTML = '<div class="paid-media-empty">No data for selected platform</div>';
        return;
    }
    
    // Collect all unique metrics (standard + additional)
    const allMetrics = {
        'Total Spend': { icon: '💰', values: [], isStandard: true, isCurrency: true },
        'Total Impressions': { icon: '👁️', values: [], isStandard: true, isCurrency: false },
        'Average CPM': { icon: '📈', values: [], isStandard: true, isCurrency: true }
    };
    
    // Collect additional metrics
    filteredData.forEach(item => {
        // Standard metrics - only add if they have values
        if (item.spend && item.spend > 0) {
            allMetrics['Total Spend'].values.push(item.spend);
        }
        if (item.impressions && item.impressions > 0) {
            allMetrics['Total Impressions'].values.push(item.impressions);
        }
        if (item.cpm && item.cpm > 0) {
            allMetrics['Average CPM'].values.push(item.cpm);
        }
        
        // Additional metrics from dynamic columns
        Object.entries(item.additionalMetrics).forEach(([key, metricData]) => {
            if (!allMetrics[key]) {
                // Check if metric has currency flag from the original cell
                const hasCurrencyInCell = metricData.isCurrency === true;
                
                allMetrics[key] = {
                    icon: getPaidMediaIcon(key),
                    values: [],
                    isStandard: false,
                    isCurrency: hasCurrencyInCell || 
                               key.toLowerCase().includes('cost') || 
                               key.toLowerCase().includes('spend') ||
                               key.toLowerCase().includes('revenue') ||
                               key.toLowerCase().includes('cpc') ||
                               key.toLowerCase().includes('cpa'),
                    isPercentage: key.toLowerCase().includes('rate') || 
                                 key.toLowerCase().includes('ctr') ||
                                 key.toLowerCase().includes('%')
                };
            }
            const numValue = parseNumber(metricData.value);
            if (numValue > 0) {
                allMetrics[key].values.push(numValue);
            }
        });
    });
    
    // Generate metric cards
    const metricCards = [];
    
    Object.entries(allMetrics).forEach(([metricName, metricData]) => {
        // Skip metrics with no values at all
        if (metricData.values.length === 0) {
            // Still show the card but with N/A
            metricCards.push(`
                <div class="paid-media-metric-card">
                    <div class="paid-media-metric-header">
                        <div class="paid-media-metric-icon">${metricData.icon}</div>
                        <div class="paid-media-metric-title">${metricName}</div>
                    </div>
                    <div class="paid-media-metric-content">
                        <div style="text-align: center; padding: 1rem 0;">
                            <div style="font-size: 2.5rem; font-weight: 700; color: #86868b;">N/A</div>
                        </div>
                    </div>
                </div>
            `);
            return;
        }
        
        const total = metricData.values.reduce((sum, val) => sum + val, 0);
        let displayValue;
        
        console.log(`Rendering ${metricName}:`, { total, values: metricData.values });
        
        if (metricName === 'Average CPM') {
            // Special handling for CPM - calculate average
            const avg = total / metricData.values.length;
            displayValue = `$${avg.toFixed(2)}`;
        } else if (metricData.isCurrency) {
            // Ensure we have a valid number before formatting
            const amount = isNaN(total) ? 0 : total;
            displayValue = `$${amount.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}`;
        } else if (metricData.isPercentage) {
            const avg = total / metricData.values.length;
            displayValue = `${avg.toFixed(2)}%`;
        } else {
            displayValue = formatNumber(total);
        }
        
        console.log(`Display value for ${metricName}:`, displayValue);
        
        // Create breakdown by platform if "All Platforms" is selected
        let breakdownHtml = '';
        if (currentPaidMediaPlatform === 'all' && paidMediaData.length > 1) {
            const platforms = [...new Set(paidMediaData.map(item => item.socialMedia))];
            breakdownHtml = '<div style="margin-top: 1rem; padding-top: 1rem; border-top: 1px solid #e5e5e7;">';
            
            platforms.forEach(platform => {
                const platformData = paidMediaData.filter(item => item.socialMedia === platform);
                let platformValue = 0;
                let hasValue = false;
                
                if (metricName === 'Total Spend') {
                    platformData.forEach(item => {
                        if (item.spend && item.spend > 0) {
                            platformValue += item.spend;
                            hasValue = true;
                        }
                    });
                } else if (metricName === 'Total Impressions') {
                    platformData.forEach(item => {
                        if (item.impressions && item.impressions > 0) {
                            platformValue += item.impressions;
                            hasValue = true;
                        }
                    });
                } else if (metricName === 'Average CPM') {
                    let cpmCount = 0;
                    platformData.forEach(item => {
                        if (item.cpm && item.cpm > 0) {
                            platformValue += item.cpm;
                            cpmCount++;
                            hasValue = true;
                        }
                    });
                    if (cpmCount > 0) {
                        platformValue = platformValue / cpmCount;
                    }
                } else {
                    platformData.forEach(item => {
                        if (item.additionalMetrics[metricName]) {
                            const val = parseNumber(item.additionalMetrics[metricName].value);
                            if (val > 0) {
                                platformValue += val;
                                hasValue = true;
                            }
                        }
                    });
                }
                
                let platformDisplayValue;
                if (!hasValue) {
                    platformDisplayValue = 'N/A';
                } else if (metricName === 'Average CPM') {
                    platformDisplayValue = `$${platformValue.toFixed(2)}`;
                } else if (metricData.isCurrency) {
                    const amount = isNaN(platformValue) ? 0 : platformValue;
                    platformDisplayValue = `$${amount.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}`;
                } else if (metricData.isPercentage) {
                    platformDisplayValue = `${platformValue.toFixed(2)}%`;
                } else {
                    platformDisplayValue = formatNumber(platformValue);
                }
                
                breakdownHtml += `
                    <div class="paid-media-item">
                        <span class="paid-media-label">${platform}</span>
                        <span class="paid-media-value">${platformDisplayValue}</span>
                    </div>
                `;
            });
            
            breakdownHtml += '</div>';
        }
        
        metricCards.push(`
            <div class="paid-media-metric-card">
                <div class="paid-media-metric-header">
                    <div class="paid-media-metric-icon">${metricData.icon}</div>
                    <div class="paid-media-metric-title">${metricName}</div>
                </div>
                <div class="paid-media-metric-content">
                    <div style="text-align: center; padding: 1rem 0;">
                        <div style="font-size: 2.5rem; font-weight: 700; color: #1d1d1f;">${displayValue}</div>
                    </div>
                    ${breakdownHtml}
                </div>
            </div>
        `);
    });
    
    metricsGridContainer.innerHTML = metricCards.join('');
}

/**
 * Filter paid media by platform
 */
function filterPaidMedia(platform) {
    currentPaidMediaPlatform = platform;
    
    // Update active button
    document.querySelectorAll('.paid-media-filter-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    event.target.classList.add('active');
    
    // Re-render
    renderPaidMedia();
}